"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const validate_extract_tree_1 = require("./validate-extract-tree");
const enums_1 = require("./enums");
const error_1 = require("./error");
const UITypes_1 = __importDefault(require("../UITypes"));
const sqlUi_1 = require("../../lib/sqlUi");
// Mock dependencies
const mockColumns = [
    {
        id: 'col1',
        title: 'Column1',
        uidt: UITypes_1.default.SingleLineText,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
    },
    {
        id: 'col2',
        title: 'Column2',
        uidt: UITypes_1.default.Number,
        dt: 'int',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
    },
    {
        id: 'col3',
        title: 'Column3',
        uidt: UITypes_1.default.Date,
        dt: 'date',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
    },
    {
        id: 'col4',
        title: 'Column4',
        uidt: UITypes_1.default.Formula,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            formula: '{Column1}',
            formula_raw: '{Column1}',
            fk_column_id: 'col4',
            error: null,
            parsed_tree: {
                type: enums_1.JSEPNode.IDENTIFIER,
                name: 'col1',
                dataType: enums_1.FormulaDataTypes.STRING,
                referencedColumn: { id: 'col1', uidt: UITypes_1.default.SingleLineText },
            },
            getParsedTree: () => ({
                type: enums_1.JSEPNode.IDENTIFIER,
                name: 'col1',
                dataType: enums_1.FormulaDataTypes.STRING,
                referencedColumn: { id: 'col1', uidt: UITypes_1.default.SingleLineText },
            }),
        },
    },
    {
        id: 'col5',
        title: 'Column5',
        uidt: UITypes_1.default.Lookup,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            fk_relation_column_id: 'col6',
            fk_lookup_column_id: 'col7',
            fk_column_id: 'col5',
        },
    },
    {
        id: 'col6',
        title: 'Column6',
        uidt: UITypes_1.default.LinkToAnotherRecord,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            fk_related_model_id: 'model2',
            type: 'oo',
            fk_column_id: 'col6',
            id: 'col6',
        },
    },
    {
        id: 'col7',
        title: 'Column7',
        uidt: UITypes_1.default.SingleLineText,
        dt: 'varchar',
        pv: true,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model2',
        deleted: false,
    },
    {
        id: 'col8',
        title: 'Column8',
        uidt: UITypes_1.default.Checkbox,
        dt: 'boolean',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
    },
    {
        id: 'col9',
        title: 'Column9',
        uidt: UITypes_1.default.Time,
        dt: 'time',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
    },
    {
        id: 'col10',
        title: 'Column10',
        uidt: UITypes_1.default.Formula,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            formula: '{Column4}', // Circular reference: Column10 -> Column4 -> Column1
            formula_raw: '{Column4}',
            fk_column_id: 'col10',
            error: null,
            parsed_tree: {
                type: enums_1.JSEPNode.IDENTIFIER,
                name: 'col4',
                dataType: enums_1.FormulaDataTypes.STRING,
                referencedColumn: { id: 'col4', uidt: UITypes_1.default.Formula },
            },
            getParsedTree: () => ({
                type: enums_1.JSEPNode.IDENTIFIER,
                name: 'col4',
                dataType: enums_1.FormulaDataTypes.STRING,
                referencedColumn: { id: 'col4', uidt: UITypes_1.default.Formula },
            }),
        },
    },
    {
        id: 'cdYAQE3SFa1F4ys',
        title: 'Column11',
        uidt: UITypes_1.default.Formula,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            formula: '{c3hOMA9YnfkD4WW}',
            formula_raw: '{Column12}',
            fk_column_id: 'cdYAQE3SFa1F4ys',
            error: null,
            parsed_tree: {
                type: enums_1.JSEPNode.IDENTIFIER,
                name: 'c3hOMA9YnfkD4WW',
                dataType: enums_1.FormulaDataTypes.STRING,
                referencedColumn: { id: 'c3hOMA9YnfkD4WW', uidt: UITypes_1.default.Formula },
            },
            getParsedTree: () => ({
                type: enums_1.JSEPNode.IDENTIFIER,
                name: 'c3hOMA9YnfkD4WW',
                dataType: enums_1.FormulaDataTypes.STRING,
                referencedColumn: { id: 'c3hOMA9YnfkD4WW', uidt: UITypes_1.default.Formula },
            }),
        },
    },
    {
        id: 'c3hOMA9YnfkD4WW',
        title: 'Column12',
        uidt: UITypes_1.default.Formula,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            formula: '{cdYAQE3SFa1F4ys}',
            formula_raw: '{Column11}',
            fk_column_id: 'col12',
            error: null,
            parsed_tree: {
                type: enums_1.JSEPNode.IDENTIFIER,
                name: 'cdYAQE3SFa1F4ys',
                dataType: enums_1.FormulaDataTypes.STRING,
                referencedColumn: { id: 'cdYAQE3SFa1F4ys', uidt: UITypes_1.default.Formula },
            },
            getParsedTree: () => ({
                type: enums_1.JSEPNode.IDENTIFIER,
                name: 'cdYAQE3SFa1F4ys',
                dataType: enums_1.FormulaDataTypes.STRING,
                referencedColumn: { id: 'cdYAQE3SFa1F4ys', uidt: UITypes_1.default.Formula },
            }),
        },
    },
    // case Function to rollup
    {
        id: 'ciro4tf7xfab6pj',
        title: 'Max(Number) from Table-L1',
        uidt: UITypes_1.default.SingleLineText,
        base_id: 'base1',
    },
];
const mockGetMeta = jest.fn(async (_context, options) => {
    if (options.id === 'model2') {
        return {
            id: 'model2',
            title: 'Model2',
            base_id: 'base1',
            columns: [
                {
                    id: 'col7',
                    title: 'Column7',
                    uidt: UITypes_1.default.SingleLineText,
                    dt: 'varchar',
                    pv: true,
                    base_id: 'base1',
                    fk_workspace_id: 'ws1',
                    fk_model_id: 'model2',
                    deleted: false,
                },
            ],
        };
    }
    return {
        id: 'model1',
        title: 'Model1',
        base_id: 'base1',
        columns: mockColumns,
    };
});
const mockClientOrSqlUi = sqlUi_1.SqlUiFactory.create({ client: 'pg' });
const caseIdentifierRollup = {
    column: {
        id: 'c4o52mufchcfgi8',
        source_id: 'bv4r9c7t9enq9vs',
        base_id: 'p4zgm45rxfc6izi',
        fk_model_id: 'mroakgv13nikn55',
        title: 'Formula',
        column_name: 'Formula',
        uidt: 'Formula',
        system: false,
        order: 10,
        meta: {
            display_column_meta: {
                meta: {},
                custom: {},
            },
            defaultViewColOrder: 10,
            defaultViewColVisibility: true,
        },
        readonly: false,
        fk_workspace_id: 'w5x8zatd',
        colOptions: {
            parsed_tree: {
                type: 'Identifier',
                name: 'cpqnd2bdmzg00x1',
                raw: '{{cpqnd2bdmzg00x1}}',
                dataType: 'string',
                isDataArray: true,
                referencedColumn: {
                    id: 'czb6z4e94l7kiyr',
                    uidt: 'SingleLineText',
                    intermediaryUidt: 'LinkToAnotherRecord',
                    intermediaryId: 'cpqnd2bdmzg00x1',
                },
            },
            id: 'fhl7s1abjhwj6e0',
            fk_column_id: 'c4o52mufchcfgi8',
            formula: '{{cpqnd2bdmzg00x1}}',
            formula_raw: '{Table-L1}',
            base_id: 'p4zgm45rxfc6izi',
            fk_workspace_id: 'w5x8zatd',
        },
        extra: {
            display_type: 'SingleLineText',
        },
    },
    columns: [
        {
            id: 'cizch3j3b3inbiv',
            source_id: 'bv4r9c7t9enq9vs',
            base_id: 'p4zgm45rxfc6izi',
            fk_model_id: 'mroakgv13nikn55',
            title: 'Id',
            column_name: 'id',
            uidt: 'ID',
            order: 1,
            meta: {
                defaultViewColOrder: 2,
                defaultViewColVisibility: true,
            },
            readonly: false,
            fk_workspace_id: 'w5x8zatd',
        },
        {
            id: 'cpxst4a7s3wf3pb',
            source_id: 'bv4r9c7t9enq9vs',
            base_id: 'p4zgm45rxfc6izi',
            fk_model_id: 'mroakgv13nikn55',
            title: 'Title',
            column_name: 'title',
            uidt: 'SingleLineText',
            pv: true,
            system: false,
            order: 7,
            meta: {
                defaultViewColOrder: 1,
                defaultViewColVisibility: true,
            },
            readonly: false,
            fk_workspace_id: 'w5x8zatd',
            extra: {},
        },
        {
            id: 'cpqnd2bdmzg00x1',
            source_id: 'bv4r9c7t9enq9vs',
            base_id: 'p4zgm45rxfc6izi',
            fk_model_id: 'mroakgv13nikn55',
            title: 'Table-L1',
            uidt: 'LinkToAnotherRecord',
            virtual: true,
            order: 8,
            meta: {
                plural: 'Table-L1s',
                singular: 'Table-L1',
                custom: false,
                defaultViewColOrder: 8,
                defaultViewColVisibility: true,
            },
            readonly: false,
            fk_workspace_id: 'w5x8zatd',
            colOptions: {
                virtual: true,
                id: 'lurlzpshvoroi9a',
                type: 'hm',
                fk_column_id: 'cpqnd2bdmzg00x1',
                fk_related_model_id: 'mr6xbpmnuuusa9y',
                fk_child_column_id: 'c1tmwp5y0mx5036',
                fk_parent_column_id: 'cizch3j3b3inbiv',
                base_id: 'p4zgm45rxfc6izi',
                fk_workspace_id: 'w5x8zatd',
            },
            extra: {},
        },
        {
            id: 'ccuxnqvpoueby3r',
            source_id: 'bv4r9c7t9enq9vs',
            base_id: 'p4zgm45rxfc6izi',
            fk_model_id: 'mroakgv13nikn55',
            title: 'Max(Number) from Table-L1',
            column_name: 'Max(Number) from Table-L1',
            uidt: 'Rollup',
            system: false,
            order: 9,
            meta: {
                precision: 0,
                isLocaleString: false,
                defaultViewColOrder: 9,
                defaultViewColVisibility: true,
            },
            readonly: false,
            fk_workspace_id: 'w5x8zatd',
            colOptions: {
                id: 'rl6o2v6aesvywmry',
                fk_column_id: 'ccuxnqvpoueby3r',
                fk_relation_column_id: 'cpqnd2bdmzg00x1',
                fk_rollup_column_id: 'ciro4tf7xfab6pj',
                rollup_function: 'max',
                base_id: 'p4zgm45rxfc6izi',
                fk_workspace_id: 'w5x8zatd',
            },
            extra: {},
        },
    ],
    formula: '{Max(Number) from Table-L1}',
};
describe('validateFormulaAndExtractTreeWithType', () => {
    beforeEach(() => {
        jest.clearAllMocks();
    });
    // Test cases for function name validation
    it('should throw INVALID_FUNCTION_NAME for unknown function', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'UNKNOWN_FUNC()',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'UNKNOWN_FUNC()',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.INVALID_FUNCTION_NAME);
    });
    it('should throw INVALID_FUNCTION_NAME for unsupported function by database', async () => {
        const mocked = jest
            .spyOn(mockClientOrSqlUi, 'getUnsupportedFnList')
            .mockReturnValue(['UPPER']);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'UPPER("test")',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'UPPER("test")',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.INVALID_FUNCTION_NAME);
        mocked.mockRestore();
    });
    // Test cases for argument validation
    it('should throw INVALID_ARG for missing required arguments', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'CONCAT()', // CONCAT requires at least 1 argument
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'CONCAT()',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.MIN_ARG);
    });
    it('should throw INVALID_ARG for too many arguments', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'UPPER("a", "b")', // NOT takes exactly 1 argument
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'UPPER("a", "b")',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.INVALID_ARG);
    });
    it('should throw INVALID_ARG for incorrect argument type (identifier)', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'MIN({Column1},{Column2})', // Column2 is numeric, UPPER expects string
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'MIN({Column1},{Column2})', // Column2 is numeric, UPPER expects string
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.INVALID_ARG);
    });
    it('should throw INVALID_ARG for incorrect argument type (literal)', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'MIN("day", 1)',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'MIN("day", 1)',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.INVALID_ARG);
    });
    // Test cases for identifier resolution
    it('should resolve column by title', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column1}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col1');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.STRING);
    });
    it('should resolve column by id', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{col1}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col1');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.STRING);
    });
    it('should throw INVALID_COLUMN for unknown column', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{UNKNOWN_COL}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{UNKNOWN_COL}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.INVALID_COLUMN);
    });
    it('should handle formula column correctly', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column4}', // Column4 is a formula column that references Column1
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col4');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.STRING);
    });
    it('should handle lookup column correctly', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column5}', // Column5 is a lookup column
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col5');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.STRING);
    });
    it('should handle LTAR column correctly', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column6}', // Column6 is an LTAR column
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col6');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.STRING);
    });
    it('should detect circular reference in formula columns', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column11}',
            column: mockColumns.find((c) => c.id === 'c3hOMA9YnfkD4WW'), // col12 references col11, col11 references col12
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column11}',
            column: mockColumns.find((c) => c.id === 'c3hOMA9YnfkD4WW'),
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.CIRCULAR_REFERENCE);
    });
    // Test cases for literal handling
    it('should identify numeric literal', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '123',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.LITERAL);
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.NUMERIC);
    });
    it('should identify string literal', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '"hello"',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.LITERAL);
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.STRING);
    });
    it('should identify boolean CALL_EXP', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'ISBLANK("")',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.CALL_EXP);
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.BOOLEAN);
    });
    // Test cases for unary expression
    it('should handle negative numeric literal', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '-123',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.UNARY_EXP);
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.NUMERIC);
    });
    it('should throw NOT_SUPPORTED for unsupported unary expression', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '!TRUE', // NOT is a function, not a unary operator in this context
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '!TRUE',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.NOT_SUPPORTED);
    });
    // Test cases for binary expression
    it('should handle date subtraction (DATE - DATE)', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column3} - {Column3}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.CALL_EXP);
        expect(result.callee.name).toBe('DATETIME_DIFF');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.NUMERIC);
    });
    it('should handle time subtraction (INTERVAL - INTERVAL)', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column9} - {Column9}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.CALL_EXP);
        expect(result.callee.name).toBe('DATETIME_DIFF');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.NUMERIC);
    });
    it('should handle date and interval subtraction (DATE - INTERVAL)', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column3} - {Column9}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('-');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.DATE);
    });
    it('should handle date and interval addition (DATE + INTERVAL)', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column3} + {Column9}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('+');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.DATE);
    });
    it('should handle interval addition (INTERVAL + INTERVAL)', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column9} + {Column9}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('+');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.NUMERIC);
    });
    it('should handle comparison operators', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column2} > 10',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('>');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.COND_EXP);
    });
    it('should handle arithmetic operators', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column2} * 2',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('*');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.NUMERIC);
    });
    it('should handle string concatenation with &', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column1} & " suffix"',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('&');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.STRING);
    });
    it('should handle string concatenation with + if one operand is string', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column2} + " suffix"',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('+');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.STRING);
    });
    // Test cases for unsupported nodes
    it('should throw NOT_SUPPORTED for member expression', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'obj.property',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'obj.property',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.NOT_SUPPORTED);
    });
    it('should throw NOT_SUPPORTED for array expression', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '[1, 2, 3]',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '[1, 2, 3]',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.NOT_SUPPORTED);
    });
    it('should throw NOT_SUPPORTED for compound statement', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '1; 2',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '1; 2',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.NOT_SUPPORTED);
    });
    it('should handle nested formulas correctly', async () => {
        const formulaCol10 = mockColumns.find((c) => c.id === 'col10');
        formulaCol10.colOptions.formula =
            '{Column4}'; // col10 -> col4 -> col1
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: '{Column10}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col10');
        expect(result.dataType).toBe(enums_1.FormulaDataTypes.STRING);
    });
    it('should cast non-string arguments to string if expected type is string', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'CONCAT({Column1}, {Column2})', // Column2 is numeric, should be cast to string
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(enums_1.JSEPNode.CALL_EXP);
        expect(result.callee.name).toBe('CONCAT');
        expect(result.arguments[1].dataType).toBe(enums_1.FormulaDataTypes.NUMERIC);
    });
    it('should throw INVALID_SYNTAX for missing parentheses after function name', async () => {
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'UPPER',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(error_1.FormulaError);
        await expect((0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: 'UPPER',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', enums_1.FormulaErrorType.INVALID_SYNTAX);
    });
    it('should return correct when identifier to rollup', async () => {
        const result = await (0, validate_extract_tree_1.validateFormulaAndExtractTreeWithType)({
            formula: caseIdentifierRollup.formula,
            columns: caseIdentifierRollup.columns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.referencedColumn.uidt).toBe('SingleLineText');
    });
});
//# sourceMappingURL=data:application/json;base64,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