"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const formulaHelpers_1 = require("./formulaHelpers");
const UITypes_1 = __importDefault(require("./UITypes"));
const base_id = 'pIJkwfxdDwd';
const getMeta = async (_context, _param) => ({
    base_id,
    columns: [],
    id: 'mWhcgaGWdKE',
    title: 'Table1',
});
describe('Formula parsing and type validation', () => {
    it('Simple formula', async () => {
        const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
            formula: '1 + 2',
            columns: [],
            clientOrSqlUi: 'mysql2',
            getMeta,
        });
        expect(result.dataType).toEqual(formulaHelpers_1.FormulaDataTypes.NUMERIC);
    });
    it('Formula with IF condition', async () => {
        const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
            formula: 'IF({column}, "Found", BLANK())',
            columns: [
                {
                    base_id,
                    id: 'cid',
                    title: 'column',
                    uidt: UITypes_1.default.Number,
                },
            ],
            clientOrSqlUi: 'mysql2',
            getMeta,
        });
        expect(result.dataType).toEqual(formulaHelpers_1.FormulaDataTypes.STRING);
    });
    it('Complex formula', async () => {
        const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
            formula: 'SWITCH({column2},"value1",IF({column1}, "Found", BLANK()),"value2", 2)',
            columns: [
                {
                    base_id,
                    id: 'id1',
                    title: 'column1',
                    uidt: UITypes_1.default.Number,
                },
                {
                    base_id,
                    id: 'id2',
                    title: 'column2',
                    uidt: UITypes_1.default.SingleLineText,
                },
            ],
            clientOrSqlUi: 'mysql2',
            getMeta,
        });
        expect(result.dataType).toEqual(formulaHelpers_1.FormulaDataTypes.STRING);
        const result1 = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
            formula: 'SWITCH({column2},"value1",IF({column1}, 1, 2),"value2", 2)',
            columns: [
                {
                    base_id,
                    id: 'id1',
                    title: 'column1',
                    uidt: UITypes_1.default.Number,
                },
                {
                    base_id,
                    id: 'id2',
                    title: 'column2',
                    uidt: UITypes_1.default.SingleLineText,
                },
            ],
            clientOrSqlUi: 'mysql2',
            getMeta,
        });
        expect(result1.dataType).toEqual(formulaHelpers_1.FormulaDataTypes.NUMERIC);
    });
    describe('Date and time interaction', () => {
        it('Time - time equals numeric', async () => {
            const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '{Time1} - {Time2}',
                columns: [
                    {
                        base_id,
                        id: 'TUrXeTf4JUHdnRvn',
                        title: 'Time1',
                        uidt: UITypes_1.default.Time,
                    },
                    {
                        base_id,
                        id: 'J3aD/yLDT2GF6NEB',
                        title: 'Time2',
                        uidt: UITypes_1.default.Time,
                    },
                ],
                clientOrSqlUi: 'pg',
                getMeta,
            });
            expect(result.dataType).toEqual(formulaHelpers_1.FormulaDataTypes.NUMERIC);
        });
        it('Time - time equals numeric', async () => {
            const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '{Time1} - {Time2}',
                columns: [
                    {
                        base_id,
                        id: 'TUrXeTf4JUHdnRvn',
                        title: 'Time1',
                        uidt: UITypes_1.default.Time,
                    },
                    {
                        base_id,
                        id: 'J3aD/yLDT2GF6NEB',
                        title: 'Time2',
                        uidt: UITypes_1.default.Time,
                    },
                ],
                clientOrSqlUi: 'pg',
                getMeta,
            });
            expect(result.dataType).toEqual(formulaHelpers_1.FormulaDataTypes.NUMERIC);
        });
        it('Date + time equals date', async () => {
            const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '{Date1} + {Time2}',
                columns: [
                    {
                        base_id,
                        id: 'TUrXeTf4JUHdnRvn',
                        title: 'Date1',
                        uidt: UITypes_1.default.Date,
                    },
                    {
                        base_id,
                        id: 'J3aD/yLDT2GF6NEB',
                        title: 'Time2',
                        uidt: UITypes_1.default.Time,
                    },
                ],
                clientOrSqlUi: 'pg',
                getMeta,
            });
            expect(result.dataType).toEqual(formulaHelpers_1.FormulaDataTypes.DATE);
        });
    });
    describe('binary expression', () => {
        it(`& operator will return string`, async () => {
            const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '"Hello" & "World"',
                columns: [],
                clientOrSqlUi: 'pg',
                getMeta,
            });
            expect(result.dataType).toBe(formulaHelpers_1.FormulaDataTypes.STRING);
        });
    });
    describe('errors', () => {
        it(`will provide position for syntax error`, async () => {
            try {
                await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                    formula: '1 +',
                    columns: [],
                    clientOrSqlUi: 'mysql2',
                    getMeta,
                    trackPosition: true,
                });
            }
            catch (ex) {
                expect(ex.extra.position).toEqual({
                    column: 3,
                    row: 0,
                    length: 1,
                });
            }
            try {
                await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                    formula: '(1 + 1',
                    columns: [],
                    clientOrSqlUi: 'mysql2',
                    getMeta,
                    trackPosition: true,
                });
            }
            catch (ex) {
                expect(ex.extra.position).toEqual({
                    column: 6,
                    row: 0,
                    length: 1,
                });
            }
            try {
                await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                    formula: 'CONCAT)',
                    columns: [],
                    clientOrSqlUi: 'mysql2',
                    getMeta,
                    trackPosition: true,
                });
            }
            catch (ex) {
                expect(ex.extra.position).toEqual({
                    column: 6,
                    row: 0,
                    length: 1,
                });
            }
        });
        it(`will provide position for column not found`, async () => {
            try {
                await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                    formula: '1 + __a_',
                    columns: [],
                    clientOrSqlUi: 'mysql2',
                    getMeta,
                    trackPosition: true,
                });
            }
            catch (ex) {
                expect(ex.extra.position).toEqual({
                    column: 4,
                    row: 0,
                    length: 4,
                });
            }
            try {
                await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                    formula: '__a_',
                    columns: [],
                    clientOrSqlUi: 'mysql2',
                    getMeta,
                    trackPosition: true,
                });
            }
            catch (ex) {
                expect(ex.extra.position).toEqual({
                    column: 0,
                    row: 0,
                    length: 4,
                });
            }
            try {
                await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                    formula: 'CONCAT(__a_  , "A")',
                    columns: [],
                    clientOrSqlUi: 'mysql2',
                    getMeta,
                    trackPosition: true,
                });
            }
            catch (ex) {
                expect(ex.extra.position).toEqual({
                    column: 7,
                    row: 0,
                    length: 6,
                });
            }
        });
        it(`will handle formula missing parentheses`, async () => {
            try {
                await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                    formula: 'CONCAT',
                    columns: [],
                    clientOrSqlUi: 'mysql2',
                    getMeta,
                    trackPosition: true,
                });
            }
            catch (ex) {
                expect(ex.message).toContain('Missing parentheses after function name');
                expect(ex.extra.position).toEqual({
                    column: 6,
                    row: 0,
                    length: 1,
                });
            }
            try {
                await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                    formula: 'CONCAT(CONCAT)',
                    columns: [],
                    clientOrSqlUi: 'mysql2',
                    getMeta,
                    trackPosition: true,
                });
            }
            catch (ex) {
                expect(ex.message).toContain('Missing parentheses after function name');
                expect(ex.extra.position).toEqual({
                    column: 13,
                    row: 0,
                    length: 1,
                });
            }
        });
        it(`will handle formula minimum argument`, async () => {
            try {
                await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                    formula: 'CONCAT(CONCAT())',
                    columns: [],
                    clientOrSqlUi: 'mysql2',
                    getMeta,
                    trackPosition: true,
                });
            }
            catch (ex) {
                expect(ex.extra.position).toEqual({
                    column: 7,
                    row: 0,
                    length: 8,
                });
            }
        });
    });
    describe('referenced info', () => {
        it(`will return referenced column when directly referenced`, async () => {
            const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '{column1}',
                columns: [
                    {
                        base_id,
                        id: 'id1',
                        title: 'column1',
                        uidt: UITypes_1.default.Number,
                    },
                ],
                clientOrSqlUi: 'mysql2',
                getMeta,
            });
            expect(result.referencedColumn.id).toEqual('id1');
            expect(result.referencedColumn.uidt).toEqual(UITypes_1.default.Number);
        });
        it(`will return referenced column with binary operation`, async () => {
            const supportedTypes = [
                UITypes_1.default.Decimal,
                UITypes_1.default.Currency,
                UITypes_1.default.Percent,
                UITypes_1.default.SingleLineText,
                UITypes_1.default.LongText,
            ];
            for (const supportedType of supportedTypes) {
                const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                    formula: '{column1} + 3',
                    columns: [
                        {
                            base_id,
                            id: 'id1',
                            title: 'column1',
                            uidt: supportedType,
                        },
                    ],
                    clientOrSqlUi: 'mysql2',
                    getMeta,
                });
                expect(result.referencedColumn.id).toEqual('id1');
                expect(result.referencedColumn.uidt).toEqual(supportedType);
            }
        });
        it(`will not return referenced column with impure binary operation`, async () => {
            const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '{column1} + 3',
                columns: [
                    {
                        base_id,
                        id: 'id1',
                        title: 'column1',
                        uidt: UITypes_1.default.Number,
                    },
                ],
                clientOrSqlUi: 'mysql2',
                getMeta,
            });
            expect(result.referencedColumn).toBeUndefined();
            expect(result.uidtCandidates).toEqual([UITypes_1.default.Decimal]);
        });
        it(`will not return referenced column when multiple columns is used`, async () => {
            const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '{column1} + {column2}',
                columns: [
                    {
                        base_id,
                        id: 'id1',
                        title: 'column1',
                        uidt: UITypes_1.default.Decimal,
                    },
                    {
                        base_id,
                        id: 'id2',
                        title: 'column2',
                        uidt: UITypes_1.default.Decimal,
                    },
                ],
                clientOrSqlUi: 'mysql2',
                getMeta,
            });
            expect(result.referencedColumn).toBeUndefined();
            const result2 = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '{column1} + ({column2} + {column3})',
                columns: [
                    {
                        base_id,
                        id: 'id1',
                        title: 'column1',
                        uidt: UITypes_1.default.Decimal,
                    },
                    {
                        base_id,
                        id: 'id2',
                        title: 'column2',
                        uidt: UITypes_1.default.Decimal,
                    },
                    {
                        base_id,
                        id: 'id3',
                        title: 'column3',
                        uidt: UITypes_1.default.Decimal,
                    },
                ],
                clientOrSqlUi: 'mysql2',
                getMeta,
            });
            expect(result2.referencedColumn).toBeUndefined();
            const result3 = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '{column1} + ({column2} + 1)',
                columns: [
                    {
                        base_id,
                        id: 'id1',
                        title: 'column1',
                        uidt: UITypes_1.default.Decimal,
                    },
                    {
                        base_id,
                        id: 'id2',
                        title: 'column2',
                        uidt: UITypes_1.default.Number,
                    },
                ],
                clientOrSqlUi: 'mysql2',
                getMeta,
            });
            expect(result3.referencedColumn).toBeUndefined();
        });
        it(`will return referenced column with pure call expression`, async () => {
            const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: 'MAX({column1}, 3)',
                columns: [
                    {
                        base_id,
                        id: 'id1',
                        title: 'column1',
                        uidt: UITypes_1.default.Number,
                    },
                ],
                clientOrSqlUi: 'mysql2',
                getMeta,
            });
            expect(result.referencedColumn.id).toEqual('id1');
            expect(result.referencedColumn.uidt).toEqual(UITypes_1.default.Number);
        });
        it(`will return referenced column with pure call expression for arrays`, async () => {
            const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '{column1}',
                columns: [
                    {
                        base_id,
                        id: 'id1',
                        title: 'column1',
                        uidt: UITypes_1.default.LinkToAnotherRecord,
                        colOptions: {
                            fk_column_id: '',
                            id: '',
                            type: 'hm',
                            getRelatedTable: async () => {
                                return {
                                    base_id,
                                    columns: [
                                        {
                                            base_id,
                                            id: 'col_lok1',
                                            uidt: UITypes_1.default.SingleLineText,
                                            title: 'LOK1',
                                            pv: true,
                                        },
                                    ],
                                    id: 'tbl1',
                                    title: 'tbl1',
                                };
                            },
                        },
                    },
                ],
                clientOrSqlUi: 'mysql2',
                getMeta,
            });
            expect(result.referencedColumn.id).toEqual('col_lok1');
            expect(result.referencedColumn.uidt).toEqual(UITypes_1.default.SingleLineText);
            const result1 = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: '{column1}',
                columns: [
                    {
                        base_id,
                        id: 'id1',
                        title: 'column1',
                        uidt: UITypes_1.default.Lookup,
                        colOptions: {
                            fk_column_id: '',
                            fk_relation_column_id: 'col_rel1',
                            fk_lookup_column_id: 'col_lok1',
                            getRelationColumn: async () => {
                                return {
                                    base_id,
                                    fk_model_id: 'tbl1',
                                    id: 'col_rel1',
                                    uidt: UITypes_1.default.SingleLineText,
                                    title: 'LOK1',
                                    colOptions: {
                                        fk_column_id: '',
                                        id: '',
                                        type: 'hm',
                                        getRelatedTable: async () => {
                                            return {
                                                base_id,
                                                columns: [
                                                    {
                                                        base_id,
                                                        id: 'col_lok1',
                                                        uidt: UITypes_1.default.SingleLineText,
                                                        title: 'LOK1',
                                                        pv: true,
                                                    },
                                                ],
                                                id: 'tbl1',
                                                title: 'tbl1',
                                            };
                                        },
                                    },
                                };
                            },
                            getLookupColumn: async () => {
                                return {
                                    base_id,
                                    id: 'col_lok1',
                                    uidt: UITypes_1.default.SingleLineText,
                                    title: 'LOK1',
                                };
                            },
                        },
                    },
                ],
                clientOrSqlUi: 'mysql2',
                getMeta,
            });
            expect(result1.referencedColumn.id).toEqual('col_lok1');
            expect(result1.referencedColumn.uidt).toEqual(UITypes_1.default.SingleLineText);
        });
        it(`will not return referenced column with impure call expression`, async () => {
            const result = await (0, formulaHelpers_1.validateFormulaAndExtractTreeWithType)({
                formula: 'CEILING({column1})',
                columns: [
                    {
                        base_id,
                        id: 'id1',
                        title: 'column1',
                        uidt: UITypes_1.default.Number,
                    },
                ],
                clientOrSqlUi: 'mysql2',
                getMeta,
            });
            expect(result.referencedColumn).toBeUndefined();
            expect(result.uidtCandidates).toEqual([UITypes_1.default.Decimal]);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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