"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const error_message_parser_1 = require("./error-message-parser");
const query_filter_parser_1 = require("./query-filter-parser");
describe('query-filter-parser', () => {
    it('will parse eq expression with double quote', async () => {
        const text = `(field1, eq, "hello, 'world")`;
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field1',
                    comparison_op: 'eq',
                    value: "hello, 'world",
                },
            ],
        };
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse eq expression with single quote', async () => {
        const text = `(field1, eq, 'hello, "world')`;
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field1',
                    comparison_op: 'eq',
                    value: 'hello, "world',
                },
            ],
        };
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse eq expression with mandarin, japanese and cryillic', async () => {
        const text = `(Тест, eq, "新年快乐おはよう")`;
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'Тест',
                    comparison_op: 'eq',
                    value: '新年快乐おはよう',
                },
            ],
        };
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse wrong expression', async () => {
        const text = '(field1)';
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        expect(result.parseErrors.length).toBeGreaterThan(0);
    });
    it('will parse blank expression', async () => {
        const text = '(field1, blank)';
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field1',
                    comparison_op: 'blank',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse blank value', async () => {
        const text = '(Category,gb_eq,)';
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'Category',
                    comparison_op: 'gb_eq',
                    value: '',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse value and field with parentheses', async () => {
        const text = '("field(1)",eq,"(hello)")';
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field(1)',
                    comparison_op: 'eq',
                    value: '(hello)',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse value separated by comma', async () => {
        const text = '("field(1)",eq, hello, world,  baby!)';
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field(1)',
                    comparison_op: 'eq',
                    value: 'hello,world,baby!',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse complex nested logic', async () => {
        const text = '~not(field1, isWithin, nextNumberOfDays, 10)~and((field2, eq, 2)~or(field2, eq, 3))~or(field3, not, 4)';
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field1',
                    comparison_op: 'isWithin',
                    logical_op: 'not',
                    comparison_sub_op: undefined,
                    value: ['nextNumberOfDays', '10'],
                },
                {
                    is_group: true,
                    logical_op: 'and',
                    children: [
                        {
                            is_group: false,
                            field: 'field2',
                            comparison_op: 'eq',
                            value: '2',
                        },
                        {
                            is_group: false,
                            field: 'field2',
                            comparison_op: 'eq',
                            logical_op: 'or',
                            value: '3',
                        },
                    ],
                },
                {
                    is_group: false,
                    field: 'field3',
                    comparison_op: 'not',
                    logical_op: 'or',
                    value: '4',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse keyword as value', async () => {
        const text = '(Category,is,blank)';
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'Category',
                    comparison_op: 'blank',
                    value: undefined,
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse empty quote as value', async () => {
        const text = "(Category,eq,'')";
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'Category',
                    comparison_op: 'eq',
                    value: '',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse null as null', async () => {
        const text = '("field(1)",eq, null)';
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field(1)',
                    comparison_op: 'eq',
                    value: null,
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    it('will parse empty string as empty', async () => {
        const text = `("field(1)",eq,'')`;
        const result = query_filter_parser_1.QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field(1)',
                    comparison_op: 'eq',
                    value: '',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    });
    describe('error-handling', () => {
        it(`will error when and/or operation is wrong`, async () => {
            expect.hasAssertions();
            const text = `(fSingleLineText,eq,"sample,text")or(fSingleLineText,eq,"sample text")`;
            const result = query_filter_parser_1.QueryFilterParser.parse(text);
            const message = (0, error_message_parser_1.parseParsingError)(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter expression. Expected a valid logical operator like '~or' or '~and', but found 'or'`);
        });
        it(`will handle parsing error when operation is wrong`, async () => {
            const text = `(fSingleLineText,noneInOperation,"sample,text")`;
            const result = query_filter_parser_1.QueryFilterParser.parse(text);
            const message = (0, error_message_parser_1.parseParsingError)(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter expression: 'noneInOperation' is not a recognized operator. Please use a valid comparison or logical operator`);
        });
        it(`will handle parsing error when operation is missing`, async () => {
            const text = `(fSingleLineText,)`;
            const result = query_filter_parser_1.QueryFilterParser.parse(text);
            const message = (0, error_message_parser_1.parseParsingError)(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter expression: ')' is not a recognized operator. Please use a valid comparison or logical operator`);
        });
        it(`will handle parsing error when operation is wrapped in quotes`, async () => {
            const text = `(fSingleLineText,"eq")`;
            const result = query_filter_parser_1.QueryFilterParser.parse(text);
            const message = (0, error_message_parser_1.parseParsingError)(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter expression: '"eq"' is not a recognized operator. Please use a valid comparison or logical operator`);
        });
        it(`will handle parsing error when no opening parentheses`, async () => {
            const text = `fSingleLineText,eq)`;
            const result = query_filter_parser_1.QueryFilterParser.parse(text);
            const message = (0, error_message_parser_1.parseParsingError)(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter syntax: expected a logical operator like '~not' or opening parenthesis, but found 'fSingleLineText'`);
        });
        it(`will handle parsing error when no closing parentheses`, async () => {
            const text = `(fSingleLineText,eq`;
            const result = query_filter_parser_1.QueryFilterParser.parse(text);
            const message = (0, error_message_parser_1.parseParsingError)(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter syntax: expected a closing parentheses ')', but found ''`);
        });
        it(`will handle parsing error when not operator is wrong`, async () => {
            const text = `not(fSingleLineText,eq,1)`;
            const result = query_filter_parser_1.QueryFilterParser.parse(text);
            const message = (0, error_message_parser_1.parseParsingError)(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter syntax: expected a logical operator like '~not' or opening parenthesis, but found 'not'`);
        });
        it(`will handle parsing error when missing comma`, async () => {
            const text = `(fSingleLineText`;
            const result = query_filter_parser_1.QueryFilterParser.parse(text);
            const message = (0, error_message_parser_1.parseParsingError)(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter syntax: expected comma ',' followed with operator (and value) after field`);
        });
        it(`will handle parsing error when missing arguments`, async () => {
            const text = `(fSingleLineText)`;
            const result = query_filter_parser_1.QueryFilterParser.parse(text);
            const message = (0, error_message_parser_1.parseParsingError)(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter syntax: expected comma ',' followed with operator (and value) after field`);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicXVlcnktZmlsdGVyLXBhcnNlci5zcGVjLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vc3JjL2xpYi9wYXJzZXIvcXVlcnlGaWx0ZXIvcXVlcnktZmlsdGVyLXBhcnNlci5zcGVjLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsaUVBQTJEO0FBQzNELCtEQUEwRDtBQUUxRCxRQUFRLENBQUMscUJBQXFCLEVBQUUsR0FBRyxFQUFFO0lBQ25DLEVBQUUsQ0FBQyw0Q0FBNEMsRUFBRSxLQUFLLElBQUksRUFBRTtRQUMxRCxNQUFNLElBQUksR0FBRywrQkFBK0IsQ0FBQztRQUM3QyxNQUFNLGlCQUFpQixHQUFHO1lBQ3hCLFFBQVEsRUFBRSxJQUFJO1lBQ2QsVUFBVSxFQUFFLEtBQUs7WUFDakIsUUFBUSxFQUFFO2dCQUNSO29CQUNFLFFBQVEsRUFBRSxLQUFLO29CQUNmLEtBQUssRUFBRSxRQUFRO29CQUNmLGFBQWEsRUFBRSxJQUFJO29CQUNuQixLQUFLLEVBQUUsZUFBZTtpQkFDdkI7YUFDRjtTQUNGLENBQUM7UUFDRixNQUFNLE1BQU0sR0FBRyx1Q0FBaUIsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0MsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUN0RCxDQUFDLENBQUMsQ0FBQztJQUNILEVBQUUsQ0FBQyw0Q0FBNEMsRUFBRSxLQUFLLElBQUksRUFBRTtRQUMxRCxNQUFNLElBQUksR0FBRywrQkFBK0IsQ0FBQztRQUM3QyxNQUFNLGlCQUFpQixHQUFHO1lBQ3hCLFFBQVEsRUFBRSxJQUFJO1lBQ2QsVUFBVSxFQUFFLEtBQUs7WUFDakIsUUFBUSxFQUFFO2dCQUNSO29CQUNFLFFBQVEsRUFBRSxLQUFLO29CQUNmLEtBQUssRUFBRSxRQUFRO29CQUNmLGFBQWEsRUFBRSxJQUFJO29CQUNuQixLQUFLLEVBQUUsZUFBZTtpQkFDdkI7YUFDRjtTQUNGLENBQUM7UUFDRixNQUFNLE1BQU0sR0FBRyx1Q0FBaUIsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0MsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUN0RCxDQUFDLENBQUMsQ0FBQztJQUNILEVBQUUsQ0FBQywrREFBK0QsRUFBRSxLQUFLLElBQUksRUFBRTtRQUM3RSxNQUFNLElBQUksR0FBRyx3QkFBd0IsQ0FBQztRQUN0QyxNQUFNLGlCQUFpQixHQUFHO1lBQ3hCLFFBQVEsRUFBRSxJQUFJO1lBQ2QsVUFBVSxFQUFFLEtBQUs7WUFDakIsUUFBUSxFQUFFO2dCQUNSO29CQUNFLFFBQVEsRUFBRSxLQUFLO29CQUNmLEtBQUssRUFBRSxNQUFNO29CQUNiLGFBQWEsRUFBRSxJQUFJO29CQUNuQixLQUFLLEVBQUUsVUFBVTtpQkFDbEI7YUFDRjtTQUNGLENBQUM7UUFDRixNQUFNLE1BQU0sR0FBRyx1Q0FBaUIsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0MsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUN0RCxDQUFDLENBQUMsQ0FBQztJQUNILEVBQUUsQ0FBQyw2QkFBNkIsRUFBRSxLQUFLLElBQUksRUFBRTtRQUMzQyxNQUFNLElBQUksR0FBRyxVQUFVLENBQUM7UUFDeEIsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdDLE1BQU0sQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN2RCxDQUFDLENBQUMsQ0FBQztJQUNILEVBQUUsQ0FBQyw2QkFBNkIsRUFBRSxLQUFLLElBQUksRUFBRTtRQUMzQyxNQUFNLElBQUksR0FBRyxpQkFBaUIsQ0FBQztRQUMvQixNQUFNLE1BQU0sR0FBRyx1Q0FBaUIsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0MsTUFBTSxpQkFBaUIsR0FBRztZQUN4QixRQUFRLEVBQUUsSUFBSTtZQUNkLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRTtnQkFDUjtvQkFDRSxRQUFRLEVBQUUsS0FBSztvQkFDZixLQUFLLEVBQUUsUUFBUTtvQkFDZixhQUFhLEVBQUUsT0FBTztpQkFDdkI7YUFDRjtTQUNGLENBQUM7UUFDRixNQUFNLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3RELENBQUMsQ0FBQyxDQUFDO0lBQ0gsRUFBRSxDQUFDLHdCQUF3QixFQUFFLEtBQUssSUFBSSxFQUFFO1FBQ3RDLE1BQU0sSUFBSSxHQUFHLG1CQUFtQixDQUFDO1FBQ2pDLE1BQU0sTUFBTSxHQUFHLHVDQUFpQixDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QyxNQUFNLGlCQUFpQixHQUFHO1lBQ3hCLFFBQVEsRUFBRSxJQUFJO1lBQ2QsVUFBVSxFQUFFLEtBQUs7WUFDakIsUUFBUSxFQUFFO2dCQUNSO29CQUNFLFFBQVEsRUFBRSxLQUFLO29CQUNmLEtBQUssRUFBRSxVQUFVO29CQUNqQixhQUFhLEVBQUUsT0FBTztvQkFDdEIsS0FBSyxFQUFFLEVBQUU7aUJBQ1Y7YUFDRjtTQUNGLENBQUM7UUFDRixNQUFNLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3RELENBQUMsQ0FBQyxDQUFDO0lBQ0gsRUFBRSxDQUFDLDZDQUE2QyxFQUFFLEtBQUssSUFBSSxFQUFFO1FBQzNELE1BQU0sSUFBSSxHQUFHLDJCQUEyQixDQUFDO1FBQ3pDLE1BQU0sTUFBTSxHQUFHLHVDQUFpQixDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QyxNQUFNLGlCQUFpQixHQUFHO1lBQ3hCLFFBQVEsRUFBRSxJQUFJO1lBQ2QsVUFBVSxFQUFFLEtBQUs7WUFDakIsUUFBUSxFQUFFO2dCQUNSO29CQUNFLFFBQVEsRUFBRSxLQUFLO29CQUNmLEtBQUssRUFBRSxVQUFVO29CQUNqQixhQUFhLEVBQUUsSUFBSTtvQkFDbkIsS0FBSyxFQUFFLFNBQVM7aUJBQ2pCO2FBQ0Y7U0FDRixDQUFDO1FBQ0YsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUN0RCxDQUFDLENBQUMsQ0FBQztJQUNILEVBQUUsQ0FBQyxxQ0FBcUMsRUFBRSxLQUFLLElBQUksRUFBRTtRQUNuRCxNQUFNLElBQUksR0FBRyx1Q0FBdUMsQ0FBQztRQUNyRCxNQUFNLE1BQU0sR0FBRyx1Q0FBaUIsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0MsTUFBTSxpQkFBaUIsR0FBRztZQUN4QixRQUFRLEVBQUUsSUFBSTtZQUNkLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRTtnQkFDUjtvQkFDRSxRQUFRLEVBQUUsS0FBSztvQkFDZixLQUFLLEVBQUUsVUFBVTtvQkFDakIsYUFBYSxFQUFFLElBQUk7b0JBQ25CLEtBQUssRUFBRSxtQkFBbUI7aUJBQzNCO2FBQ0Y7U0FDRixDQUFDO1FBQ0YsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUN0RCxDQUFDLENBQUMsQ0FBQztJQUNILEVBQUUsQ0FBQyxpQ0FBaUMsRUFBRSxLQUFLLElBQUksRUFBRTtRQUMvQyxNQUFNLElBQUksR0FDUix3R0FBd0csQ0FBQztRQUMzRyxNQUFNLE1BQU0sR0FBRyx1Q0FBaUIsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0MsTUFBTSxpQkFBaUIsR0FBRztZQUN4QixRQUFRLEVBQUUsSUFBSTtZQUNkLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRTtnQkFDUjtvQkFDRSxRQUFRLEVBQUUsS0FBSztvQkFDZixLQUFLLEVBQUUsUUFBUTtvQkFDZixhQUFhLEVBQUUsVUFBVTtvQkFDekIsVUFBVSxFQUFFLEtBQUs7b0JBQ2pCLGlCQUFpQixFQUFFLFNBQVM7b0JBQzVCLEtBQUssRUFBRSxDQUFDLGtCQUFrQixFQUFFLElBQUksQ0FBQztpQkFDbEM7Z0JBQ0Q7b0JBQ0UsUUFBUSxFQUFFLElBQUk7b0JBQ2QsVUFBVSxFQUFFLEtBQUs7b0JBQ2pCLFFBQVEsRUFBRTt3QkFDUjs0QkFDRSxRQUFRLEVBQUUsS0FBSzs0QkFDZixLQUFLLEVBQUUsUUFBUTs0QkFDZixhQUFhLEVBQUUsSUFBSTs0QkFDbkIsS0FBSyxFQUFFLEdBQUc7eUJBQ1g7d0JBQ0Q7NEJBQ0UsUUFBUSxFQUFFLEtBQUs7NEJBQ2YsS0FBSyxFQUFFLFFBQVE7NEJBQ2YsYUFBYSxFQUFFLElBQUk7NEJBQ25CLFVBQVUsRUFBRSxJQUFJOzRCQUNoQixLQUFLLEVBQUUsR0FBRzt5QkFDWDtxQkFDRjtpQkFDRjtnQkFDRDtvQkFDRSxRQUFRLEVBQUUsS0FBSztvQkFDZixLQUFLLEVBQUUsUUFBUTtvQkFDZixhQUFhLEVBQUUsS0FBSztvQkFDcEIsVUFBVSxFQUFFLElBQUk7b0JBQ2hCLEtBQUssRUFBRSxHQUFHO2lCQUNYO2FBQ0Y7U0FDRixDQUFDO1FBQ0YsTUFBTSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUN0RCxDQUFDLENBQUMsQ0FBQztJQUNILEVBQUUsQ0FBQyw2QkFBNkIsRUFBRSxLQUFLLElBQUksRUFBRTtRQUMzQyxNQUFNLElBQUksR0FBRyxxQkFBcUIsQ0FBQztRQUNuQyxNQUFNLE1BQU0sR0FBRyx1Q0FBaUIsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0MsTUFBTSxpQkFBaUIsR0FBRztZQUN4QixRQUFRLEVBQUUsSUFBSTtZQUNkLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRTtnQkFDUjtvQkFDRSxRQUFRLEVBQUUsS0FBSztvQkFDZixLQUFLLEVBQUUsVUFBVTtvQkFDakIsYUFBYSxFQUFFLE9BQU87b0JBQ3RCLEtBQUssRUFBRSxTQUFTO2lCQUNqQjthQUNGO1NBQ0YsQ0FBQztRQUNGLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsT0FBTyxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDdEQsQ0FBQyxDQUFDLENBQUM7SUFDSCxFQUFFLENBQUMsaUNBQWlDLEVBQUUsS0FBSyxJQUFJLEVBQUU7UUFDL0MsTUFBTSxJQUFJLEdBQUcsa0JBQWtCLENBQUM7UUFDaEMsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdDLE1BQU0saUJBQWlCLEdBQUc7WUFDeEIsUUFBUSxFQUFFLElBQUk7WUFDZCxVQUFVLEVBQUUsS0FBSztZQUNqQixRQUFRLEVBQUU7Z0JBQ1I7b0JBQ0UsUUFBUSxFQUFFLEtBQUs7b0JBQ2YsS0FBSyxFQUFFLFVBQVU7b0JBQ2pCLGFBQWEsRUFBRSxJQUFJO29CQUNuQixLQUFLLEVBQUUsRUFBRTtpQkFDVjthQUNGO1NBQ0YsQ0FBQztRQUNGLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsT0FBTyxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDdEQsQ0FBQyxDQUFDLENBQUM7SUFDSCxFQUFFLENBQUMseUJBQXlCLEVBQUUsS0FBSyxJQUFJLEVBQUU7UUFDdkMsTUFBTSxJQUFJLEdBQUcsdUJBQXVCLENBQUM7UUFDckMsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdDLE1BQU0saUJBQWlCLEdBQUc7WUFDeEIsUUFBUSxFQUFFLElBQUk7WUFDZCxVQUFVLEVBQUUsS0FBSztZQUNqQixRQUFRLEVBQUU7Z0JBQ1I7b0JBQ0UsUUFBUSxFQUFFLEtBQUs7b0JBQ2YsS0FBSyxFQUFFLFVBQVU7b0JBQ2pCLGFBQWEsRUFBRSxJQUFJO29CQUNuQixLQUFLLEVBQUUsSUFBSTtpQkFDWjthQUNGO1NBQ0YsQ0FBQztRQUNGLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsT0FBTyxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDdEQsQ0FBQyxDQUFDLENBQUM7SUFDSCxFQUFFLENBQUMsa0NBQWtDLEVBQUUsS0FBSyxJQUFJLEVBQUU7UUFDaEQsTUFBTSxJQUFJLEdBQUcsb0JBQW9CLENBQUM7UUFDbEMsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdDLE1BQU0saUJBQWlCLEdBQUc7WUFDeEIsUUFBUSxFQUFFLElBQUk7WUFDZCxVQUFVLEVBQUUsS0FBSztZQUNqQixRQUFRLEVBQUU7Z0JBQ1I7b0JBQ0UsUUFBUSxFQUFFLEtBQUs7b0JBQ2YsS0FBSyxFQUFFLFVBQVU7b0JBQ2pCLGFBQWEsRUFBRSxJQUFJO29CQUNuQixLQUFLLEVBQUUsRUFBRTtpQkFDVjthQUNGO1NBQ0YsQ0FBQztRQUNGLE1BQU0sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsT0FBTyxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDdEQsQ0FBQyxDQUFDLENBQUM7SUFDSCxRQUFRLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxFQUFFO1FBQzlCLEVBQUUsQ0FBQywyQ0FBMkMsRUFBRSxLQUFLLElBQUksRUFBRTtZQUN6RCxNQUFNLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDdkIsTUFBTSxJQUFJLEdBQUcsd0VBQXdFLENBQUM7WUFDdEYsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdDLE1BQU0sT0FBTyxHQUFHLElBQUEsd0NBQWlCLEVBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pELE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQ2xCLG1HQUFtRyxDQUNwRyxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7UUFDSCxFQUFFLENBQUMsbURBQW1ELEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDakUsTUFBTSxJQUFJLEdBQUcsaURBQWlELENBQUM7WUFDL0QsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdDLE1BQU0sT0FBTyxHQUFHLElBQUEsd0NBQWlCLEVBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pELE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQ2xCLDhIQUE4SCxDQUMvSCxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7UUFDSCxFQUFFLENBQUMscURBQXFELEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDbkUsTUFBTSxJQUFJLEdBQUcsb0JBQW9CLENBQUM7WUFDbEMsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdDLE1BQU0sT0FBTyxHQUFHLElBQUEsd0NBQWlCLEVBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pELE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQ2xCLGdIQUFnSCxDQUNqSCxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7UUFDSCxFQUFFLENBQUMsK0RBQStELEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDN0UsTUFBTSxJQUFJLEdBQUcsd0JBQXdCLENBQUM7WUFDdEMsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdDLE1BQU0sT0FBTyxHQUFHLElBQUEsd0NBQWlCLEVBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pELE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQ2xCLG1IQUFtSCxDQUNwSCxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7UUFDSCxFQUFFLENBQUMsdURBQXVELEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDckUsTUFBTSxJQUFJLEdBQUcscUJBQXFCLENBQUM7WUFDbkMsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdDLE1BQU0sT0FBTyxHQUFHLElBQUEsd0NBQWlCLEVBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pELE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQ2xCLG9IQUFvSCxDQUNySCxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7UUFDSCxFQUFFLENBQUMsdURBQXVELEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDckUsTUFBTSxJQUFJLEdBQUcscUJBQXFCLENBQUM7WUFDbkMsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdDLE1BQU0sT0FBTyxHQUFHLElBQUEsd0NBQWlCLEVBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pELE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQ2xCLHlFQUF5RSxDQUMxRSxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7UUFDSCxFQUFFLENBQUMsc0RBQXNELEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDcEUsTUFBTSxJQUFJLEdBQUcsMkJBQTJCLENBQUM7WUFDekMsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdDLE1BQU0sT0FBTyxHQUFHLElBQUEsd0NBQWlCLEVBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pELE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQ2xCLHdHQUF3RyxDQUN6RyxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7UUFDSCxFQUFFLENBQUMsOENBQThDLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDNUQsTUFBTSxJQUFJLEdBQUcsa0JBQWtCLENBQUM7WUFDaEMsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdDLE1BQU0sT0FBTyxHQUFHLElBQUEsd0NBQWlCLEVBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pELE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQ2xCLDBGQUEwRixDQUMzRixDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7UUFDSCxFQUFFLENBQUMsa0RBQWtELEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDaEUsTUFBTSxJQUFJLEdBQUcsbUJBQW1CLENBQUM7WUFDakMsTUFBTSxNQUFNLEdBQUcsdUNBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdDLE1BQU0sT0FBTyxHQUFHLElBQUEsd0NBQWlCLEVBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pELE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQ2xCLDBGQUEwRixDQUMzRixDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIn0=