"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractProjectRolePower = extractProjectRolePower;
exports.extractWorkspaceRolePower = extractWorkspaceRolePower;
exports.mapWorkspaceRolesObjToProjectRolesObj = mapWorkspaceRolesObjToProjectRolesObj;
exports.getProjectRole = getProjectRole;
exports.hasMinimumRoleAccess = hasMinimumRoleAccess;
exports.extractBaseRoleFromWorkspaceRole = extractBaseRoleFromWorkspaceRole;
exports.getEffectiveBaseRole = getEffectiveBaseRole;
const enums_1 = require("./enums");
const helperFunctions_1 = require("./helperFunctions");
function extractProjectRolePower(user, 
/**
 * forbiddenCallback is used to keep old function behaviour
 */
forbiddenCallback) {
    const reverseOrderedProjectRoles = [...enums_1.OrderedProjectRoles].reverse();
    if (!user.base_roles) {
        return -1;
    }
    // get most powerful role of user (TODO moving forward we will confirm that user has only one role)
    let role = null;
    let power = -1;
    for (const r of Object.keys(user.base_roles)) {
        const ind = reverseOrderedProjectRoles.indexOf(r);
        if (ind > power) {
            role = r;
            power = ind;
        }
    }
    const ind = reverseOrderedProjectRoles.indexOf(role);
    if (ind === -1) {
        forbiddenCallback === null || forbiddenCallback === void 0 ? void 0 : forbiddenCallback();
    }
    return ind;
}
/**
 * Get the power of the workspace role of the user.
 * @param user - The user object.
 * @returns The power of the workspace role of the user.
 */
function extractWorkspaceRolePower(user, 
/**
 * forbiddenCallback is used to keep old function behaviour
 */ forbiddenCallback) {
    const reverseOrderedWorkspaceRoles = [...enums_1.OrderedWorkspaceRoles].reverse();
    if (!user.workspace_roles) {
        return -1;
    }
    // get most powerful role of user (TODO moving forward we will confirm that user has only one role)
    let role = null;
    let power = -1;
    for (const r of Object.keys(user.workspace_roles)) {
        const ind = reverseOrderedWorkspaceRoles.indexOf(r);
        if (ind > power) {
            role = r;
            power = ind;
        }
    }
    const ind = reverseOrderedWorkspaceRoles.indexOf(role);
    if (ind === -1) {
        forbiddenCallback === null || forbiddenCallback === void 0 ? void 0 : forbiddenCallback();
    }
    return ind;
}
/**
 * Map the workspace roles object to the project roles object.
 * @param wsRoles - The workspace roles object.
 * @returns The project roles object.
 */
function mapWorkspaceRolesObjToProjectRolesObj(wsRoles) {
    wsRoles = (0, helperFunctions_1.extractRolesObj)(wsRoles);
    let baseRoles = null;
    if (wsRoles) {
        for (const r of Object.keys(wsRoles)) {
            if (!baseRoles)
                baseRoles = {};
            baseRoles[enums_1.WorkspaceRolesToProjectRoles[r]] = wsRoles[r];
        }
    }
    return baseRoles;
}
function getProjectRole(user, inheritFromWorkspace = false) {
    // get most powerful role of user (TODO moving forward we will confirm that user has only one role)
    let role = null;
    let power = -1;
    if (user.base_roles) {
        for (const r of Object.keys(user.base_roles)) {
            const ind = enums_1.OrderedProjectRoles.indexOf(r);
            if (ind > power) {
                role = r;
                power = ind;
            }
        }
        return role;
    }
    else if (inheritFromWorkspace && user.workspace_roles) {
        for (const r of Object.keys(user.workspace_roles)) {
            const ind = enums_1.OrderedWorkspaceRoles.indexOf(r);
            if (ind > power) {
                role = r;
                power = ind;
            }
        }
        return role;
    }
    else {
        return null;
    }
}
function hasMinimumRoleAccess(user, minimumRole, 
/**
 * forbiddenCallback is used to keep old function behaviour
 */
forbiddenCallback) {
    const power = extractProjectRolePower(user, forbiddenCallback);
    const reverseOrderedProjectRoles = [...enums_1.OrderedProjectRoles].reverse();
    const minimumRoleIndex = reverseOrderedProjectRoles.indexOf(minimumRole);
    return power >= minimumRoleIndex;
}
// extract corresponding base role from workspace role
function extractBaseRoleFromWorkspaceRole(workspaceRole) {
    if (!workspaceRole)
        return null;
    let workspaceRoleStr;
    if (typeof workspaceRole === 'object') {
        // If workspaceRole is an object, extract the first key
        workspaceRoleStr = Object.keys(workspaceRole)[0];
    }
    else if (typeof workspaceRole === 'string') {
        // If workspaceRole is a string, use it directly
        workspaceRoleStr = workspaceRole;
    }
    // Extract base role from workspace role
    const baseRole = enums_1.WorkspaceRolesToProjectRoles[workspaceRoleStr];
    return baseRole || null;
}
function getEffectiveBaseRole({ baseRole, baseTeamRole, workspaceRole, workspaceTeamRole, defaultBaseRole, }) {
    // Apply role priority hierarchy for base roles:
    // 1. Direct base role (highest priority)
    // 2. Role inherited from base-team
    // 3. Default base role
    // 4. Role inherited from workspace role
    // 5. Role inherited from workspace team role (lowest priority)
    let finalBaseRole = baseRole;
    if (!finalBaseRole || finalBaseRole === enums_1.ProjectRoles.INHERIT) {
        if (baseTeamRole) {
            finalBaseRole = baseTeamRole;
        }
        else if (defaultBaseRole) {
            finalBaseRole = defaultBaseRole;
        }
        else if (workspaceRole && workspaceRole !== enums_1.WorkspaceUserRoles.INHERIT) {
            finalBaseRole = extractBaseRoleFromWorkspaceRole(workspaceRole);
        }
        else if (workspaceTeamRole) {
            finalBaseRole = extractBaseRoleFromWorkspaceRole(workspaceTeamRole);
        }
    }
    return finalBaseRole === enums_1.ProjectRoles.INHERIT || !finalBaseRole
        ? enums_1.ProjectRoles.NO_ACCESS
        : finalBaseRole;
}
//# sourceMappingURL=data:application/json;base64,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