import { ViewSettingOverrideOptions } from './enums';
import { ViewTypes } from './globals';
import { ViewType } from '../lib/Api';
/**
 * Interface representing a copy view config option with its metadata.
 * Used to define the properties and constraints for each type of view configuration that can be copied.
 */
export interface CopyViewConfigOption {
    /** The display order of this option in the UI */
    order: number;
    /** The human-readable label for this option */
    label: string;
    /** The copy view config type value */
    value: ViewSettingOverrideOptions;
    /** The i18n translation key for the label */
    i18nLabel: string;
    /** Array of view types that support this configuration option */
    supportedViewTypes: ViewTypes[];
    /** IconMapKey */
    icon: string;
    /** Whether this option is disabled */
    disabled?: boolean;
}
/**
 * A comprehensive mapping of all available copy view configuration options.
 *
 * This map defines the metadata and constraints for each type of view configuration that can be copied,
 * including display order, labels, i18n keys, and which view types support each configuration option.
 *
 * @remarks
 * Use this map to:
 * - Get the configuration options available for a specific view type
 * - Determine which view types support a particular configuration option
 * - Access i18n labels and display ordering for UI rendering
 *
 * @example
 * ```ts
 * // Get the field visibility option metadata
 * const fieldVisibilityOption = copyViewConfigOptionMap[ViewSettingOverrideOptions.FIELD_VISIBILITY]
 * // Check if it's supported for grid views
 * const isSupported = fieldVisibilityOption.supportedViewTypes.includes(ViewTypes.GRID)
 * ```
 */
export declare const copyViewConfigOptionMap: Record<ViewSettingOverrideOptions, Omit<CopyViewConfigOption, 'disabled'>>;
/**
 * Retrieves all copy view configuration options with their availability status for a specific view type.
 *
 * This function returns all available configuration options, marking each as enabled or disabled
 * based on whether the source view type supports that particular configuration.
 * The returned options are sorted by their display order.
 *
 * @param sourceViewType - The view type from which configurations will be copied
 * @param destinationViewType - The view type to which configurations will be copied
 * @returns An array of supported destination view type configuration options with a 'disabled' flag indicating support status, sorted by order
 */
export declare const getCopyViewConfigOptions: (sourceViewType?: ViewTypes, destinationViewType?: ViewTypes) => Omit<CopyViewConfigOption, "supportedViewTypes">[];
/**
 * Filters a list of copy view configuration types to only include those supported by the source view type.
 *
 * This function validates each configuration type against the source view type's capabilities,
 * removing any unsupported or invalid configuration types from the list.
 *
 * @param viewSettingOverrideOptions - Array of configuration types to be validated
 * @param sourceViewType - The view type from which configurations will be copied
 * @param destinationViewType - The view type to which configurations will be copied
 * @returns A filtered array containing only the configuration types supported by the source view type and destination view type
 *
 * @example
 * ```ts
 * // Validate selected config types for a Form view
 * const selectedTypes = [ViewSettingOverrideOptions.FIELD_VISIBILITY, ViewSettingOverrideOptions.FILTER_CONDITION, ViewSettingOverrideOptions.GROUP]
 * const supportedTypes = extractSupportedViewSettingOverrideOptions(selectedTypes, ViewTypes.FORM, ViewTypes.GRID)
 * // Result: Only FieldVisibility will be included since destination Grid view support filter and group but source Forms don't support Filters or Groups
 * ```
 */
export declare const extractSupportedViewSettingOverrideOptions: (viewSettingOverrideOptions: ViewSettingOverrideOptions[], sourceViewType: ViewTypes, destinationViewType?: ViewTypes) => ViewSettingOverrideOptions[];
/**
 * Retrieves all view setting override options supported by a specific view type.
 */
export declare const getViewSettingOverrideOptionsByViewType: (viewType?: ViewTypes) => ViewSettingOverrideOptions[];
/**
 * Finds the first non-personal view from an array of views based on optional filters.
 *
 * @param views - Array of views to search through
 * @param options - Filter options
 * @param options.excludeViewType - View type(s) to exclude from the search (single type or array)
 * @param options.includeViewType - View type(s) to include in the search (single type or array)
 * @returns The first non-personal view matching the criteria, or undefined if none found
 *
 * @example
 * // Find first non-personal view
 * const view = getFirstNonPersonalView(views, {});
 *
 * @example
 * // Find first non-personal grid view
 * const gridView = getFirstNonPersonalView(views, {
 *   includeViewType: ViewTypes.GRID
 * });
 *
 * @example
 * // Find first non-personal view excluding gallery and kanban
 * const view = getFirstNonPersonalView(views, {
 *   excludeViewType: [ViewTypes.GALLERY, ViewTypes.KANBAN]
 * });
 */
export declare const getFirstNonPersonalView: (views: Array<ViewType>, { excludeViewType, includeViewType, }?: {
    excludeViewType?: ViewTypes | Array<ViewTypes>;
    includeViewType?: ViewTypes | Array<ViewTypes>;
}) => ViewType | undefined;
