"use strict";
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFirstNonPersonalView = exports.getViewSettingOverrideOptionsByViewType = exports.extractSupportedViewSettingOverrideOptions = exports.getCopyViewConfigOptions = exports.copyViewConfigOptionMap = void 0;
const enums_1 = require("./enums");
const globals_1 = require("./globals");
const is_1 = require("./is");
/**
 * A comprehensive mapping of all available copy view configuration options.
 *
 * This map defines the metadata and constraints for each type of view configuration that can be copied,
 * including display order, labels, i18n keys, and which view types support each configuration option.
 *
 * @remarks
 * Use this map to:
 * - Get the configuration options available for a specific view type
 * - Determine which view types support a particular configuration option
 * - Access i18n labels and display ordering for UI rendering
 *
 * @example
 * ```ts
 * // Get the field visibility option metadata
 * const fieldVisibilityOption = copyViewConfigOptionMap[ViewSettingOverrideOptions.FIELD_VISIBILITY]
 * // Check if it's supported for grid views
 * const isSupported = fieldVisibilityOption.supportedViewTypes.includes(ViewTypes.GRID)
 * ```
 */
exports.copyViewConfigOptionMap = {
    [enums_1.ViewSettingOverrideOptions.FIELD_VISIBILITY]: {
        order: 1,
        label: 'Field visibility',
        value: enums_1.ViewSettingOverrideOptions.FIELD_VISIBILITY,
        i18nLabel: `objects.copyViewConfig.${enums_1.ViewSettingOverrideOptions.FIELD_VISIBILITY}`,
        supportedViewTypes: [
            globals_1.ViewTypes.GRID,
            globals_1.ViewTypes.GALLERY,
            globals_1.ViewTypes.KANBAN,
            globals_1.ViewTypes.FORM,
            globals_1.ViewTypes.CALENDAR,
            globals_1.ViewTypes.MAP,
        ],
        icon: 'eyeSlash',
    },
    [enums_1.ViewSettingOverrideOptions.FIELD_ORDER]: {
        order: 2,
        label: 'Field order',
        value: enums_1.ViewSettingOverrideOptions.FIELD_ORDER,
        i18nLabel: `objects.copyViewConfig.${enums_1.ViewSettingOverrideOptions.FIELD_ORDER}`,
        supportedViewTypes: [
            globals_1.ViewTypes.GRID,
            globals_1.ViewTypes.GALLERY,
            globals_1.ViewTypes.KANBAN,
            globals_1.ViewTypes.FORM,
            globals_1.ViewTypes.CALENDAR,
            globals_1.ViewTypes.MAP,
        ],
        icon: 'ncNumberList',
    },
    [enums_1.ViewSettingOverrideOptions.COLUMN_WIDTH]: {
        order: 3,
        label: 'Column width',
        value: enums_1.ViewSettingOverrideOptions.COLUMN_WIDTH,
        i18nLabel: `objects.copyViewConfig.${enums_1.ViewSettingOverrideOptions.COLUMN_WIDTH}`,
        supportedViewTypes: [globals_1.ViewTypes.GRID],
        icon: 'columnWidth',
    },
    [enums_1.ViewSettingOverrideOptions.ROW_HEIGHT]: {
        order: 4,
        label: 'Row height',
        value: enums_1.ViewSettingOverrideOptions.ROW_HEIGHT,
        i18nLabel: `objects.copyViewConfig.${enums_1.ViewSettingOverrideOptions.ROW_HEIGHT}`,
        supportedViewTypes: [globals_1.ViewTypes.GRID],
        icon: 'rowHeight',
    },
    [enums_1.ViewSettingOverrideOptions.FILTER_CONDITION]: {
        order: 5,
        label: 'Filter condition',
        value: enums_1.ViewSettingOverrideOptions.FILTER_CONDITION,
        i18nLabel: `objects.copyViewConfig.${enums_1.ViewSettingOverrideOptions.FILTER_CONDITION}`,
        supportedViewTypes: [
            globals_1.ViewTypes.GRID,
            globals_1.ViewTypes.GALLERY,
            globals_1.ViewTypes.KANBAN,
            globals_1.ViewTypes.CALENDAR,
            globals_1.ViewTypes.MAP,
        ],
        icon: 'filter',
    },
    [enums_1.ViewSettingOverrideOptions.GROUP]: {
        order: 6,
        label: 'Group',
        value: enums_1.ViewSettingOverrideOptions.GROUP,
        i18nLabel: `objects.copyViewConfig.${enums_1.ViewSettingOverrideOptions.GROUP}`,
        supportedViewTypes: [globals_1.ViewTypes.GRID],
        icon: 'group',
    },
    [enums_1.ViewSettingOverrideOptions.SORT]: {
        order: 7,
        label: 'Sort',
        value: enums_1.ViewSettingOverrideOptions.SORT,
        i18nLabel: `objects.copyViewConfig.${enums_1.ViewSettingOverrideOptions.SORT}`,
        supportedViewTypes: [
            globals_1.ViewTypes.GRID,
            globals_1.ViewTypes.GALLERY,
            globals_1.ViewTypes.KANBAN,
            globals_1.ViewTypes.CALENDAR,
        ],
        icon: 'sort',
    },
    [enums_1.ViewSettingOverrideOptions.ROW_COLORING]: {
        order: 8,
        label: 'Row coloring',
        value: enums_1.ViewSettingOverrideOptions.ROW_COLORING,
        i18nLabel: `objects.copyViewConfig.${enums_1.ViewSettingOverrideOptions.ROW_COLORING}`,
        supportedViewTypes: [
            globals_1.ViewTypes.GRID,
            globals_1.ViewTypes.GALLERY,
            globals_1.ViewTypes.KANBAN,
            globals_1.ViewTypes.CALENDAR,
        ],
        icon: 'ncPaintRoller',
    },
};
/**
 * Retrieves all copy view configuration options with their availability status for a specific view type.
 *
 * This function returns all available configuration options, marking each as enabled or disabled
 * based on whether the source view type supports that particular configuration.
 * The returned options are sorted by their display order.
 *
 * @param sourceViewType - The view type from which configurations will be copied
 * @param destinationViewType - The view type to which configurations will be copied
 * @returns An array of supported destination view type configuration options with a 'disabled' flag indicating support status, sorted by order
 */
const getCopyViewConfigOptions = (sourceViewType, destinationViewType) => {
    return Object.values(exports.copyViewConfigOptionMap)
        .filter((option) => {
        if (!destinationViewType)
            return true;
        return option.supportedViewTypes.includes(destinationViewType);
    })
        .map((option) => {
        const { supportedViewTypes } = option, rest = __rest(option, ["supportedViewTypes"]);
        return Object.assign(Object.assign({}, rest), { disabled: !(0, is_1.ncIsUndefined)(sourceViewType) &&
                !supportedViewTypes.includes(sourceViewType) });
    })
        .sort((a, b) => a.order - b.order);
};
exports.getCopyViewConfigOptions = getCopyViewConfigOptions;
/**
 * Filters a list of copy view configuration types to only include those supported by the source view type.
 *
 * This function validates each configuration type against the source view type's capabilities,
 * removing any unsupported or invalid configuration types from the list.
 *
 * @param viewSettingOverrideOptions - Array of configuration types to be validated
 * @param sourceViewType - The view type from which configurations will be copied
 * @param destinationViewType - The view type to which configurations will be copied
 * @returns A filtered array containing only the configuration types supported by the source view type and destination view type
 *
 * @example
 * ```ts
 * // Validate selected config types for a Form view
 * const selectedTypes = [ViewSettingOverrideOptions.FIELD_VISIBILITY, ViewSettingOverrideOptions.FILTER_CONDITION, ViewSettingOverrideOptions.GROUP]
 * const supportedTypes = extractSupportedViewSettingOverrideOptions(selectedTypes, ViewTypes.FORM, ViewTypes.GRID)
 * // Result: Only FieldVisibility will be included since destination Grid view support filter and group but source Forms don't support Filters or Groups
 * ```
 */
const extractSupportedViewSettingOverrideOptions = (viewSettingOverrideOptions, sourceViewType, destinationViewType) => {
    // extract destination view type options
    const destinationViewTypeOptions = Object.values(enums_1.ViewSettingOverrideOptions).filter((option) => {
        if (!destinationViewType)
            return true;
        return exports.copyViewConfigOptionMap[option].supportedViewTypes.includes(destinationViewType);
    });
    // return only options which supported in destination as well as source view type
    return (viewSettingOverrideOptions || []).filter((type) => {
        if (!exports.copyViewConfigOptionMap[type] ||
            !destinationViewTypeOptions.includes(type)) {
            return false;
        }
        return exports.copyViewConfigOptionMap[type].supportedViewTypes.includes(sourceViewType);
    });
};
exports.extractSupportedViewSettingOverrideOptions = extractSupportedViewSettingOverrideOptions;
/**
 * Retrieves all view setting override options supported by a specific view type.
 */
const getViewSettingOverrideOptionsByViewType = (viewType) => {
    return (0, exports.extractSupportedViewSettingOverrideOptions)(Object.values(enums_1.ViewSettingOverrideOptions), viewType);
};
exports.getViewSettingOverrideOptionsByViewType = getViewSettingOverrideOptionsByViewType;
/**
 * Finds the first non-personal view from an array of views based on optional filters.
 *
 * @param views - Array of views to search through
 * @param options - Filter options
 * @param options.excludeViewType - View type(s) to exclude from the search (single type or array)
 * @param options.includeViewType - View type(s) to include in the search (single type or array)
 * @returns The first non-personal view matching the criteria, or undefined if none found
 *
 * @example
 * // Find first non-personal view
 * const view = getFirstNonPersonalView(views, {});
 *
 * @example
 * // Find first non-personal grid view
 * const gridView = getFirstNonPersonalView(views, {
 *   includeViewType: ViewTypes.GRID
 * });
 *
 * @example
 * // Find first non-personal view excluding gallery and kanban
 * const view = getFirstNonPersonalView(views, {
 *   excludeViewType: [ViewTypes.GALLERY, ViewTypes.KANBAN]
 * });
 */
const getFirstNonPersonalView = (views, { excludeViewType, includeViewType, } = {}) => {
    if (!(views === null || views === void 0 ? void 0 : views.length))
        return undefined;
    return views
        .sort((a, b) => a.order - b.order)
        .find((view) => {
        // Skip personal views
        if (view.lock_type === enums_1.ViewLockType.Personal) {
            return false;
        }
        // Exclude specified view types
        if (excludeViewType) {
            const excludeTypes = Array.isArray(excludeViewType)
                ? excludeViewType
                : [excludeViewType];
            if (excludeTypes.includes(view.type)) {
                return false;
            }
        }
        // Include only specified view types
        if (includeViewType) {
            const includeTypes = Array.isArray(includeViewType)
                ? includeViewType
                : [includeViewType];
            if (!includeTypes.includes(view.type)) {
                return false;
            }
        }
        return true;
    });
};
exports.getFirstNonPersonalView = getFirstNonPersonalView;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmlld1V0aWxzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL2xpYi92aWV3VXRpbHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7Ozs7Ozs7QUFBQSxtQ0FBbUU7QUFDbkUsdUNBQXNDO0FBQ3RDLDZCQUFxQztBQXdCckM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FtQkc7QUFDVSxRQUFBLHVCQUF1QixHQUdoQztJQUNGLENBQUMsa0NBQTBCLENBQUMsZ0JBQWdCLENBQUMsRUFBRTtRQUM3QyxLQUFLLEVBQUUsQ0FBQztRQUNSLEtBQUssRUFBRSxrQkFBa0I7UUFDekIsS0FBSyxFQUFFLGtDQUEwQixDQUFDLGdCQUFnQjtRQUNsRCxTQUFTLEVBQUUsMEJBQTBCLGtDQUEwQixDQUFDLGdCQUFnQixFQUFFO1FBQ2xGLGtCQUFrQixFQUFFO1lBQ2xCLG1CQUFTLENBQUMsSUFBSTtZQUNkLG1CQUFTLENBQUMsT0FBTztZQUNqQixtQkFBUyxDQUFDLE1BQU07WUFDaEIsbUJBQVMsQ0FBQyxJQUFJO1lBQ2QsbUJBQVMsQ0FBQyxRQUFRO1lBQ2xCLG1CQUFTLENBQUMsR0FBRztTQUNkO1FBQ0QsSUFBSSxFQUFFLFVBQVU7S0FDakI7SUFDRCxDQUFDLGtDQUEwQixDQUFDLFdBQVcsQ0FBQyxFQUFFO1FBQ3hDLEtBQUssRUFBRSxDQUFDO1FBQ1IsS0FBSyxFQUFFLGFBQWE7UUFDcEIsS0FBSyxFQUFFLGtDQUEwQixDQUFDLFdBQVc7UUFDN0MsU0FBUyxFQUFFLDBCQUEwQixrQ0FBMEIsQ0FBQyxXQUFXLEVBQUU7UUFDN0Usa0JBQWtCLEVBQUU7WUFDbEIsbUJBQVMsQ0FBQyxJQUFJO1lBQ2QsbUJBQVMsQ0FBQyxPQUFPO1lBQ2pCLG1CQUFTLENBQUMsTUFBTTtZQUNoQixtQkFBUyxDQUFDLElBQUk7WUFDZCxtQkFBUyxDQUFDLFFBQVE7WUFDbEIsbUJBQVMsQ0FBQyxHQUFHO1NBQ2Q7UUFDRCxJQUFJLEVBQUUsY0FBYztLQUNyQjtJQUNELENBQUMsa0NBQTBCLENBQUMsWUFBWSxDQUFDLEVBQUU7UUFDekMsS0FBSyxFQUFFLENBQUM7UUFDUixLQUFLLEVBQUUsY0FBYztRQUNyQixLQUFLLEVBQUUsa0NBQTBCLENBQUMsWUFBWTtRQUM5QyxTQUFTLEVBQUUsMEJBQTBCLGtDQUEwQixDQUFDLFlBQVksRUFBRTtRQUM5RSxrQkFBa0IsRUFBRSxDQUFDLG1CQUFTLENBQUMsSUFBSSxDQUFDO1FBQ3BDLElBQUksRUFBRSxhQUFhO0tBQ3BCO0lBQ0QsQ0FBQyxrQ0FBMEIsQ0FBQyxVQUFVLENBQUMsRUFBRTtRQUN2QyxLQUFLLEVBQUUsQ0FBQztRQUNSLEtBQUssRUFBRSxZQUFZO1FBQ25CLEtBQUssRUFBRSxrQ0FBMEIsQ0FBQyxVQUFVO1FBQzVDLFNBQVMsRUFBRSwwQkFBMEIsa0NBQTBCLENBQUMsVUFBVSxFQUFFO1FBQzVFLGtCQUFrQixFQUFFLENBQUMsbUJBQVMsQ0FBQyxJQUFJLENBQUM7UUFDcEMsSUFBSSxFQUFFLFdBQVc7S0FDbEI7SUFDRCxDQUFDLGtDQUEwQixDQUFDLGdCQUFnQixDQUFDLEVBQUU7UUFDN0MsS0FBSyxFQUFFLENBQUM7UUFDUixLQUFLLEVBQUUsa0JBQWtCO1FBQ3pCLEtBQUssRUFBRSxrQ0FBMEIsQ0FBQyxnQkFBZ0I7UUFDbEQsU0FBUyxFQUFFLDBCQUEwQixrQ0FBMEIsQ0FBQyxnQkFBZ0IsRUFBRTtRQUNsRixrQkFBa0IsRUFBRTtZQUNsQixtQkFBUyxDQUFDLElBQUk7WUFDZCxtQkFBUyxDQUFDLE9BQU87WUFDakIsbUJBQVMsQ0FBQyxNQUFNO1lBQ2hCLG1CQUFTLENBQUMsUUFBUTtZQUNsQixtQkFBUyxDQUFDLEdBQUc7U0FDZDtRQUNELElBQUksRUFBRSxRQUFRO0tBQ2Y7SUFDRCxDQUFDLGtDQUEwQixDQUFDLEtBQUssQ0FBQyxFQUFFO1FBQ2xDLEtBQUssRUFBRSxDQUFDO1FBQ1IsS0FBSyxFQUFFLE9BQU87UUFDZCxLQUFLLEVBQUUsa0NBQTBCLENBQUMsS0FBSztRQUN2QyxTQUFTLEVBQUUsMEJBQTBCLGtDQUEwQixDQUFDLEtBQUssRUFBRTtRQUN2RSxrQkFBa0IsRUFBRSxDQUFDLG1CQUFTLENBQUMsSUFBSSxDQUFDO1FBQ3BDLElBQUksRUFBRSxPQUFPO0tBQ2Q7SUFDRCxDQUFDLGtDQUEwQixDQUFDLElBQUksQ0FBQyxFQUFFO1FBQ2pDLEtBQUssRUFBRSxDQUFDO1FBQ1IsS0FBSyxFQUFFLE1BQU07UUFDYixLQUFLLEVBQUUsa0NBQTBCLENBQUMsSUFBSTtRQUN0QyxTQUFTLEVBQUUsMEJBQTBCLGtDQUEwQixDQUFDLElBQUksRUFBRTtRQUN0RSxrQkFBa0IsRUFBRTtZQUNsQixtQkFBUyxDQUFDLElBQUk7WUFDZCxtQkFBUyxDQUFDLE9BQU87WUFDakIsbUJBQVMsQ0FBQyxNQUFNO1lBQ2hCLG1CQUFTLENBQUMsUUFBUTtTQUNuQjtRQUNELElBQUksRUFBRSxNQUFNO0tBQ2I7SUFDRCxDQUFDLGtDQUEwQixDQUFDLFlBQVksQ0FBQyxFQUFFO1FBQ3pDLEtBQUssRUFBRSxDQUFDO1FBQ1IsS0FBSyxFQUFFLGNBQWM7UUFDckIsS0FBSyxFQUFFLGtDQUEwQixDQUFDLFlBQVk7UUFDOUMsU0FBUyxFQUFFLDBCQUEwQixrQ0FBMEIsQ0FBQyxZQUFZLEVBQUU7UUFDOUUsa0JBQWtCLEVBQUU7WUFDbEIsbUJBQVMsQ0FBQyxJQUFJO1lBQ2QsbUJBQVMsQ0FBQyxPQUFPO1lBQ2pCLG1CQUFTLENBQUMsTUFBTTtZQUNoQixtQkFBUyxDQUFDLFFBQVE7U0FDbkI7UUFDRCxJQUFJLEVBQUUsZUFBZTtLQUN0QjtDQUNGLENBQUM7QUFFRjs7Ozs7Ozs7OztHQVVHO0FBQ0ksTUFBTSx3QkFBd0IsR0FBRyxDQUN0QyxjQUEwQixFQUMxQixtQkFBK0IsRUFDcUIsRUFBRTtJQUN0RCxPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUMsK0JBQXVCLENBQUM7U0FDMUMsTUFBTSxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUU7UUFDakIsSUFBSSxDQUFDLG1CQUFtQjtZQUFFLE9BQU8sSUFBSSxDQUFDO1FBRXRDLE9BQU8sTUFBTSxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ2pFLENBQUMsQ0FBQztTQUNELEdBQUcsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFO1FBQ2QsTUFBTSxFQUFFLGtCQUFrQixLQUFjLE1BQU0sRUFBZixJQUFJLFVBQUssTUFBTSxFQUF4QyxzQkFBK0IsQ0FBUyxDQUFDO1FBQy9DLHVDQUNLLElBQUksS0FDUCxRQUFRLEVBQ04sQ0FBQyxJQUFBLGtCQUFhLEVBQUMsY0FBYyxDQUFDO2dCQUM5QixDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUMsSUFDOUM7SUFDSixDQUFDLENBQUM7U0FDRCxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxHQUFHLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUN2QyxDQUFDLENBQUM7QUFwQlcsUUFBQSx3QkFBd0IsNEJBb0JuQztBQUVGOzs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FrQkc7QUFDSSxNQUFNLDBDQUEwQyxHQUFHLENBQ3hELDBCQUF3RCxFQUN4RCxjQUF5QixFQUN6QixtQkFBK0IsRUFDL0IsRUFBRTtJQUNGLHdDQUF3QztJQUN4QyxNQUFNLDBCQUEwQixHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQzlDLGtDQUEwQixDQUMzQixDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFO1FBQ2xCLElBQUksQ0FBQyxtQkFBbUI7WUFBRSxPQUFPLElBQUksQ0FBQztRQUV0QyxPQUFPLCtCQUF1QixDQUFDLE1BQU0sQ0FBQyxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FDaEUsbUJBQW1CLENBQ3BCLENBQUM7SUFDSixDQUFDLENBQUMsQ0FBQztJQUVILGlGQUFpRjtJQUNqRixPQUFPLENBQUMsMEJBQTBCLElBQUksRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUU7UUFDeEQsSUFDRSxDQUFDLCtCQUF1QixDQUFDLElBQUksQ0FBQztZQUM5QixDQUFDLDBCQUEwQixDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFDMUMsQ0FBQztZQUNELE9BQU8sS0FBSyxDQUFDO1FBQ2YsQ0FBQztRQUVELE9BQU8sK0JBQXVCLENBQUMsSUFBSSxDQUFDLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUM5RCxjQUFjLENBQ2YsQ0FBQztJQUNKLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDO0FBN0JXLFFBQUEsMENBQTBDLDhDQTZCckQ7QUFFRjs7R0FFRztBQUNJLE1BQU0sdUNBQXVDLEdBQUcsQ0FDckQsUUFBb0IsRUFDcEIsRUFBRTtJQUNGLE9BQU8sSUFBQSxrREFBMEMsRUFDL0MsTUFBTSxDQUFDLE1BQU0sQ0FBQyxrQ0FBMEIsQ0FBQyxFQUN6QyxRQUFRLENBQ1QsQ0FBQztBQUNKLENBQUMsQ0FBQztBQVBXLFFBQUEsdUNBQXVDLDJDQU9sRDtBQUVGOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0F3Qkc7QUFDSSxNQUFNLHVCQUF1QixHQUFHLENBQ3JDLEtBQXNCLEVBQ3RCLEVBQ0UsZUFBZSxFQUNmLGVBQWUsTUFJYixFQUFFLEVBQ2dCLEVBQUU7SUFDeEIsSUFBSSxDQUFDLENBQUEsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLE1BQU0sQ0FBQTtRQUFFLE9BQU8sU0FBUyxDQUFDO0lBRXJDLE9BQU8sS0FBSztTQUNULElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLEdBQUcsQ0FBQyxDQUFDLEtBQUssQ0FBQztTQUNqQyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRTtRQUNiLHNCQUFzQjtRQUN0QixJQUFJLElBQUksQ0FBQyxTQUFTLEtBQUssb0JBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUM3QyxPQUFPLEtBQUssQ0FBQztRQUNmLENBQUM7UUFFRCwrQkFBK0I7UUFDL0IsSUFBSSxlQUFlLEVBQUUsQ0FBQztZQUNwQixNQUFNLFlBQVksR0FBcUIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUM7Z0JBQ25FLENBQUMsQ0FBQyxlQUFlO2dCQUNqQixDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUMsQ0FBQztZQUN0QixJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7Z0JBQ3JDLE9BQU8sS0FBSyxDQUFDO1lBQ2YsQ0FBQztRQUNILENBQUM7UUFFRCxvQ0FBb0M7UUFDcEMsSUFBSSxlQUFlLEVBQUUsQ0FBQztZQUNwQixNQUFNLFlBQVksR0FBcUIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUM7Z0JBQ25FLENBQUMsQ0FBQyxlQUFlO2dCQUNqQixDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUMsQ0FBQztZQUN0QixJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztnQkFDdEMsT0FBTyxLQUFLLENBQUM7WUFDZixDQUFDO1FBQ0gsQ0FBQztRQUVELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQyxDQUFDLENBQUM7QUFDUCxDQUFDLENBQUM7QUExQ1csUUFBQSx1QkFBdUIsMkJBMENsQyJ9