import UITypes from './UITypes';
import { FormulaDataTypes } from '../lib/formula/enums';
import { getDateTimeValue, getDateValue } from '../lib/dateTimeHelper';
import { formatBytes, parseProp } from '../lib/helperFunctions';
import { convertMS2Duration } from '../lib/durationUtils';
import { getCurrencyValue, roundTo } from '../lib/numberUtils';
import { ColumnHelper } from './columnHelper';
var NumericalAggregations;
(function (NumericalAggregations) {
    NumericalAggregations["Sum"] = "sum";
    NumericalAggregations["Min"] = "min";
    NumericalAggregations["Max"] = "max";
    NumericalAggregations["Avg"] = "avg";
    NumericalAggregations["Median"] = "median";
    NumericalAggregations["StandardDeviation"] = "std_dev";
    //   Histogram = 'histogram',
    NumericalAggregations["Range"] = "range";
})(NumericalAggregations || (NumericalAggregations = {}));
var CommonAggregations;
(function (CommonAggregations) {
    CommonAggregations["Count"] = "count";
    CommonAggregations["CountEmpty"] = "count_empty";
    CommonAggregations["CountFilled"] = "count_filled";
    CommonAggregations["CountUnique"] = "count_unique";
    CommonAggregations["PercentEmpty"] = "percent_empty";
    CommonAggregations["PercentFilled"] = "percent_filled";
    CommonAggregations["PercentUnique"] = "percent_unique";
    CommonAggregations["None"] = "none";
})(CommonAggregations || (CommonAggregations = {}));
var AttachmentAggregations;
(function (AttachmentAggregations) {
    AttachmentAggregations["AttachmentSize"] = "attachment_size";
})(AttachmentAggregations || (AttachmentAggregations = {}));
var BooleanAggregations;
(function (BooleanAggregations) {
    BooleanAggregations["Checked"] = "checked";
    BooleanAggregations["Unchecked"] = "unchecked";
    BooleanAggregations["PercentChecked"] = "percent_checked";
    BooleanAggregations["PercentUnchecked"] = "percent_unchecked";
})(BooleanAggregations || (BooleanAggregations = {}));
var DateAggregations;
(function (DateAggregations) {
    DateAggregations["EarliestDate"] = "earliest_date";
    DateAggregations["LatestDate"] = "latest_date";
    DateAggregations["DateRange"] = "date_range";
    DateAggregations["MonthRange"] = "month_range";
})(DateAggregations || (DateAggregations = {}));
const AllAggregations = Object.assign(Object.assign(Object.assign(Object.assign(Object.assign({}, CommonAggregations), NumericalAggregations), AttachmentAggregations), BooleanAggregations), DateAggregations);
const getAvailableAggregations = (type, parsed_tree) => {
    let returnAggregations = [];
    if (type === UITypes.Formula && (parsed_tree === null || parsed_tree === void 0 ? void 0 : parsed_tree.dataType)) {
        switch (parsed_tree.dataType) {
            case FormulaDataTypes.BOOLEAN:
                returnAggregations = [
                    ...Object.values(BooleanAggregations),
                    CommonAggregations.None,
                ];
                break;
            case FormulaDataTypes.DATE:
                returnAggregations = [
                    ...Object.values(DateAggregations),
                    ...Object.values(CommonAggregations),
                ];
                break;
            case FormulaDataTypes.NUMERIC:
                returnAggregations = [
                    ...Object.values(NumericalAggregations),
                    ...Object.values(CommonAggregations),
                ];
                break;
            default:
                returnAggregations = [...Object.values(CommonAggregations)];
                break;
        }
    }
    switch (type) {
        case UITypes.Number:
        case UITypes.Decimal:
        case UITypes.Currency:
        case UITypes.Percent:
        case UITypes.Duration:
        case UITypes.Rating:
        case UITypes.Rollup:
        case UITypes.Links:
            returnAggregations = [
                ...Object.values(NumericalAggregations),
                ...Object.values(CommonAggregations),
            ];
            break;
        case UITypes.Checkbox:
            returnAggregations = [
                ...Object.values(BooleanAggregations),
                CommonAggregations.None,
            ];
            break;
        case UITypes.Date:
        case UITypes.DateTime:
        case UITypes.LastModifiedTime:
        case UITypes.CreatedTime:
            returnAggregations = [
                ...Object.values(DateAggregations),
                ...Object.values(CommonAggregations),
            ];
            break;
        case UITypes.SpecificDBType:
        case UITypes.ForeignKey:
            returnAggregations = [CommonAggregations.None];
            break;
        case UITypes.Button:
        case UITypes.Attachment:
            return [CommonAggregations.None, AttachmentAggregations.AttachmentSize];
    }
    if (!returnAggregations.length) {
        returnAggregations = [...Object.values(CommonAggregations)];
    }
    return returnAggregations;
};
const formatAggregation = (aggregation, value, column, 
/**
 * If columnHelperParams is provided then it will be used to format the aggregation value
 */
columnHelperParams) => {
    var _a, _b, _c, _d, _e, _f;
    if (column.uidt === UITypes.Formula) {
        if ((_a = column === null || column === void 0 ? void 0 : column.meta) === null || _a === void 0 ? void 0 : _a.display_type) {
            const childColumn = Object.assign({ uidt: (_b = column === null || column === void 0 ? void 0 : column.meta) === null || _b === void 0 ? void 0 : _b.display_type }, (_c = column === null || column === void 0 ? void 0 : column.meta) === null || _c === void 0 ? void 0 : _c.display_column_meta);
            return formatAggregation(aggregation, value, childColumn, Object.assign(Object.assign({}, columnHelperParams), { col: childColumn }));
        }
    }
    if ([DateAggregations.EarliestDate, DateAggregations.LatestDate].includes(aggregation)) {
        if (column.uidt === UITypes.DateTime) {
            return getDateTimeValue(value, column);
        }
        else if (column.uidt === UITypes.Date) {
            return getDateValue(value, column);
        }
        return getDateTimeValue(value, column);
    }
    if ([
        CommonAggregations.PercentEmpty,
        CommonAggregations.PercentFilled,
        CommonAggregations.PercentUnique,
        BooleanAggregations.PercentChecked,
        BooleanAggregations.PercentUnchecked,
    ].includes(aggregation)) {
        return `${(_d = roundTo(value, 1)) !== null && _d !== void 0 ? _d : 0}%`;
    }
    if ([DateAggregations.MonthRange, DateAggregations.DateRange].includes(aggregation)) {
        return aggregation === DateAggregations.DateRange
            ? `${value !== null && value !== void 0 ? value : 0} days`
            : `${value !== null && value !== void 0 ? value : 0} months`;
    }
    if ([
        CommonAggregations.Count,
        CommonAggregations.CountEmpty,
        CommonAggregations.CountFilled,
        CommonAggregations.CountUnique,
    ].includes(aggregation)) {
        return value;
    }
    if ([AttachmentAggregations.AttachmentSize].includes(aggregation)) {
        return formatBytes(value !== null && value !== void 0 ? value : 0);
    }
    if (columnHelperParams &&
        (columnHelperParams === null || columnHelperParams === void 0 ? void 0 : columnHelperParams.col) &&
        Object.values(NumericalAggregations).includes(aggregation) &&
        [
            UITypes.Number,
            UITypes.Decimal,
            UITypes.Currency,
            UITypes.Percent,
            UITypes.Duration,
            UITypes.Rollup,
            UITypes.Time,
        ].includes(column.uidt)) {
        return ColumnHelper.parsePlainCellValue(value, Object.assign(Object.assign({}, columnHelperParams), { isAggregation: true }));
    }
    if (column.uidt === UITypes.Currency) {
        return getCurrencyValue(value, column);
    }
    if (column.uidt === UITypes.Percent) {
        return `${roundTo(value, 1)}%`;
    }
    if (column.uidt === UITypes.Duration) {
        return convertMS2Duration(value, ((_e = parseProp(column.meta)) === null || _e === void 0 ? void 0 : _e.duration) || 0);
    }
    if (typeof value === 'number') {
        return (_f = roundTo(value, 1)) !== null && _f !== void 0 ? _f : '∞';
    }
    return value;
};
export { getAvailableAggregations, NumericalAggregations, CommonAggregations, BooleanAggregations, DateAggregations, AttachmentAggregations, AllAggregations, formatAggregation, };
//# sourceMappingURL=data:application/json;base64,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