import dayjs from 'dayjs';
import { dateFormats } from '../../../lib/dateTimeHelper';
import { DATE_DIFF_TO_SCALE_LABEL_MAP, DATE_SCALE_LABEL_TO_DIFF_MAP, isNumberRound, parseDateValue, serializeDateOrDateTimeValue, } from '..';
import AbstractColumnHelper from '../column.interface';
import { SilentTypeConversionError } from '../../../lib/error';
import { parseProp } from '../../../lib/helperFunctions';
import { ncIsNullOrUndefined, ncIsUndefined } from '../../../lib/is';
import { populateFillHandleStrictCopy } from '../utils/fill-handler';
export class DateHelper extends AbstractColumnHelper {
    constructor() {
        super(...arguments);
        this.columnDefaultMeta = {
            date_format: dateFormats[0],
        };
    }
    serializeValue(value, params) {
        if (params.serializeSearchQuery)
            return null;
        value = serializeDateOrDateTimeValue(value, params);
        if (value === null) {
            if (params.isMultipleCellPaste) {
                return null;
            }
            else {
                throw new SilentTypeConversionError();
            }
        }
        return value;
    }
    parseValue(value, params) {
        return parseDateValue(value, params.col, params.isSystemCol);
    }
    parsePlainCellValue(value, params) {
        var _a;
        return (_a = parseDateValue(value, params.col, params.isSystemCol)) !== null && _a !== void 0 ? _a : '';
    }
    populateFillHandle(params) {
        var _a;
        const { column, highlightedData, numberOfRows } = params;
        if (highlightedData.length < 2) {
            return populateFillHandleStrictCopy(params);
        }
        // data is in form like 'YYYY-MM-DD' depends on meta dateformat
        const meta = parseProp(column.meta);
        const metaDateFormat = (_a = meta.date_format) !== null && _a !== void 0 ? _a : 'YYYY-MM-DD';
        const dateFormat = metaDateFormat + ' HH:mm:ss';
        let lastData;
        let modifier = undefined;
        let scale = 'D';
        const setModifierForScaleIfRound = (currentData, lastData, diffScale) => {
            const currentModifier = currentData.diff(lastData, diffScale, true);
            if (isNumberRound(currentModifier)) {
                scale = DATE_DIFF_TO_SCALE_LABEL_MAP[diffScale];
                modifier = currentModifier;
            }
        };
        // map to dayjs
        const dayJsHighlightedData = [];
        for (const date of highlightedData) {
            if (ncIsNullOrUndefined(date) ||
                date === '' ||
                typeof date !== 'string') {
                return populateFillHandleStrictCopy(params);
            }
            const currentData = dayjs(date + ' 00:00:00', dateFormat);
            // unlikely on normal case
            if (!currentData.isValid()) {
                return populateFillHandleStrictCopy(params);
            }
            if (!lastData) {
                lastData = currentData;
            }
            else {
                if (ncIsUndefined(modifier)) {
                    const modifierInDays = currentData.diff(lastData, 'day');
                    if (modifierInDays > 354) {
                        setModifierForScaleIfRound(currentData, lastData, 'year');
                    }
                    else if (modifierInDays > 28) {
                        setModifierForScaleIfRound(currentData, lastData, 'month');
                    }
                    else {
                        modifier = modifierInDays;
                        scale = 'D';
                    }
                }
                else {
                    const currentModifier = currentData.diff(lastData, DATE_SCALE_LABEL_TO_DIFF_MAP[scale]);
                    if (currentModifier !== modifier) {
                        return populateFillHandleStrictCopy(params);
                    }
                }
                lastData = currentData;
            }
            dayJsHighlightedData.push(lastData);
        }
        if (modifier === 0) {
            return populateFillHandleStrictCopy(params);
        }
        const numberToGenerate = numberOfRows - highlightedData.length;
        return Array.from({ length: numberToGenerate }).map(() => {
            const currentData = lastData.add(modifier, DATE_SCALE_LABEL_TO_DIFF_MAP[scale]);
            lastData = currentData;
            return currentData.format(metaDateFormat);
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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