import { constructTimeFormat, dateFormats, timeFormats, } from '../../../lib/dateTimeHelper';
import AbstractColumnHelper from '../column.interface';
import { DATE_DIFF_TO_SCALE_LABEL_MAP, DATE_SCALE_LABEL_TO_DIFF_MAP, isNumberRound, parseDateTimeValue, serializeDateOrDateTimeValue, } from '../utils';
import { SilentTypeConversionError } from '../../../lib/error';
import { populateFillHandleStrictCopy } from '../utils/fill-handler';
import { parseProp } from '../../../lib/helperFunctions';
import dayjs from 'dayjs';
import { ncIsNullOrUndefined, ncIsUndefined } from '../../../lib/is';
export class DateTimeHelper extends AbstractColumnHelper {
    constructor() {
        super(...arguments);
        this.columnDefaultMeta = {
            date_format: dateFormats[0],
            time_format: timeFormats[0],
            is12hrFormat: false,
        };
    }
    serializeValue(value, params) {
        if (params.serializeSearchQuery)
            return null;
        value = serializeDateOrDateTimeValue(value, params);
        if (value === null) {
            if (params.isMultipleCellPaste) {
                return null;
            }
            else {
                throw new SilentTypeConversionError();
            }
        }
        return value;
    }
    parseValue(value, params) {
        return parseDateTimeValue(value, params);
    }
    parsePlainCellValue(value, params) {
        var _a;
        return (_a = parseDateTimeValue(value, params)) !== null && _a !== void 0 ? _a : '';
    }
    populateFillHandle(params) {
        var _a;
        const { column, highlightedData, numberOfRows } = params;
        if (highlightedData.length < 2) {
            return populateFillHandleStrictCopy(params);
        }
        // data is in form like 'YYYY-MM-DD HH:mm' depends on meta dateformat
        const meta = parseProp(column.meta);
        const metaDateFormat = (_a = meta.date_format) !== null && _a !== void 0 ? _a : 'YYYY-MM-DD';
        const metaTimeFormat = constructTimeFormat(params.column);
        const dateTimeFormat = `${metaDateFormat} ${metaTimeFormat}`;
        let lastData;
        let modifier = undefined;
        let scale = 's';
        let canUseSecondScale = true;
        if (!dateTimeFormat.includes('ss')) {
            canUseSecondScale = false;
        }
        const setModifierForScaleIfRound = (currentData, lastData, diffScale) => {
            const currentModifier = currentData.diff(lastData, diffScale, true);
            if (isNumberRound(currentModifier)) {
                scale = DATE_DIFF_TO_SCALE_LABEL_MAP[diffScale];
                modifier = currentModifier;
            }
        };
        // map to dayjs
        const dayJsHighlightedData = [];
        for (const date of highlightedData) {
            if (ncIsNullOrUndefined(date) ||
                date === '' ||
                typeof date !== 'string') {
                return populateFillHandleStrictCopy(params);
            }
            const currentData = dayjs(date, dateTimeFormat);
            // unlikely on normal case
            if (!currentData.isValid()) {
                return populateFillHandleStrictCopy(params);
            }
            if (!lastData) {
                lastData = currentData;
            }
            else {
                if (ncIsUndefined(modifier)) {
                    const modifierInSeconds = currentData.diff(lastData, 'second');
                    if (modifierInSeconds > 60 * 60 * 24 * 354) {
                        setModifierForScaleIfRound(currentData, lastData, 'year');
                    }
                    else if (modifierInSeconds > 60 * 60 * 24 * 28) {
                        setModifierForScaleIfRound(currentData, lastData, 'month');
                    }
                    else if (modifierInSeconds > 60 * 60 * 24) {
                        setModifierForScaleIfRound(currentData, lastData, 'day');
                    }
                    else if (modifierInSeconds > 60 * 60) {
                        setModifierForScaleIfRound(currentData, lastData, 'hour');
                    }
                    else if (modifierInSeconds > 60) {
                        setModifierForScaleIfRound(currentData, lastData, 'minute');
                    }
                    else if (!canUseSecondScale) {
                        modifier = currentData.diff(lastData, 'minute');
                        scale = 'm';
                    }
                    else {
                        modifier = modifierInSeconds;
                        scale = 's';
                    }
                }
                else {
                    const currentModifier = currentData.diff(lastData, DATE_SCALE_LABEL_TO_DIFF_MAP[scale]);
                    if (currentModifier !== modifier) {
                        return populateFillHandleStrictCopy(params);
                    }
                }
                lastData = currentData;
            }
            dayJsHighlightedData.push(lastData);
        }
        if (modifier === 0) {
            return populateFillHandleStrictCopy(params);
        }
        const numberToGenerate = numberOfRows - highlightedData.length;
        return Array.from({ length: numberToGenerate }).map(() => {
            const currentData = lastData.add(modifier, DATE_SCALE_LABEL_TO_DIFF_MAP[scale]);
            lastData = currentData;
            return currentData.format(dateTimeFormat);
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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