import { parseDateTimeValue, parseDateValue, serializeDateOrDateTimeValue, } from './date-time';
import dayjs from 'dayjs';
import UITypes from '../../../lib/UITypes';
describe('parse date', () => {
    it('should parse a valid date string with default format', () => {
        const col = { meta: null, uidt: UITypes.Date };
        expect(parseDateValue('2023-10-27', col)).toBe('2023-10-27');
    });
    it('should parse a valid date string with custom format', () => {
        const col = { meta: { date_format: 'MM/DD/YYYY' }, uidt: UITypes.Date };
        expect(parseDateValue('10/27/2023', col)).toBe('10/27/2023');
    });
    it('should parse a valid numeric timestamp', () => {
        const col = { meta: null, uidt: UITypes.Date };
        // Timestamp for 2023-10-27T00:00:00.000Z
        expect(parseDateValue('1698364800000', col)).toBe('2023-10-27');
    });
    it('should parse a valid date string for a system column', () => {
        const col = { meta: null, uidt: UITypes.Date };
        expect(parseDateValue('2023-10-27 10:30:00', col, true)).toBe('2023-10-27 10:30:00');
    });
    it('should serialize a valid date value for Date type', () => {
        const col = { uidt: UITypes.Date, meta: null };
        expect(serializeDateOrDateTimeValue('2023-10-27', { col: col })).toBe('2023-10-27');
    });
});
describe('parse date-time', () => {
    it('should parse a valid ISO datetime string with quotes', () => {
        const col = { uidt: UITypes.DateTime, meta: {} };
        const params = { col, isMysql: () => false };
        const value = '"2023-10-27T10:30:00.000Z"';
        const expected = dayjs(value.replace(/["']/g, '')).format('YYYY-MM-DD HH:mm');
        expect(parseDateTimeValue(value, params)).toBe(expected);
    });
    it('should parse a valid ISO datetime string without quotes', () => {
        const col = { uidt: UITypes.DateTime, meta: {} };
        const params = { col, isMysql: () => false };
        const value = '2023-10-27T10:30:00.000Z';
        const expected = dayjs(value).format('YYYY-MM-DD HH:mm');
        expect(parseDateTimeValue(value, params)).toBe(expected);
    });
    it('should parse a valid datetime string in YYYY-MM-DD HH:mm:ss format with isMysql true', () => {
        const col = { uidt: UITypes.DateTime, meta: {} };
        const params = { col, isMysql: () => true };
        const value = '2023-10-27 10:30:00';
        const expected = dayjs(value, 'YYYY-MM-DD HH:mm:ss').format('YYYY-MM-DD HH:mm');
        expect(parseDateTimeValue(value, params)).toBe(expected);
    });
    it('should parse a valid datetime string in YYYY-MM-DD HH:mm:ssZ format', () => {
        const col = { uidt: UITypes.DateTime, meta: {} };
        const params = { col, isMysql: () => false };
        const value = '2023-10-27 10:30:00Z';
        const expected = dayjs(value, 'YYYY-MM-DD HH:mm:ssZ').format('YYYY-MM-DD HH:mm');
        expect(parseDateTimeValue(value, params)).toBe(expected);
    });
    it('should return null for invalid datetime string', () => {
        const col = { uidt: UITypes.DateTime, meta: {} };
        const params = { col, isMysql: () => false };
        const value = 'invalid-date-string';
        expect(parseDateTimeValue(value, params)).toBeNull();
    });
    it('should return null for null input', () => {
        const col = { uidt: UITypes.DateTime, meta: {} };
        const params = { col, isMysql: () => false };
        const value = null;
        expect(parseDateTimeValue(value, params)).toBeFalsy();
    });
});
describe('serialize', () => {
    it('should serialize a valid datetime value for non-Date type', () => {
        const col = {
            uidt: UITypes.DateTime,
            meta: {},
        };
        const value = '2023-10-27 10:00:00';
        const expected = dayjs(value).utc().format('YYYY-MM-DD HH:mm:ssZ');
        expect(serializeDateOrDateTimeValue(value, { col: col })).toBe(expected);
    });
    it('should serialize a date string to datetime format for DateTime type', () => {
        const col = { uidt: UITypes.DateTime, meta: {} };
        const value = '2023-10-27';
        const expected = dayjs(value).utc().format('YYYY-MM-DD HH:mm:ssZ');
        expect(serializeDateOrDateTimeValue(value, { col: col })).toBe(expected);
    });
    it('should serialize a date string to datetime format for custom date format (1)', () => {
        const col = {
            uidt: UITypes.Date,
            meta: { date_format: 'MM/DD/YYYY' },
        };
        const value = '09/05/1980';
        const expected = dayjs(value, 'MM/DD/YYYY').format('YYYY-MM-DD');
        expect(serializeDateOrDateTimeValue(value, { col: col })).toBe(expected);
    });
    it('should serialize a date string to datetime format for custom date format (2)', () => {
        const col = {
            uidt: UITypes.Date,
            meta: { date_format: 'DD/MM/YYYY' },
        };
        const value = '09/05/1980';
        const expected = dayjs(value, 'DD/MM/YYYY').format('YYYY-MM-DD');
        expect(serializeDateOrDateTimeValue(value, { col: col })).toBe(expected);
    });
});
//# sourceMappingURL=data:application/json;base64,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