import dayjs from 'dayjs';
import { convertMS2Duration } from '../../../lib/durationUtils';
import { parseProp, roundUpToPrecision } from '../../../lib/helperFunctions';
import { ncIsArray, ncIsBoolean, ncIsNaN, ncIsNull, ncIsNumber, ncIsObject, ncIsString, ncIsUndefined, } from '../../../lib/is';
import { constructTimeFormat } from '../../../lib/dateTimeHelper';
import { checkboxTypeMap } from '../../../lib/columnHelper/utils/common';
export const parseDefault = (value) => {
    var _a;
    try {
        if (typeof value === 'object') {
            return JSON.stringify(value);
        }
        else {
            return (_a = value === null || value === void 0 ? void 0 : value.toString()) !== null && _a !== void 0 ? _a : null;
        }
    }
    catch (_b) {
        return null;
    }
};
export const parseIntValue = (value, col) => {
    if (ncIsNaN(value)) {
        return null;
    }
    value = parseInt(value.toString(), 10);
    const columnMeta = parseProp(col === null || col === void 0 ? void 0 : col.meta);
    if (columnMeta.isLocaleString) {
        return Number(value).toLocaleString();
    }
    return Number(value);
};
export const parseDecimalValue = (value, col) => {
    var _a, _b, _c, _d;
    if (ncIsNaN(value)) {
        return null;
    }
    const columnMeta = parseProp(col.meta);
    if (columnMeta.isLocaleString) {
        return Number(roundUpToPrecision(Number(value), (_a = columnMeta.precision) !== null && _a !== void 0 ? _a : 1)).toLocaleString(undefined, {
            minimumFractionDigits: (_b = columnMeta.precision) !== null && _b !== void 0 ? _b : 1,
            maximumFractionDigits: (_c = columnMeta.precision) !== null && _c !== void 0 ? _c : 1,
        });
    }
    return roundUpToPrecision(Number(value), (_d = columnMeta.precision) !== null && _d !== void 0 ? _d : 1);
};
export const parsePercentValue = (value, col) => {
    var _a;
    if (ncIsNaN(value)) {
        return null;
    }
    /**
     * We have to keep cell display and parse value (copy) consistent
     * ref: check `formatPercentage` function in `~/utils/cell.ts`
     */
    if (Number(value) % 1 === 0) {
        return `${Number(value)}%`;
    }
    const columnMeta = parseProp(col.meta);
    const percentValue = roundUpToPrecision(Number(value), (_a = columnMeta.precision) !== null && _a !== void 0 ? _a : 2);
    return `${percentValue}%`;
};
export const parseDurationValue = (value, col) => {
    var _a;
    const durationType = ((_a = parseProp(col.meta)) === null || _a === void 0 ? void 0 : _a.duration) || 0;
    return convertMS2Duration(value, durationType);
};
export const parseCheckboxValue = (value) => {
    if (ncIsBoolean(value))
        return value;
    if (ncIsString(value)) {
        const strval = value.trim().toLowerCase();
        const parsedValue = checkboxTypeMap[strval];
        if (!ncIsNull(parsedValue) && !ncIsUndefined(parsedValue)) {
            return parsedValue;
        }
    }
    if (ncIsNumber(value)) {
        if (value === 1)
            return true;
        if (value === 0)
            return false;
    }
    return null;
};
export const parseJsonValue = (value) => {
    try {
        return ncIsString(value)
            ? JSON.stringify(JSON.parse(value)) // Ensure it's a valid JSON string
            : JSON.stringify(value); // Convert object/array to JSON
    }
    catch (_a) {
        return null;
    }
};
export const parseCurrencyValue = (value, col) => {
    var _a, _b, _c;
    if (ncIsNaN(value)) {
        return null;
    }
    const columnMeta = parseProp(col.meta);
    try {
        // Round the value to the specified precision
        const roundedValue = roundUpToPrecision(Number(value), (_a = columnMeta.precision) !== null && _a !== void 0 ? _a : 2);
        return new Intl.NumberFormat(columnMeta.currency_locale || 'en-US', {
            style: 'currency',
            currency: columnMeta.currency_code || 'USD',
            minimumFractionDigits: (_b = columnMeta.precision) !== null && _b !== void 0 ? _b : 2,
            maximumFractionDigits: (_c = columnMeta.precision) !== null && _c !== void 0 ? _c : 2,
        }).format(+roundedValue);
    }
    catch (_d) {
        return value;
    }
};
export const parseTimeValue = (value, params) => {
    var _a, _b;
    value = value === null || value === void 0 ? void 0 : value.toString().trim();
    if (!value)
        return null;
    // remove `"`
    // e.g. "2023-05-12T08:03:53.000Z" -> 2023-05-12T08:03:53.000Z
    value = value.replace(/["']/g, '');
    const isMySQL = (_a = params.isMysql) === null || _a === void 0 ? void 0 : _a.call(params, params.col.source_id);
    const isPostgres = (_b = params.isPg) === null || _b === void 0 ? void 0 : _b.call(params, params.col.source_id);
    let d = dayjs(value);
    if (!d.isValid()) {
        // insert a datetime value, copy the value without refreshing
        // e.g. value = 2023-05-12T03:49:25.000Z
        // feed custom parse format
        d = dayjs(value, isMySQL ? 'YYYY-MM-DD HH:mm:ss' : 'YYYY-MM-DD HH:mm:ssZ');
    }
    if (!d.isValid()) {
        // MySQL and Postgres store time in HH:mm:ss format so we need to feed custom parse format
        d = isMySQL || isPostgres ? dayjs(value, 'HH:mm:ss') : dayjs(value);
    }
    if (!d.isValid()) {
        d = dayjs(`1999-01-01 ${value}`);
    }
    if (!d.isValid()) {
        // return empty string for invalid time
        return null;
    }
    return d.format(constructTimeFormat(params.col));
};
export const parseYearValue = (value) => {
    value = value === null || value === void 0 ? void 0 : value.toString().trim();
    if (!value || !dayjs(value).isValid()) {
        return null;
    }
    value = dayjs(value, 'YYYY').format('YYYY');
    return value ? +value : value;
};
export const parseUserValue = (value, withDisplayName = false, useUserId = false) => {
    let data = value;
    try {
        if (typeof value === 'string') {
            data = JSON.parse(value);
        }
    }
    catch (_a) { }
    return (ncIsArray(data) ? data : ncIsObject(data) ? [data] : [])
        .map((user) => withDisplayName && user.display_name
        ? `${user.display_name}<${user.email}>`
        : useUserId
            ? `${user.id}`
            : `${user.email}`)
        .join(', ');
};
export const parseLinksValue = (value, params) => {
    var _a, _b, _c;
    const { col, t } = params;
    const columnMeta = parseProp(col.meta);
    const parsedValue = +value || 0;
    if (!parsedValue) {
        return `0 ${columnMeta.plural || ((_a = t === null || t === void 0 ? void 0 : t('general.links')) !== null && _a !== void 0 ? _a : 'Links')}`;
    }
    else if (parsedValue === 1) {
        return `1 ${columnMeta.singular || ((_b = t === null || t === void 0 ? void 0 : t('general.link')) !== null && _b !== void 0 ? _b : 'Link')}`;
    }
    else {
        return `${parsedValue} ${columnMeta.plural || ((_c = t('general.links')) !== null && _c !== void 0 ? _c : 'Links')}`;
    }
};
//# sourceMappingURL=data:application/json;base64,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