import dayjs from 'dayjs';
import { convertDurationToSeconds } from '../../../lib/durationUtils';
import { parseProp } from '../../../lib/helperFunctions';
import { ncIsBoolean, ncIsFunction, ncIsNaN, ncIsNull, ncIsNumber, ncIsString, ncIsUndefined, } from '../../../lib/is';
import UITypes from '../../../lib/UITypes';
import { SelectTypeConversionError } from '../../../lib/error';
import { checkboxTypeMap } from '../../../lib/columnHelper/utils/common';
import { getGroupDecimalSymbolFromLocale } from '../../../lib/currencyHelpers';
/**
 * Remove outer quotes & unescape
 */
export const serializeStringValue = (value) => {
    var _a;
    value = (_a = value === null || value === void 0 ? void 0 : value.toString()) !== null && _a !== void 0 ? _a : null;
    if (!value)
        return null;
    if (value.match(/^"[\s\S]*"$/)) {
        value = value.slice(1, -1).replace(/\\"/g, '"');
    }
    return value;
};
export const serializeDecimalValue = (value, callback, params) => {
    var _a;
    // If we have clipboard data, use it
    if (((_a = params === null || params === void 0 ? void 0 : params.clipboardItem) === null || _a === void 0 ? void 0 : _a.dbCellValue) &&
        ncIsNumber(params.clipboardItem.dbCellValue)) {
        return params.clipboardItem.dbCellValue;
    }
    if (ncIsNumber(value)) {
        return Number(value);
    }
    // If it's a string, remove commas and check if it's a valid number
    if (ncIsString(value)) {
        const cleanedValue = ncIsFunction(callback)
            ? callback(value)
            : value
                .replace(/[\s\u00A0]/g, '') // remove spaces/non-breaking spaces
                .replace(/(?!^-)[^\d.-]/g, ''); // keep only digits, dot, one leading minus
        if (!cleanedValue)
            return null;
        // Try converting the cleaned value to a number
        const numberValue = Number(cleanedValue);
        // If it's a valid number, return it
        if (!isNaN(numberValue)) {
            return numberValue;
        }
    }
    return null;
};
export const serializeIntValue = (value) => {
    value = serializeDecimalValue(value);
    if (ncIsNumber(value)) {
        return parseInt(value.toString(), 10);
    }
    return null; // Return null if it's not a valid number
};
export const serializeDurationValue = (value, col) => {
    var _a;
    // Check if the value is a pure number (interpreted as seconds)
    if (!ncIsNaN(value)) {
        return parseInt(value, 10); // Directly return seconds
    }
    const columnMeta = parseProp(col.meta);
    const res = convertDurationToSeconds(value, (_a = columnMeta.duration) !== null && _a !== void 0 ? _a : 0);
    return res._isValid ? res._sec : null;
};
export const serializeCheckboxValue = (value) => {
    if (ncIsBoolean(value))
        return value;
    if (ncIsString(value)) {
        const strval = value.trim().toLowerCase();
        const parsedValue = checkboxTypeMap[strval];
        if (!ncIsNull(parsedValue) && !ncIsUndefined(parsedValue)) {
            return parsedValue;
        }
    }
    return null;
};
export const serializeJsonValue = (value) => {
    try {
        return ncIsString(value)
            ? JSON.stringify(JSON.parse(value)) // Ensure it's a valid JSON string
            : JSON.stringify(value); // Convert object/array to JSON
    }
    catch (_a) {
        return null;
    }
};
export const serializeCurrencyValue = (value, params) => {
    var _a;
    // If we have clipboard data, use it
    if (((_a = params === null || params === void 0 ? void 0 : params.clipboardItem) === null || _a === void 0 ? void 0 : _a.dbCellValue) &&
        ncIsNumber(params.clipboardItem.dbCellValue)) {
        return params.clipboardItem.dbCellValue;
    }
    return serializeDecimalValue(value, (value) => {
        var _a;
        const columnMeta = parseProp(params.col.meta);
        // Create a number formatter for the target locale (e.g., 'de-DE', 'en-US')
        const formatter = new Intl.NumberFormat((columnMeta === null || columnMeta === void 0 ? void 0 : columnMeta.currency_locale) || 'en-US');
        // If the locale is not set or is 'en-US', or the formatter does not support formatToParts, use the default behavior
        if (!(columnMeta === null || columnMeta === void 0 ? void 0 : columnMeta.currency_locale) ||
            columnMeta.currency_locale === 'en-US' ||
            typeof formatter.formatToParts !== 'function') {
            return (_a = value === null || value === void 0 ? void 0 : value.replace(/[^0-9.]/g, '')) === null || _a === void 0 ? void 0 : _a.trim();
        }
        const { group, decimal } = getGroupDecimalSymbolFromLocale(columnMeta === null || columnMeta === void 0 ? void 0 : columnMeta.currency_locale);
        return value
            .replace(new RegExp('\\' + group, 'g'), '') // 1. Remove all group (thousands) separators
            .replace(new RegExp('\\' + decimal), '.') // 2. Replace the locale-specific decimal separator with a dot (.)
            .replace(/[^\d.-]/g, '') // 3. Remove any non-digit, non-dot, non-minus characters (e.g., currency symbols, spaces)
            .trim(); // 4. Trim whitespace from both ends of the string
    });
};
export const serializeTimeValue = (value, params) => {
    var _a;
    value = value === null || value === void 0 ? void 0 : value.toString().trim();
    if (!value)
        return null;
    let parsedTime = dayjs(value);
    if (!parsedTime.isValid()) {
        parsedTime = dayjs(value, 'HH:mm:ss');
    }
    if (!parsedTime.isValid()) {
        parsedTime = dayjs(`1999-01-01 ${value}`);
    }
    if (!parsedTime.isValid()) {
        return null;
    }
    const dateFormat = ((_a = params.isMysql) === null || _a === void 0 ? void 0 : _a.call(params, params.col.source_id))
        ? 'YYYY-MM-DD HH:mm:ss'
        : 'YYYY-MM-DD HH:mm:ssZ';
    return parsedTime.format(dateFormat);
};
export const serializeYearValue = (value) => {
    value = value === null || value === void 0 ? void 0 : value.toString().trim();
    if (!value)
        return null;
    value = serializeIntValue(value);
    const parsedDate = dayjs(value === null || value === void 0 ? void 0 : value.toString());
    return parsedDate.isValid() ? +parsedDate.format('YYYY') : null;
};
export const serializeSelectValue = (value, col) => {
    var _a;
    // Always trim values for select fields to prevent creating options with leading/trailing spaces
    value = value === null || value === void 0 ? void 0 : value.toString().trim();
    // return null if value is empty
    if (!value)
        return null;
    // Todo: discuss new value creation
    const availableOptions = (((_a = col.colOptions) === null || _a === void 0 ? void 0 : _a.options) || []).map((o) => o.title);
    const optionsSet = new Set(availableOptions);
    let vals = value.split(',').map((val) => val.trim());
    const invalidVals = vals.filter((v) => !optionsSet.has(v));
    if (vals.length && col.uidt === UITypes.SingleSelect) {
        vals = [vals[0]];
    }
    // return null if no valid values
    if (invalidVals.length > 0) {
        throw new SelectTypeConversionError(vals, invalidVals);
    }
    return vals.join(',');
};
export const serializeEmail = (v) => {
    const matches = v.match(/(([^<>()[\].,;:\s@"]+(\.[^<>()[\].,;:\s@"]+)*)|(".+"))@(([^<>()[\].,;:\s@"]+\.)+[^<>()[\].,;:\s@"]{2,})/i);
    return matches ? matches[0] : null;
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VyaWFsaXplci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uL3NyYy9saWIvY29sdW1uSGVscGVyL3V0aWxzL3NlcmlhbGl6ZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsT0FBTyxLQUFLLE1BQU0sT0FBTyxDQUFDO0FBRTFCLE9BQU8sRUFBRSx3QkFBd0IsRUFBRSxNQUFNLHFCQUFxQixDQUFDO0FBQy9ELE9BQU8sRUFBRSxTQUFTLEVBQUUsTUFBTSx1QkFBdUIsQ0FBQztBQUNsRCxPQUFPLEVBQ0wsV0FBVyxFQUNYLFlBQVksRUFDWixPQUFPLEVBQ1AsUUFBUSxFQUNSLFVBQVUsRUFDVixVQUFVLEVBQ1YsYUFBYSxHQUNkLE1BQU0sVUFBVSxDQUFDO0FBQ2xCLE9BQU8sT0FBTyxNQUFNLGVBQWUsQ0FBQztBQUVwQyxPQUFPLEVBQUUseUJBQXlCLEVBQUUsTUFBTSxhQUFhLENBQUM7QUFDeEQsT0FBTyxFQUFFLGVBQWUsRUFBRSxNQUFNLGlDQUFpQyxDQUFDO0FBQ2xFLE9BQU8sRUFBRSwrQkFBK0IsRUFBRSxNQUFNLHVCQUF1QixDQUFDO0FBRXhFOztHQUVHO0FBQ0gsTUFBTSxDQUFDLE1BQU0sb0JBQW9CLEdBQUcsQ0FBQyxLQUFVLEVBQUUsRUFBRTs7SUFDakQsS0FBSyxHQUFHLE1BQUEsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLFFBQVEsRUFBRSxtQ0FBSSxJQUFJLENBQUM7SUFFbEMsSUFBSSxDQUFDLEtBQUs7UUFBRSxPQUFPLElBQUksQ0FBQztJQUV4QixJQUFJLEtBQUssQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQztRQUMvQixLQUFLLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRCxPQUFPLEtBQUssQ0FBQztBQUNmLENBQUMsQ0FBQztBQUVGLE1BQU0sQ0FBQyxNQUFNLHFCQUFxQixHQUFHLENBQ25DLEtBQTZCLEVBQzdCLFFBQTRCLEVBQzVCLE1BQTRDLEVBQzVDLEVBQUU7O0lBQ0Ysb0NBQW9DO0lBQ3BDLElBQ0UsQ0FBQSxNQUFBLE1BQU0sYUFBTixNQUFNLHVCQUFOLE1BQU0sQ0FBRSxhQUFhLDBDQUFFLFdBQVc7UUFDbEMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLEVBQzVDLENBQUM7UUFDRCxPQUFPLE1BQU0sQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDO0lBQzFDLENBQUM7SUFFRCxJQUFJLFVBQVUsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3RCLE9BQU8sTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3ZCLENBQUM7SUFFRCxtRUFBbUU7SUFDbkUsSUFBSSxVQUFVLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUN0QixNQUFNLFlBQVksR0FBRyxZQUFZLENBQUMsUUFBUSxDQUFDO1lBQ3pDLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDO1lBQ2pCLENBQUMsQ0FBQyxLQUFLO2lCQUNGLE9BQU8sQ0FBQyxhQUFhLEVBQUUsRUFBRSxDQUFDLENBQUMsb0NBQW9DO2lCQUMvRCxPQUFPLENBQUMsZ0JBQWdCLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQywyQ0FBMkM7UUFFakYsSUFBSSxDQUFDLFlBQVk7WUFBRSxPQUFPLElBQUksQ0FBQztRQUUvQiwrQ0FBK0M7UUFDL0MsTUFBTSxXQUFXLEdBQUcsTUFBTSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBRXpDLG9DQUFvQztRQUNwQyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxFQUFFLENBQUM7WUFDeEIsT0FBTyxXQUFXLENBQUM7UUFDckIsQ0FBQztJQUNILENBQUM7SUFFRCxPQUFPLElBQUksQ0FBQztBQUNkLENBQUMsQ0FBQztBQUVGLE1BQU0sQ0FBQyxNQUFNLGlCQUFpQixHQUFHLENBQUMsS0FBNkIsRUFBRSxFQUFFO0lBQ2pFLEtBQUssR0FBRyxxQkFBcUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVyQyxJQUFJLFVBQVUsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3RCLE9BQU8sUUFBUSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQsT0FBTyxJQUFJLENBQUMsQ0FBQyx5Q0FBeUM7QUFDeEQsQ0FBQyxDQUFDO0FBRUYsTUFBTSxDQUFDLE1BQU0sc0JBQXNCLEdBQUcsQ0FDcEMsS0FBb0IsRUFDcEIsR0FBZSxFQUNmLEVBQUU7O0lBQ0YsK0RBQStEO0lBQy9ELElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixPQUFPLFFBQVEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQywwQkFBMEI7SUFDeEQsQ0FBQztJQUVELE1BQU0sVUFBVSxHQUFHLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7SUFFdkMsTUFBTSxHQUFHLEdBQUcsd0JBQXdCLENBQUMsS0FBSyxFQUFFLE1BQUEsVUFBVSxDQUFDLFFBQVEsbUNBQUksQ0FBQyxDQUFDLENBQUM7SUFFdEUsT0FBTyxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7QUFDeEMsQ0FBQyxDQUFDO0FBRUYsTUFBTSxDQUFDLE1BQU0sc0JBQXNCLEdBQUcsQ0FDcEMsS0FBNEMsRUFDNUMsRUFBRTtJQUNGLElBQUksV0FBVyxDQUFDLEtBQUssQ0FBQztRQUFFLE9BQU8sS0FBSyxDQUFDO0lBRXJDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLElBQUksRUFBRSxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBQzFDLE1BQU0sV0FBVyxHQUFHLGVBQWUsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM1QyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxFQUFFLENBQUM7WUFDMUQsT0FBTyxXQUFXLENBQUM7UUFDckIsQ0FBQztJQUNILENBQUM7SUFFRCxPQUFPLElBQUksQ0FBQztBQUNkLENBQUMsQ0FBQztBQUVGLE1BQU0sQ0FBQyxNQUFNLGtCQUFrQixHQUFHLENBQUMsS0FBVSxFQUFFLEVBQUU7SUFDL0MsSUFBSSxDQUFDO1FBQ0gsT0FBTyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ3RCLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxrQ0FBa0M7WUFDdEUsQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQywrQkFBK0I7SUFDNUQsQ0FBQztJQUFDLFdBQU0sQ0FBQztRQUNQLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztBQUNILENBQUMsQ0FBQztBQUVGLE1BQU0sQ0FBQyxNQUFNLHNCQUFzQixHQUFHLENBQ3BDLEtBQVUsRUFDVixNQUEyQyxFQUMzQyxFQUFFOztJQUNGLG9DQUFvQztJQUNwQyxJQUNFLENBQUEsTUFBQSxNQUFNLGFBQU4sTUFBTSx1QkFBTixNQUFNLENBQUUsYUFBYSwwQ0FBRSxXQUFXO1FBQ2xDLFVBQVUsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxFQUM1QyxDQUFDO1FBQ0QsT0FBTyxNQUFNLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQztJQUMxQyxDQUFDO0lBRUQsT0FBTyxxQkFBcUIsQ0FBQyxLQUFLLEVBQUUsQ0FBQyxLQUFLLEVBQUUsRUFBRTs7UUFDNUMsTUFBTSxVQUFVLEdBQUcsU0FBUyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDOUMsMkVBQTJFO1FBQzNFLE1BQU0sU0FBUyxHQUFHLElBQUksSUFBSSxDQUFDLFlBQVksQ0FDckMsQ0FBQSxVQUFVLGFBQVYsVUFBVSx1QkFBVixVQUFVLENBQUUsZUFBZSxLQUFJLE9BQU8sQ0FDdkMsQ0FBQztRQUVGLG9IQUFvSDtRQUNwSCxJQUNFLENBQUMsQ0FBQSxVQUFVLGFBQVYsVUFBVSx1QkFBVixVQUFVLENBQUUsZUFBZSxDQUFBO1lBQzVCLFVBQVUsQ0FBQyxlQUFlLEtBQUssT0FBTztZQUN0QyxPQUFRLFNBQWlCLENBQUMsYUFBYSxLQUFLLFVBQVUsRUFDdEQsQ0FBQztZQUNELE9BQU8sTUFBQSxLQUFLLGFBQUwsS0FBSyx1QkFBTCxLQUFLLENBQUUsT0FBTyxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsMENBQUUsSUFBSSxFQUFFLENBQUM7UUFDaEQsQ0FBQztRQUVELE1BQU0sRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLEdBQUcsK0JBQStCLENBQ3hELFVBQVUsYUFBVixVQUFVLHVCQUFWLFVBQVUsQ0FBRSxlQUFlLENBQzVCLENBQUM7UUFFRixPQUFPLEtBQUs7YUFDVCxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsSUFBSSxHQUFHLEtBQUssRUFBRSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyw2Q0FBNkM7YUFDeEYsT0FBTyxDQUFDLElBQUksTUFBTSxDQUFDLElBQUksR0FBRyxPQUFPLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxrRUFBa0U7YUFDM0csT0FBTyxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsQ0FBQywwRkFBMEY7YUFDbEgsSUFBSSxFQUFFLENBQUMsQ0FBQyxrREFBa0Q7SUFDL0QsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUM7QUFFRixNQUFNLENBQUMsTUFBTSxrQkFBa0IsR0FBRyxDQUNoQyxLQUFVLEVBQ1YsTUFBMkMsRUFDM0MsRUFBRTs7SUFDRixLQUFLLEdBQUcsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLFFBQVEsR0FBRyxJQUFJLEVBQUUsQ0FBQztJQUVqQyxJQUFJLENBQUMsS0FBSztRQUFFLE9BQU8sSUFBSSxDQUFDO0lBRXhCLElBQUksVUFBVSxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUU5QixJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUM7UUFDMUIsVUFBVSxHQUFHLEtBQUssQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVELElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQztRQUMxQixVQUFVLEdBQUcsS0FBSyxDQUFDLGNBQWMsS0FBSyxFQUFFLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDO1FBQzFCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVELE1BQU0sVUFBVSxHQUFHLENBQUEsTUFBQSxNQUFNLENBQUMsT0FBTyx1REFBRyxNQUFNLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQztRQUN2RCxDQUFDLENBQUMscUJBQXFCO1FBQ3ZCLENBQUMsQ0FBQyxzQkFBc0IsQ0FBQztJQUUzQixPQUFPLFVBQVUsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUM7QUFDdkMsQ0FBQyxDQUFDO0FBRUYsTUFBTSxDQUFDLE1BQU0sa0JBQWtCLEdBQUcsQ0FBQyxLQUFVLEVBQUUsRUFBRTtJQUMvQyxLQUFLLEdBQUcsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLFFBQVEsR0FBRyxJQUFJLEVBQUUsQ0FBQztJQUVqQyxJQUFJLENBQUMsS0FBSztRQUFFLE9BQU8sSUFBSSxDQUFDO0lBRXhCLEtBQUssR0FBRyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVqQyxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsS0FBSyxhQUFMLEtBQUssdUJBQUwsS0FBSyxDQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7SUFFNUMsT0FBTyxVQUFVLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO0FBQ2xFLENBQUMsQ0FBQztBQUVGLE1BQU0sQ0FBQyxNQUFNLG9CQUFvQixHQUFHLENBQUMsS0FBVSxFQUFFLEdBQWUsRUFBRSxFQUFFOztJQUNsRSxnR0FBZ0c7SUFDaEcsS0FBSyxHQUFHLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxRQUFRLEdBQUcsSUFBSSxFQUFFLENBQUM7SUFFakMsZ0NBQWdDO0lBQ2hDLElBQUksQ0FBQyxLQUFLO1FBQUUsT0FBTyxJQUFJLENBQUM7SUFFeEIsbUNBQW1DO0lBQ25DLE1BQU0sZ0JBQWdCLEdBQUcsQ0FDdkIsQ0FBQSxNQUFDLEdBQUcsQ0FBQyxVQUFnQywwQ0FBRSxPQUFPLEtBQUksRUFBRSxDQUNyRCxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRXRCLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFFN0MsSUFBSSxJQUFJLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBQ3JELE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBRTNELElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxHQUFHLENBQUMsSUFBSSxLQUFLLE9BQU8sQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUNyRCxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNuQixDQUFDO0lBRUQsaUNBQWlDO0lBQ2pDLElBQUksV0FBVyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQztRQUMzQixNQUFNLElBQUkseUJBQXlCLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUFDO0lBQ3pELENBQUM7SUFFRCxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7QUFDeEIsQ0FBQyxDQUFDO0FBRUYsTUFBTSxDQUFDLE1BQU0sY0FBYyxHQUFHLENBQUMsQ0FBUyxFQUFFLEVBQUU7SUFDMUMsTUFBTSxPQUFPLEdBQUcsQ0FBQyxDQUFDLEtBQUssQ0FDckIsMEdBQTBHLENBQzNHLENBQUM7SUFDRixPQUFPLE9BQU8sQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7QUFDckMsQ0FBQyxDQUFDIn0=