import { ncIsEmptyObject, ncIsObject } from './is';
/**
 * Extracts properties from an object.
 * @param body - The object to extract properties from.
 * @param props - The properties to extract.
 * @param _throwError - Whether to throw an error if no properties are found.
 * @returns - The extracted properties.
 */
export function extractProps(body, props, _throwError) {
    if (!ncIsObject(body))
        return {};
    // todo: throw error if no props found
    return props.reduce((o, key) => {
        if (key in body && body[key] !== undefined)
            o[key] = body[key];
        return o;
    }, {});
}
/**
 * Removes all `undefined` values and empty objects (`{}`) from an object.
 * Can optionally run recursively with the `deep` flag.
 *
 * - Works only on plain objects (`ncIsObject`).
 * - Arrays are preserved as-is (including `undefined` entries).
 * - Empty objects are always removed if they occur as object values.
 *
 * @typeParam T - Type of the input value.
 * @param obj - The object or value to clean.
 * @param deep - If `true`, cleans recursively. If `false`, cleans only top-level. Defaults to `true`.
 * @returns A cleaned copy of `obj`.
 *
 * @example
 * ```ts
 * const data = {
 *   a: 1,
 *   b: undefined,
 *   c: { d: 2, e: undefined, f: { g: undefined } },
 *   arr: [1, undefined, { x: undefined, y: 5 }]
 * }
 *
 * removeUndefinedFromObj(data)
 * // → { a: 1, c: { d: 2 }, arr: [1, undefined, { y: 5 }] }
 *
 * removeUndefinedFromObj(data, false)
 * // → { a: 1, c: { d: 2, f: { g: undefined } }, arr: [1, undefined, { x: undefined, y: 5 }] }
 * ```
 */
export const removeUndefinedFromObj = (obj, deep = true) => {
    if (ncIsObject(obj)) {
        const cleanedEntries = Object.entries(obj)
            .map(([k, v]) => {
            const cleanedValue = deep && (ncIsObject(v) || Array.isArray(v))
                ? removeUndefinedFromObj(v, deep)
                : v;
            return [k, cleanedValue];
        })
            .filter(([_, v]) => {
            if (v === undefined)
                return false;
            if (ncIsObject(v) && !Array.isArray(v) && ncIsEmptyObject(v))
                return false;
            return true;
        });
        return Object.keys(cleanedEntries).reduce((obj, key) => {
            obj[key] = cleanedEntries[key];
            return obj;
        }, {});
    }
    return obj;
};
//# sourceMappingURL=data:application/json;base64,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