import { ChartTypes, TextWidgetTypes, WidgetDataSourceTypes, WidgetTypes, } from '.';
import { AllAggregations, isBarcode, isButton, isLookup, isQrCode, isSystemColumn, UITypes, } from '../../lib';
export const calculateNextPosition = (existingWidgets, newWidgetDimensions, gridColumns = 4) => {
    if (existingWidgets.length === 0) {
        return { x: 0, y: 0 };
    }
    const { w: newW, h: newH } = newWidgetDimensions;
    // Early return if widget is too wide
    if (newW > gridColumns) {
        return { x: 0, y: 0 };
    }
    // Create a 2D grid to track occupied cells more efficiently
    const grid = new Map();
    let maxY = 0;
    // Build the occupation map
    for (const widget of existingWidgets) {
        const { x, y, w, h } = widget.position;
        const endY = y + h;
        maxY = Math.max(maxY, endY);
        for (let row = y; row < endY; row++) {
            if (!grid.has(row)) {
                grid.set(row, new Set());
            }
            const rowSet = grid.get(row);
            for (let col = x; col < x + w; col++) {
                rowSet.add(col);
            }
        }
    }
    // position checking
    const isPositionAvailable = (x, y) => {
        for (let row = y; row < y + newH; row++) {
            const rowSet = grid.get(row);
            if (rowSet) {
                for (let col = x; col < x + newW; col++) {
                    if (rowSet.has(col))
                        return false;
                }
            }
        }
        return true;
    };
    // Find the first available position, scanning row by row
    const maxX = gridColumns - newW;
    for (let y = 0; y <= maxY + 1; y++) {
        for (let x = 0; x <= maxX; x++) {
            if (isPositionAvailable(x, y)) {
                return { x, y };
            }
        }
    }
    // Fallback: place at bottom-left
    return { x: 0, y: maxY };
};
const defaultGaugeWidgetRange = [
    { color: '#FF6E76', min: 0, max: 33, label: 'Low' },
    { color: '#FDDD60', min: 33, max: 67, label: 'Medium' },
    { color: '#7CFFB2', min: 67, max: 100, label: 'High' },
];
const getDefaultConfig = (widgetType, chartType, columns) => {
    switch (widgetType) {
        case WidgetTypes.METRIC:
            return {
                dataSource: WidgetDataSourceTypes.MODEL,
                metric: {
                    type: 'count',
                    aggregation: 'count',
                },
                appearance: {
                    type: 'filled',
                    theme: 'purple',
                },
            };
        case WidgetTypes.GAUGE:
            return {
                dataSource: WidgetDataSourceTypes.MODEL,
                metric: {
                    type: 'count',
                    aggregation: 'count',
                },
                appearance: {
                    showValue: true,
                    ranges: defaultGaugeWidgetRange,
                },
            };
        case WidgetTypes.CHART:
            return getDefaultChartConfig(chartType, columns);
        case WidgetTypes.IFRAME:
            return {
                url: 'https://nocodb.com',
                allowFullScreen: false,
            };
        case WidgetTypes.TEXT:
            return {
                content: 'Enter text here',
                type: TextWidgetTypes.Markdown,
                formatting: {
                    horizontalAlign: 'flex-start',
                    verticalAlign: 'flex-start',
                    bold: false,
                    italic: false,
                    underline: false,
                    strikethrough: false,
                },
                appearance: {
                    font: {
                        family: 'Arial',
                        weight: 400,
                        size: 16,
                        lineHeight: 1,
                    },
                    color: '#000000',
                },
            };
        default:
            return {};
    }
};
const getDefaultChartConfig = (chartType, columns) => {
    const baseConfig = {
        dataSource: WidgetDataSourceTypes.MODEL,
    };
    // Helper function to get the best column for category field
    const getDefaultCategoryColumn = (columns) => {
        var _a;
        if (!columns || (columns === null || columns === void 0 ? void 0 : columns.length) === 0)
            return '';
        // Priority 1: SingleSelect
        const singleSelectColumn = columns.find((col) => col.uidt === UITypes.SingleSelect);
        if (singleSelectColumn)
            return singleSelectColumn.id;
        // Priority 2: SingleLineText
        const singleLineTextColumn = columns.find((col) => col.uidt === UITypes.SingleLineText);
        if (singleLineTextColumn)
            return singleLineTextColumn.id;
        // Fallback: first column
        return ((_a = columns[0]) === null || _a === void 0 ? void 0 : _a.id) || '';
    };
    const getDefaultYAxisColumn = (columns) => {
        var _a;
        if (!columns || (columns === null || columns === void 0 ? void 0 : columns.length) === 0)
            return '';
        // Priority 1: SingleSelect
        const singleSelectColumn = columns.find((col) => col.uidt === UITypes.SingleSelect);
        if (singleSelectColumn)
            return singleSelectColumn.id;
        // Priority 2: SingleLineText
        const singleLineTextColumn = columns.find((col) => col.uidt === UITypes.SingleLineText);
        if (singleLineTextColumn)
            return singleLineTextColumn.id;
        // Fallback: first column
        return (((_a = columns.find((col) => !isSystemColumn(col) &&
            !isButton(col) &&
            !isLookup(col) &&
            !isQrCode(col) &&
            !isBarcode(col))) === null || _a === void 0 ? void 0 : _a.id) || null);
    };
    switch (chartType) {
        case ChartTypes.PIE:
            return Object.assign(Object.assign({}, baseConfig), { chartType: ChartTypes.PIE, data: {
                    category: {
                        column_id: getDefaultCategoryColumn(columns),
                        orderBy: 'default',
                        includeEmptyRecords: false,
                        includeOthers: false,
                        categoryLimit: 10,
                    },
                    value: {
                        type: 'count',
                    },
                }, appearance: {
                    size: 'medium',
                    showCountInLegend: true,
                    showPercentageOnChart: true,
                    legendPosition: 'right',
                    colorSchema: 'default',
                    customColorSchema: [],
                }, permissions: {
                    allowUserToPrint: true,
                    allowUsersToViewRecords: false,
                } });
        case ChartTypes.DONUT:
            return Object.assign(Object.assign({}, baseConfig), { chartType: ChartTypes.DONUT, data: {
                    category: {
                        column_id: getDefaultCategoryColumn(columns),
                        orderBy: 'default',
                        includeEmptyRecords: false,
                        includeOthers: false,
                        categoryLimit: 10,
                    },
                    value: {
                        type: 'count',
                    },
                }, appearance: {
                    size: 'medium',
                    showCountInLegend: true,
                    showPercentageOnChart: true,
                    legendPosition: 'right',
                    colorSchema: 'default',
                    customColorSchema: [],
                }, permissions: {
                    allowUserToPrint: true,
                    allowUsersToViewRecords: false,
                } });
        case ChartTypes.BAR:
        case ChartTypes.LINE:
            return Object.assign(Object.assign({}, baseConfig), { chartType: chartType, appearance: {
                    showCountInLegend: true,
                    legendPosition: 'top',
                    showPercentageOnChart: true,
                }, data: {
                    xAxis: {
                        column_id: getDefaultCategoryColumn(columns),
                        sortBy: 'xAxis',
                        orderBy: 'default',
                        includeEmptyRecords: false,
                        includeOthers: false,
                        categoryLimit: 10,
                    },
                    yAxis: {
                        startAtZero: true,
                        fields: [
                            {
                                column_id: getDefaultYAxisColumn(columns),
                                aggregation: AllAggregations.CountUnique,
                            },
                        ],
                        groupBy: null,
                    },
                }, permissions: {
                    allowUserToPrint: false,
                    allowUsersToViewRecords: false,
                } });
        default:
            return Object.assign(Object.assign({}, baseConfig), { chartType });
    }
};
export { getDefaultConfig, defaultGaugeWidgetRange };
//# sourceMappingURL=data:application/json;base64,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