import { BadRequestV2, MetaError, NotAllowed, NotFound, OptionsNotExistsError, SsoError, TestConnectionError, UnprocessableEntity, } from '../error/nc-base.error';
import { NcErrorType } from '../globals';
import { HigherPlan, PlanFeatureUpgradeMessages, } from '../payment';
import { NcErrorCodexManager } from './nc-error-codex-manager';
export class NcErrorBase {
    constructor() {
        this._errorCodex = new NcErrorCodexManager();
    }
    get errorCodex() {
        return this._errorCodex;
    }
    authenticationRequired(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_AUTHENTICATION_REQUIRED, args);
    }
    apiTokenNotAllowed(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_API_TOKEN_NOT_ALLOWED, args);
    }
    workspaceNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WORKSPACE_NOT_FOUND, Object.assign({ params: id }, args));
    }
    columnAssociatedWithLink(_id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_COLUMN_ASSOCIATED_WITH_LINK, args);
    }
    tableAssociatedWithLink(_id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_TABLE_ASSOCIATED_WITH_LINK, args);
    }
    viewColumnNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_VIEW_COLUMN_NOT_FOUND, Object.assign({ params: id }, args));
    }
    baseNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_BASE_NOT_FOUND, Object.assign({ params: id }, args));
    }
    dashboardNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_DASHBOARD_NOT_FOUND, Object.assign({ params: id }, args));
    }
    workflowNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WORKFLOW_NOT_FOUND, Object.assign({ params: id }, args));
    }
    widgetNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WIDGET_NOT_FOUND, Object.assign({ params: id }, args));
    }
    apiClientNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_API_CLIENT_NOT_FOUND, Object.assign({ params: id }, args));
    }
    sourceNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_SOURCE_NOT_FOUND, Object.assign({ params: id }, args));
    }
    noSourcesFound(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_DATA_SOURCES_NOT_FOUND, args);
    }
    tableNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_TABLE_NOT_FOUND, Object.assign({ params: id }, args));
    }
    userNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_USER_NOT_FOUND, Object.assign({ params: id }, args));
    }
    teamNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_TEAM_NOT_FOUND, Object.assign({ params: id }, args));
    }
    viewNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_VIEW_NOT_FOUND, Object.assign({ params: id }, args));
    }
    hookNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_HOOK_NOT_FOUND, Object.assign({ params: id }, args));
    }
    genericNotFound(resource, id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_GENERIC_NOT_FOUND, Object.assign({ params: [resource, id] }, args));
    }
    requiredFieldMissing(field, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_REQUIRED_FIELD_MISSING, Object.assign({ params: field }, args));
    }
    duplicateRecord(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_DUPLICATE_RECORD, Object.assign({ params: id }, args));
    }
    fieldNotFound(param, args) {
        let message = '';
        if (typeof param === 'string') {
            message = `'${param}'`;
        }
        else {
            const onSection = param.onSection ? ` on ${param.onSection}` : '';
            message = `'${param.field}'${onSection}`;
        }
        throw this.errorCodex.generateError(NcErrorType.ERR_FIELD_NOT_FOUND, Object.assign({ params: message }, args));
    }
    extensionNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_EXTENSION_NOT_FOUND, Object.assign({ params: id }, args));
    }
    invalidOffsetValue(offset, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_OFFSET_VALUE, Object.assign({ params: `${offset}` }, args));
    }
    invalidPageValue(page, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_PAGE_VALUE, Object.assign({ params: `${page}` }, args));
    }
    invalidPrimaryKey(value, pkColumn, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_PK_VALUE, Object.assign({ params: [value, pkColumn] }, args));
    }
    invalidLimitValue(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_LIMIT_VALUE, Object.assign({}, args));
    }
    invalidFilter(filter, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_FILTER, Object.assign({ params: filter }, args));
    }
    invalidValueForField(payload, args) {
        const withReason = typeof payload === 'object' && payload.reason
            ? `, reason: ${payload.reason}`
            : ``;
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_VALUE_FOR_FIELD, Object.assign({ params: typeof payload === 'string'
                ? payload
                : `Invalid value '${payload.value}' for type '${payload.type}' on column '${payload.column}'${withReason}` }, args));
    }
    unsupportedFilterOperation(operation, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_UNSUPPORTED_FILTER_OPERATION, Object.assign({ params: operation }, args));
    }
    valueLengthExceedLimit(payload, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_VALUE_FOR_FIELD, Object.assign({ params: `Value length '${payload.length}' is exceeding allowed limit '${payload.maxLength}' for type '${payload.type}' on column '${payload.column}'` }, args));
    }
    invalidSharedViewPassword(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_SHARED_VIEW_PASSWORD, Object.assign({}, args));
    }
    invalidSharedDashboardPassword(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_SHARED_DASHBOARD_PASSWORD_INVALID, Object.assign({}, args));
    }
    invalidAttachmentJson(payload, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_ATTACHMENT_JSON, Object.assign({ params: payload }, args));
    }
    notImplemented(feature = 'Feature', args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_NOT_IMPLEMENTED, Object.assign({ params: feature }, args));
    }
    internalServerError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INTERNAL_SERVER, Object.assign({ params: message }, args));
    }
    formulaError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_FORMULA, Object.assign({ params: message }, args));
    }
    formulaCircularRefError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_CIRCULAR_REF_IN_FORMULA, Object.assign({ params: message !== null && message !== void 0 ? message : 'Circular reference detected in formula' }, args));
    }
    unauthorized(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_AUTHENTICATION_REQUIRED, Object.assign({ params: message }, args));
    }
    forbidden(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_FORBIDDEN, Object.assign({ params: message }, args));
    }
    sourceDataReadOnly(name) {
        this.forbidden(`Source '${name}' is read-only`);
    }
    sourceMetaReadOnly(name) {
        this.forbidden(`Source '${name}' schema is read-only`);
    }
    integrationNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INTEGRATION_NOT_FOUND, Object.assign({ params: id }, (args || {})));
    }
    cannotCalculateIntermediateOrderError() {
        throw this.errorCodex.generateError(NcErrorType.ERR_CANNOT_CALCULATE_INTERMEDIATE_ORDER, {});
    }
    reorderFailed() {
        throw this.errorCodex.generateError(NcErrorType.ERR_REORDER_FAILED, {});
    }
    integrationLinkedWithMultiple(bases, sources, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INTEGRATION_LINKED_WITH_BASES, Object.assign({ params: bases.map((s) => s.title).join(', '), details: {
                bases: bases.map((b) => {
                    return {
                        id: b.id,
                        title: b.title,
                    };
                }),
                sources: sources.map((s) => {
                    return {
                        id: s.id,
                        base_id: s.base_id,
                        title: s.alias,
                    };
                }),
            } }, (args || {})));
    }
    invalidAttachmentUploadScope(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_ATTACHMENT_UPLOAD_SCOPE, args);
    }
    webhookError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WEBHOOK_ERROR, Object.assign({ params: message }, args));
    }
    invalidWebhookUrl(url, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WEBHOOK_URL_INVALID, Object.assign(Object.assign({}, args), { params: `Invalid URl ${url || ''}` }));
    }
    planLimitExceeded(message, details, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_PLAN_LIMIT_EXCEEDED, Object.assign(Object.assign({ params: message }, args), { details: Object.assign(Object.assign({}, details), ((details === null || details === void 0 ? void 0 : details.plan) ? { higherPlan: HigherPlan[details.plan] } : {})) }));
    }
    duplicateAlias(param, args) {
        var _a, _b;
        const stackTrace = [
            ...Object.keys((_a = param.additionalTrace) !== null && _a !== void 0 ? _a : {}).map((key) => `${key} '${param.additionalTrace[key]}'`),
            `base '${param.base}'`,
        ].join(', ');
        throw this.errorCodex.generateError(NcErrorType.ERR_DUPLICATE_IN_ALIAS, Object.assign({ params: `Duplicate ${param.type} ${(_b = param.label) !== null && _b !== void 0 ? _b : 'alias'} '${param.alias}' at ${stackTrace}` }, args));
    }
    allowedOnlySSOAccess(ncWorkspaceId) {
        throw this.errorCodex.generateError(NcErrorType.ERR_SSO_LOGIN_REQUIRED, {
            params: ncWorkspaceId,
        });
    }
    allowedOnlySSOGeneratedToken(ncWorkspaceId) {
        throw this.errorCodex.generateError(NcErrorType.ERR_SSO_GENERATED_TOKEN_REQUIRED, {
            params: ncWorkspaceId,
        });
    }
    maxPayloadLimitExceeded(limit, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_MAX_PAYLOAD_LIMIT_EXCEEDED, Object.assign({ params: limit.toString() }, args));
    }
    baseUserError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_BASE_COLLABORATION, Object.assign({ params: message }, args));
    }
    orgUserError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_ORG_USER, Object.assign({ params: message }, args));
    }
    tableError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_TABLE_OP_FAILED, Object.assign({ params: message }, args));
    }
    columnError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_COLUMN_OP_FAILED, Object.assign({ params: message }, args));
    }
    baseError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_BASE_OP_FAILED, Object.assign({ params: message }, args));
    }
    maxWorkspaceLimitReached(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_MAX_WORKSPACE_LIMIT_REACHED, Object.assign({}, args));
    }
    pluginTestError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_TEST_PLUGIN_FAILED, Object.assign({ params: message }, args));
    }
    relationFieldNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_RELATION_FIELD_NOT_FOUND, Object.assign({ params: id }, args));
    }
    unSupportedRelation(relation, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_UNSUPPORTED_RELATION, Object.assign({ params: `Relation ${relation} is not supported` }, args));
    }
    // for nc-sql-executor, the error returned is possible to be an Error object
    // thus `error.message` is needed to access it
    externalError(error, args) {
        let message = '';
        if (['string'].includes(typeof error)) {
            message = `${error}`;
        }
        else if (typeof error === 'object') {
            if (error.message) {
                message = error.message;
            }
            else {
                // we log the error if we don't know the schema yet
                console.log(`Unknown error schema from nc-sql-executor: ${JSON.stringify(error)}`);
            }
        }
        if (!message || message === '') {
            // generic error message to prevent programmatic error to propagate to UI
            message = 'Error when executing query in external data source';
        }
        throw this.errorCodex.generateError(NcErrorType.ERR_IN_EXTERNAL_DATA_SOURCE, Object.assign({ params: message }, args));
    }
    externalTimeOut(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_EXTERNAL_DATA_SOURCE_TIMEOUT, Object.assign({ params: message }, args));
    }
    prohibitedSyncTableOperation(param, args) {
        let message = '';
        switch (param.operation) {
            case 'insert':
            case 'update':
            case 'delete':
                message = `Prohibited data insert / update / delete operation on synced table ${param.modelName}`;
                break;
            case 'create_form_view':
                message = `Form view creation is not supported for synced table ${param.modelName}`;
                break;
        }
        throw this.errorCodex.generateError(NcErrorType.ERR_SYNC_TABLE_OPERATION_PROHIBITED, Object.assign({ params: message }, args));
    }
    featureNotSupported(props, args) {
        var _a, _b;
        if (props.isOnPrem) {
            throw this.errorCodex.generateError(NcErrorType.ERR_FEATURE_NOT_SUPPORTED, Object.assign({ params: `Please upgrade your license ${(_a = PlanFeatureUpgradeMessages[props.feature]) !== null && _a !== void 0 ? _a : 'to use this feature.'}` }, args));
        }
        throw this.errorCodex.generateError(NcErrorType.ERR_FEATURE_NOT_SUPPORTED, Object.assign({ params: `Upgrade to a higher plan ${(_b = PlanFeatureUpgradeMessages[props.feature]) !== null && _b !== void 0 ? _b : 'to use this feature.'}` }, args));
    }
    invalidRequestBody(message) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_REQUEST_BODY, {
            params: message,
        });
    }
    unprocessableEntity(message = 'Unprocessable entity') {
        throw new UnprocessableEntity(message);
    }
    testConnectionError(message = 'Unprocessable entity', code) {
        throw new TestConnectionError(message, code);
    }
    notAllowed(message = 'Not allowed') {
        throw new NotAllowed(message);
    }
    emailDomainNotAllowed(domain) {
        throw new SsoError(`Email domain ${domain} is not allowed for this organization`);
    }
    metaError(param) {
        throw new MetaError(param);
    }
    notFound(message = 'Not found') {
        throw new NotFound(message);
    }
    badRequest(message) {
        throw new BadRequestV2(message);
    }
    optionsNotExists(props) {
        throw new OptionsNotExistsError(props);
    }
    outOfSync(message) {
        throw this.errorCodex.generateError(NcErrorType.ERR_OUT_OF_SYNC, {
            params: message,
        });
    }
    filterVerificationFailed(errors) {
        throw this.errorCodex.generateError(NcErrorType.ERR_FILTER_VERIFICATION_FAILED, {
            params: errors.join(', '),
        });
    }
    storageFileCreateError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_STORAGE_FILE_CREATE, Object.assign({ params: message }, args));
    }
    storageFileReadError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_STORAGE_FILE_READ, Object.assign({ params: message }, args));
    }
    storageFileDeleteError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_STORAGE_FILE_DELETE, Object.assign({ params: message }, args));
    }
    storageFileStreamError(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_STORAGE_FILE_STREAM, Object.assign({ params: message }, args));
    }
    subscriptionAlreadyExists(workspaceOrOrgId, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_SUBSCRIPTION_ALREADY_EXISTS, Object.assign({ params: workspaceOrOrgId }, args));
    }
    subscriptionNotFound(workspaceOrOrgId, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_SUBSCRIPTION_NOT_FOUND, Object.assign({ params: workspaceOrOrgId }, args));
    }
    planNotAvailable(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_PLAN_NOT_AVAILABLE, args);
    }
    seatCountMismatch(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_SEAT_COUNT_MISMATCH, Object.assign({ params: 'There was a mismatch in the seat count, please try again' }, args));
    }
    invalidPaymentPayload(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INVALID_PAYMENT_PAYLOAD, Object.assign({ params: 'Invalid payment payload' }, args));
    }
    stripeCustomerNotFound(customerId, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_STRIPE_CUSTOMER_NOT_FOUND, Object.assign({ params: customerId }, args));
    }
    stripeSubscriptionNotFound(subscriptionId, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_STRIPE_SUBSCRIPTION_NOT_FOUND, Object.assign({ params: subscriptionId }, args));
    }
    subscriptionOwnershipMismatch(entity, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_SUBSCRIPTION_OWNERSHIP_MISMATCH, Object.assign({ params: `Subscription does not belong to the ${entity}` }, args));
    }
    internalCustomerNotSupported(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_INTERNAL_CUSTOMER_NOT_SUPPORTED, Object.assign({ params: 'Internal customer not supported' }, args));
    }
    subscriptionCreateFailed(message, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_SUBSCRIPTION_CREATE_FAILED, Object.assign({ params: message }, args));
    }
    stripeWebhookVerificationFailed(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_STRIPE_WEBHOOK_VERIFICATION_FAILED, Object.assign({ params: 'Webhook signature verification failed' }, args));
    }
    planAlreadyExists(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_PLAN_ALREADY_EXISTS, Object.assign({ params: id }, args));
    }
    workflowEmptyNode(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WORKFLOW_EMPTY_NODE, Object.assign({}, args));
    }
    workflowTriggerNodeNotFound(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WORKFLOW_TRIGGER_NODE_NOT_FOUND, Object.assign({}, args));
    }
    workflowNodeExecutionFailed(nodeTitle, error, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WORKFLOW_NODE_EXECUTION_FAILED, Object.assign({ params: `Node "${nodeTitle}" failed: ${error}` }, args));
    }
    workflowWaitingExecutions(count, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WORKFLOW_WAITING_EXECUTIONS, Object.assign({ params: count.toString() }, args));
    }
    workflowMaxIterationsExceeded(args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WORKFLOW_MAX_ITERATIONS_EXCEEDED, Object.assign({}, args));
    }
    workflowNodeNotFound(nodeTitle, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_WORKFLOW_NODE_NOT_FOUND, Object.assign({ params: `Node "${nodeTitle}" not found` }, args));
    }
    scriptNotFound(id, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_SCRIPT_NOT_FOUND, Object.assign({ params: id }, args));
    }
    methodNotAllowed(method, args) {
        throw this.errorCodex.generateError(NcErrorType.ERR_METHOD_NOT_ALLOWED, Object.assign({ params: `${method} method not allowed` }, args));
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmMtZXJyb3ItYmFzZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uL3NyYy9saWIvZXJyb3ItaGFuZGxlci9uYy1lcnJvci1iYXNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUNBLE9BQU8sRUFDTCxZQUFZLEVBQ1osU0FBUyxFQUVULFVBQVUsRUFDVixRQUFRLEVBQ1IscUJBQXFCLEVBQ3JCLFFBQVEsRUFDUixtQkFBbUIsRUFDbkIsbUJBQW1CLEdBQ3BCLE1BQU0sd0JBQXdCLENBQUM7QUFDaEMsT0FBTyxFQUFFLFdBQVcsRUFBZ0MsTUFBTSxZQUFZLENBQUM7QUFDdkUsT0FBTyxFQUNMLFVBQVUsRUFFViwwQkFBMEIsR0FDM0IsTUFBTSxZQUFZLENBQUM7QUFFcEIsT0FBTyxFQUFFLG1CQUFtQixFQUFFLE1BQU0sMEJBQTBCLENBQUM7QUFFL0QsTUFBTSxPQUFPLFdBQVc7SUFDdEI7UUFDRSxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksbUJBQW1CLEVBQUUsQ0FBQztJQUMvQyxDQUFDO0lBRUQsSUFBSSxVQUFVO1FBQ1osT0FBTyxJQUFJLENBQUMsV0FBVyxDQUFDO0lBQzFCLENBQUM7SUFFRCxzQkFBc0IsQ0FBQyxJQUFrQjtRQUN2QyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsMkJBQTJCLEVBQ3ZDLElBQUksQ0FDTCxDQUFDO0lBQ0osQ0FBQztJQUVELGtCQUFrQixDQUFDLElBQWtCO1FBQ25DLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQyx5QkFBeUIsRUFDckMsSUFBSSxDQUNMLENBQUM7SUFDSixDQUFDO0lBRUQsaUJBQWlCLENBQUMsRUFBVSxFQUFFLElBQWtCO1FBQzlDLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLHVCQUF1QixrQkFDckUsTUFBTSxFQUFFLEVBQUUsSUFDUCxJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCx3QkFBd0IsQ0FBQyxHQUFXLEVBQUUsSUFBaUI7UUFDckQsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLCtCQUErQixFQUMzQyxJQUFJLENBQ0wsQ0FBQztJQUNKLENBQUM7SUFFRCx1QkFBdUIsQ0FBQyxHQUFXLEVBQUUsSUFBaUI7UUFDcEQsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLDhCQUE4QixFQUMxQyxJQUFJLENBQ0wsQ0FBQztJQUNKLENBQUM7SUFFRCxrQkFBa0IsQ0FBQyxFQUFVLEVBQUUsSUFBa0I7UUFDL0MsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMseUJBQXlCLGtCQUN2RSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELFlBQVksQ0FBQyxFQUFVLEVBQUUsSUFBa0I7UUFDekMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLGtCQUNoRSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELGlCQUFpQixDQUFDLEVBQVUsRUFBRSxJQUFrQjtRQUM5QyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyx1QkFBdUIsa0JBQ3JFLE1BQU0sRUFBRSxFQUFFLElBQ1AsSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBRUQsZ0JBQWdCLENBQUMsRUFBVSxFQUFFLElBQWtCO1FBQzdDLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLHNCQUFzQixrQkFDcEUsTUFBTSxFQUFFLEVBQUUsSUFDUCxJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCxjQUFjLENBQUMsRUFBVSxFQUFFLElBQWtCO1FBQzNDLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLG9CQUFvQixrQkFDbEUsTUFBTSxFQUFFLEVBQUUsSUFDUCxJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCxpQkFBaUIsQ0FBQyxFQUFVLEVBQUUsSUFBa0I7UUFDOUMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsd0JBQXdCLGtCQUN0RSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELGNBQWMsQ0FBQyxFQUFVLEVBQUUsSUFBa0I7UUFDM0MsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsb0JBQW9CLGtCQUNsRSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUNELGNBQWMsQ0FBQyxJQUFrQjtRQUMvQixNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsMEJBQTBCLEVBQ3RDLElBQUksQ0FDTCxDQUFDO0lBQ0osQ0FBQztJQUVELGFBQWEsQ0FBQyxFQUFVLEVBQUUsSUFBa0I7UUFDMUMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsbUJBQW1CLGtCQUNqRSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELFlBQVksQ0FBQyxFQUFVLEVBQUUsSUFBa0I7UUFDekMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLGtCQUNoRSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELFlBQVksQ0FBQyxFQUFVLEVBQUUsSUFBa0I7UUFDekMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLGtCQUNoRSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELFlBQVksQ0FBQyxFQUFVLEVBQUUsSUFBa0I7UUFDekMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLGtCQUNoRSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELFlBQVksQ0FBQyxFQUFVLEVBQUUsSUFBa0I7UUFDekMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLGtCQUNoRSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELGVBQWUsQ0FBQyxRQUFnQixFQUFFLEVBQVUsRUFBRSxJQUFrQjtRQUM5RCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxxQkFBcUIsa0JBQ25FLE1BQU0sRUFBRSxDQUFDLFFBQVEsRUFBRSxFQUFFLENBQUMsSUFDbkIsSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBRUQsb0JBQW9CLENBQUMsS0FBYSxFQUFFLElBQWtCO1FBQ3BELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQywwQkFBMEIsa0JBRXBDLE1BQU0sRUFBRSxLQUFLLElBQ1YsSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsZUFBZSxDQUFDLEVBQXFCLEVBQUUsSUFBa0I7UUFDdkQsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsb0JBQW9CLGtCQUNsRSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELGFBQWEsQ0FDWCxLQUtLLEVBQ0wsSUFBa0I7UUFFbEIsSUFBSSxPQUFPLEdBQUcsRUFBRSxDQUFDO1FBQ2pCLElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxFQUFFLENBQUM7WUFDOUIsT0FBTyxHQUFHLElBQUksS0FBSyxHQUFHLENBQUM7UUFDekIsQ0FBQzthQUFNLENBQUM7WUFDTixNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxPQUFPLEtBQUssQ0FBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1lBQ2xFLE9BQU8sR0FBRyxJQUFJLEtBQUssQ0FBQyxLQUFLLElBQUksU0FBUyxFQUFFLENBQUM7UUFDM0MsQ0FBQztRQUNELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLG1CQUFtQixrQkFDakUsTUFBTSxFQUFFLE9BQU8sSUFDWixJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCxpQkFBaUIsQ0FBQyxFQUFVLEVBQUUsSUFBa0I7UUFDOUMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsdUJBQXVCLGtCQUNyRSxNQUFNLEVBQUUsRUFBRSxJQUNQLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELGtCQUFrQixDQUFDLE1BQXVCLEVBQUUsSUFBa0I7UUFDNUQsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsd0JBQXdCLGtCQUN0RSxNQUFNLEVBQUUsR0FBRyxNQUFNLEVBQUUsSUFDaEIsSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBQ0QsZ0JBQWdCLENBQUMsSUFBcUIsRUFBRSxJQUFrQjtRQUN4RCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxzQkFBc0Isa0JBQ3BFLE1BQU0sRUFBRSxHQUFHLElBQUksRUFBRSxJQUNkLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELGlCQUFpQixDQUFDLEtBQVUsRUFBRSxRQUFnQixFQUFFLElBQWtCO1FBQ2hFLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLG9CQUFvQixrQkFDbEUsTUFBTSxFQUFFLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxJQUN0QixJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCxpQkFBaUIsQ0FBQyxJQUFrQjtRQUNsQyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyx1QkFBdUIsb0JBQ2xFLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELGFBQWEsQ0FBQyxNQUFjLEVBQUUsSUFBa0I7UUFDOUMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsa0JBQWtCLGtCQUNoRSxNQUFNLEVBQUUsTUFBTSxJQUNYLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELG9CQUFvQixDQUNsQixPQUVxRSxFQUNyRSxJQUFrQjtRQUVsQixNQUFNLFVBQVUsR0FDZCxPQUFPLE9BQU8sS0FBSyxRQUFRLElBQUksT0FBTyxDQUFDLE1BQU07WUFDM0MsQ0FBQyxDQUFDLGFBQWEsT0FBTyxDQUFDLE1BQU0sRUFBRTtZQUMvQixDQUFDLENBQUMsRUFBRSxDQUFDO1FBQ1QsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLDJCQUEyQixrQkFFckMsTUFBTSxFQUNKLE9BQU8sT0FBTyxLQUFLLFFBQVE7Z0JBQ3pCLENBQUMsQ0FBQyxPQUFPO2dCQUNULENBQUMsQ0FBQyxrQkFBa0IsT0FBTyxDQUFDLEtBQUssZUFBZSxPQUFPLENBQUMsSUFBSSxnQkFBZ0IsT0FBTyxDQUFDLE1BQU0sSUFBSSxVQUFVLEVBQUUsSUFDM0csSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsMEJBQTBCLENBQUMsU0FBaUIsRUFBRSxJQUFrQjtRQUM5RCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsZ0NBQWdDLGtCQUUxQyxNQUFNLEVBQUUsU0FBUyxJQUNkLElBQUksRUFFVixDQUFDO0lBQ0osQ0FBQztJQUVELHNCQUFzQixDQUNwQixPQUtDLEVBQ0QsSUFBa0I7UUFFbEIsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLDJCQUEyQixrQkFFckMsTUFBTSxFQUFFLGlCQUFpQixPQUFPLENBQUMsTUFBTSxpQ0FBaUMsT0FBTyxDQUFDLFNBQVMsZUFBZSxPQUFPLENBQUMsSUFBSSxnQkFBZ0IsT0FBTyxDQUFDLE1BQU0sR0FBRyxJQUNsSixJQUFJLEVBRVYsQ0FBQztJQUNKLENBQUM7SUFFRCx5QkFBeUIsQ0FBQyxJQUFrQjtRQUMxQyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsZ0NBQWdDLG9CQUV2QyxJQUFJLEVBRVYsQ0FBQztJQUNKLENBQUM7SUFFRCw4QkFBOEIsQ0FBQyxJQUFrQjtRQUMvQyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMscUNBQXFDLG9CQUU1QyxJQUFJLEVBRVYsQ0FBQztJQUNKLENBQUM7SUFFRCxxQkFBcUIsQ0FBQyxPQUFlLEVBQUUsSUFBa0I7UUFDdkQsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLDJCQUEyQixrQkFFckMsTUFBTSxFQUFFLE9BQU8sSUFDWixJQUFJLEVBRVYsQ0FBQztJQUNKLENBQUM7SUFFRCxjQUFjLENBQUMsVUFBa0IsU0FBUyxFQUFFLElBQWtCO1FBQzVELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLG1CQUFtQixrQkFDakUsTUFBTSxFQUFFLE9BQU8sSUFDWixJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCxtQkFBbUIsQ0FBQyxPQUFlLEVBQUUsSUFBa0I7UUFDckQsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsbUJBQW1CLGtCQUNqRSxNQUFNLEVBQUUsT0FBTyxJQUNaLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELFlBQVksQ0FBQyxPQUFlLEVBQUUsSUFBa0I7UUFDOUMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsV0FBVyxrQkFDekQsTUFBTSxFQUFFLE9BQU8sSUFDWixJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCx1QkFBdUIsQ0FBQyxPQUFnQixFQUFFLElBQWtCO1FBQzFELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQywyQkFBMkIsa0JBRXJDLE1BQU0sRUFBRSxPQUFPLGFBQVAsT0FBTyxjQUFQLE9BQU8sR0FBSSx3Q0FBd0MsSUFDeEQsSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsWUFBWSxDQUFDLE9BQWUsRUFBRSxJQUFrQjtRQUM5QyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsMkJBQTJCLGtCQUVyQyxNQUFNLEVBQUUsT0FBTyxJQUNaLElBQUksRUFFVixDQUFDO0lBQ0osQ0FBQztJQUVELFNBQVMsQ0FBQyxPQUFlLEVBQUUsSUFBa0I7UUFDM0MsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsYUFBYSxrQkFDM0QsTUFBTSxFQUFFLE9BQU8sSUFDWixJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCxrQkFBa0IsQ0FBQyxJQUFZO1FBQzdCLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxJQUFJLGdCQUFnQixDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVELGtCQUFrQixDQUFDLElBQVk7UUFDN0IsSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLElBQUksdUJBQXVCLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBRUQsbUJBQW1CLENBQUMsRUFBVSxFQUFFLElBQWtCO1FBQ2hELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLHlCQUF5QixrQkFDdkUsTUFBTSxFQUFFLEVBQUUsSUFDUCxDQUFDLElBQUksSUFBSSxFQUFFLENBQUMsRUFDZixDQUFDO0lBQ0wsQ0FBQztJQUVELHFDQUFxQztRQUNuQyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsdUNBQXVDLEVBQ25ELEVBQUUsQ0FDSCxDQUFDO0lBQ0osQ0FBQztJQUVELGFBQWE7UUFDWCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMxRSxDQUFDO0lBRUQsNkJBQTZCLENBQzNCLEtBQWlCLEVBQ2pCLE9BQXFCLEVBQ3JCLElBQWtCO1FBRWxCLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQyxpQ0FBaUMsa0JBRTNDLE1BQU0sRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUM1QyxPQUFPLEVBQUU7Z0JBQ1AsS0FBSyxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRTtvQkFDckIsT0FBTzt3QkFDTCxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUU7d0JBQ1IsS0FBSyxFQUFFLENBQUMsQ0FBQyxLQUFLO3FCQUNmLENBQUM7Z0JBQ0osQ0FBQyxDQUFDO2dCQUNGLE9BQU8sRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUU7b0JBQ3pCLE9BQU87d0JBQ0wsRUFBRSxFQUFFLENBQUMsQ0FBQyxFQUFFO3dCQUNSLE9BQU8sRUFBRSxDQUFDLENBQUMsT0FBTzt3QkFDbEIsS0FBSyxFQUFFLENBQUMsQ0FBQyxLQUFLO3FCQUNmLENBQUM7Z0JBQ0osQ0FBQyxDQUFDO2FBQ0gsSUFDRSxDQUFDLElBQUksSUFBSSxFQUFFLENBQUMsRUFFbEIsQ0FBQztJQUNKLENBQUM7SUFFRCw0QkFBNEIsQ0FBQyxJQUFrQjtRQUM3QyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsbUNBQW1DLEVBQy9DLElBQUksQ0FDTCxDQUFDO0lBQ0osQ0FBQztJQUVELFlBQVksQ0FBQyxPQUFnQixFQUFFLElBQWtCO1FBQy9DLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLGlCQUFpQixrQkFDL0QsTUFBTSxFQUFFLE9BQU8sSUFDWixJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCxpQkFBaUIsQ0FBQyxHQUFXLEVBQUUsSUFBa0I7UUFDL0MsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsdUJBQXVCLGtDQUNsRSxJQUFJLEtBQ1AsTUFBTSxFQUFFLGVBQWUsR0FBRyxJQUFJLEVBQUUsRUFBRSxJQUNsQyxDQUFDO0lBQ0wsQ0FBQztJQUVELGlCQUFpQixDQUNmLE9BQWUsRUFDZixPQUF5RCxFQUN6RCxJQUFrQjtRQUVsQixNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyx1QkFBdUIsZ0NBQ3JFLE1BQU0sRUFBRSxPQUFPLElBQ1osSUFBSSxLQUNQLE9BQU8sa0NBQ0YsT0FBTyxHQUNQLENBQUMsQ0FBQSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsSUFBSSxFQUFDLENBQUMsQ0FBQyxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxLQUVwRSxDQUFDO0lBQ0wsQ0FBQztJQUVELGNBQWMsQ0FDWixLQU1DLEVBQ0QsSUFBa0I7O1FBRWxCLE1BQU0sVUFBVSxHQUFHO1lBQ2pCLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFBLEtBQUssQ0FBQyxlQUFlLG1DQUFJLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FDN0MsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLEdBQUcsR0FBRyxLQUFLLEtBQUssQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FDbEQ7WUFDRCxTQUFTLEtBQUssQ0FBQyxJQUFJLEdBQUc7U0FDdkIsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDYixNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxzQkFBc0Isa0JBQ3BFLE1BQU0sRUFBRSxhQUFhLEtBQUssQ0FBQyxJQUFJLElBQUksTUFBQSxLQUFLLENBQUMsS0FBSyxtQ0FBSSxPQUFPLEtBQ3ZELEtBQUssQ0FBQyxLQUNSLFFBQVEsVUFBVSxFQUFFLElBQ2pCLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELG9CQUFvQixDQUFDLGFBQXFCO1FBQ3hDLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLHNCQUFzQixFQUFFO1lBQ3RFLE1BQU0sRUFBRSxhQUFhO1NBQ3RCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFDRCw0QkFBNEIsQ0FBQyxhQUFxQjtRQUNoRCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsZ0NBQWdDLEVBQzVDO1lBQ0UsTUFBTSxFQUFFLGFBQWE7U0FDdEIsQ0FDRixDQUFDO0lBQ0osQ0FBQztJQUNELHVCQUF1QixDQUFDLEtBQWEsRUFBRSxJQUFrQjtRQUN2RCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsOEJBQThCLGtCQUV4QyxNQUFNLEVBQUUsS0FBSyxDQUFDLFFBQVEsRUFBRSxJQUNyQixJQUFJLEVBRVYsQ0FBQztJQUNKLENBQUM7SUFDRCxhQUFhLENBQUMsT0FBZSxFQUFFLElBQWtCO1FBQy9DLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLHNCQUFzQixrQkFDcEUsTUFBTSxFQUFFLE9BQU8sSUFDWixJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCxZQUFZLENBQUMsT0FBZSxFQUFFLElBQWtCO1FBQzlDLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLFlBQVksa0JBQzFELE1BQU0sRUFBRSxPQUFPLElBQ1osSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBRUQsVUFBVSxDQUFDLE9BQWUsRUFBRSxJQUFrQjtRQUM1QyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxtQkFBbUIsa0JBQ2pFLE1BQU0sRUFBRSxPQUFPLElBQ1osSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBRUQsV0FBVyxDQUFDLE9BQWUsRUFBRSxJQUFrQjtRQUM3QyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxvQkFBb0Isa0JBQ2xFLE1BQU0sRUFBRSxPQUFPLElBQ1osSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBRUQsU0FBUyxDQUFDLE9BQWUsRUFBRSxJQUFrQjtRQUMzQyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxrQkFBa0Isa0JBQ2hFLE1BQU0sRUFBRSxPQUFPLElBQ1osSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBRUQsd0JBQXdCLENBQUMsSUFBa0I7UUFDekMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLCtCQUErQixvQkFFdEMsSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsZUFBZSxDQUFDLE9BQWUsRUFBRSxJQUFrQjtRQUNqRCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxzQkFBc0Isa0JBQ3BFLE1BQU0sRUFBRSxPQUFPLElBQ1osSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBRUQscUJBQXFCLENBQUMsRUFBVSxFQUFFLElBQWtCO1FBQ2xELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQyw0QkFBNEIsa0JBRXRDLE1BQU0sRUFBRSxFQUFFLElBQ1AsSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsbUJBQW1CLENBQUMsUUFBZ0IsRUFBRSxJQUFrQjtRQUN0RCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyx3QkFBd0Isa0JBQ3RFLE1BQU0sRUFBRSxZQUFZLFFBQVEsbUJBQW1CLElBQzVDLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELDRFQUE0RTtJQUM1RSw4Q0FBOEM7SUFDOUMsYUFBYSxDQUFDLEtBQXFCLEVBQUUsSUFBa0I7UUFDckQsSUFBSSxPQUFPLEdBQVcsRUFBRSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxRQUFRLENBQUMsT0FBTyxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQ3RDLE9BQU8sR0FBRyxHQUFHLEtBQUssRUFBRSxDQUFDO1FBQ3ZCLENBQUM7YUFBTSxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsRUFBRSxDQUFDO1lBQ3JDLElBQUksS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDO2dCQUNsQixPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztZQUMxQixDQUFDO2lCQUFNLENBQUM7Z0JBQ04sbURBQW1EO2dCQUNuRCxPQUFPLENBQUMsR0FBRyxDQUNULDhDQUE4QyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQ3RFLENBQUM7WUFDSixDQUFDO1FBQ0gsQ0FBQztRQUNELElBQUksQ0FBQyxPQUFPLElBQUksT0FBTyxLQUFLLEVBQUUsRUFBRSxDQUFDO1lBQy9CLHlFQUF5RTtZQUN6RSxPQUFPLEdBQUcsb0RBQW9ELENBQUM7UUFDakUsQ0FBQztRQUNELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQywyQkFBMkIsa0JBRXJDLE1BQU0sRUFBRSxPQUFPLElBQ1osSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsZUFBZSxDQUFDLE9BQWdCLEVBQUUsSUFBa0I7UUFDbEQsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLGdDQUFnQyxrQkFFMUMsTUFBTSxFQUFFLE9BQU8sSUFDWixJQUFJLEVBRVYsQ0FBQztJQUNKLENBQUM7SUFDRCw0QkFBNEIsQ0FDMUIsS0FHQyxFQUNELElBQWtCO1FBRWxCLElBQUksT0FBTyxHQUFHLEVBQUUsQ0FBQztRQUNqQixRQUFRLEtBQUssQ0FBQyxTQUFTLEVBQUUsQ0FBQztZQUN4QixLQUFLLFFBQVEsQ0FBQztZQUNkLEtBQUssUUFBUSxDQUFDO1lBQ2QsS0FBSyxRQUFRO2dCQUNYLE9BQU8sR0FBRyxzRUFBc0UsS0FBSyxDQUFDLFNBQVMsRUFBRSxDQUFDO2dCQUNsRyxNQUFNO1lBQ1IsS0FBSyxrQkFBa0I7Z0JBQ3JCLE9BQU8sR0FBRyx3REFBd0QsS0FBSyxDQUFDLFNBQVMsRUFBRSxDQUFDO2dCQUNwRixNQUFNO1FBQ1YsQ0FBQztRQUNELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQyxtQ0FBbUMsa0JBRTdDLE1BQU0sRUFBRSxPQUFPLElBQ1osSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBQ0QsbUJBQW1CLENBQ2pCLEtBR0MsRUFDRCxJQUFrQjs7UUFFbEIsSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDbkIsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLHlCQUF5QixrQkFFbkMsTUFBTSxFQUFFLCtCQUNOLE1BQUEsMEJBQTBCLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxtQ0FBSSxzQkFDL0MsRUFBRSxJQUNDLElBQUksRUFFVixDQUFDO1FBQ0osQ0FBQztRQUVELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLHlCQUF5QixrQkFDdkUsTUFBTSxFQUFFLDRCQUNOLE1BQUEsMEJBQTBCLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxtQ0FBSSxzQkFDL0MsRUFBRSxJQUNDLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELGtCQUFrQixDQUFDLE9BQWU7UUFDaEMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsd0JBQXdCLEVBQUU7WUFDeEUsTUFBTSxFQUFFLE9BQU87U0FDaEIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELG1CQUFtQixDQUFDLE9BQU8sR0FBRyxzQkFBc0I7UUFDbEQsTUFBTSxJQUFJLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRCxtQkFBbUIsQ0FBQyxPQUFPLEdBQUcsc0JBQXNCLEVBQUUsSUFBYTtRQUNqRSxNQUFNLElBQUksbUJBQW1CLENBQUMsT0FBTyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRCxVQUFVLENBQUMsT0FBTyxHQUFHLGFBQWE7UUFDaEMsTUFBTSxJQUFJLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQscUJBQXFCLENBQUMsTUFBYztRQUNsQyxNQUFNLElBQUksUUFBUSxDQUNoQixnQkFBZ0IsTUFBTSx1Q0FBdUMsQ0FDOUQsQ0FBQztJQUNKLENBQUM7SUFFRCxTQUFTLENBQUMsS0FBdUM7UUFDL0MsTUFBTSxJQUFJLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUM3QixDQUFDO0lBRUQsUUFBUSxDQUFDLE9BQU8sR0FBRyxXQUFXO1FBQzVCLE1BQU0sSUFBSSxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVELFVBQVUsQ0FBQyxPQUFPO1FBQ2hCLE1BQU0sSUFBSSxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVELGdCQUFnQixDQUFDLEtBSWhCO1FBQ0MsTUFBTSxJQUFJLHFCQUFxQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRCxTQUFTLENBQUMsT0FBZTtRQUN2QixNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxlQUFlLEVBQUU7WUFDL0QsTUFBTSxFQUFFLE9BQU87U0FDaEIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELHdCQUF3QixDQUFDLE1BQWdCO1FBQ3ZDLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQyw4QkFBOEIsRUFDMUM7WUFDRSxNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7U0FDMUIsQ0FDRixDQUFDO0lBQ0osQ0FBQztJQUNELHNCQUFzQixDQUFDLE9BQWUsRUFBRSxJQUFrQjtRQUN4RCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyx1QkFBdUIsa0JBQ3JFLE1BQU0sRUFBRSxPQUFPLElBQ1osSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBRUQsb0JBQW9CLENBQUMsT0FBZSxFQUFFLElBQWtCO1FBQ3RELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLHFCQUFxQixrQkFDbkUsTUFBTSxFQUFFLE9BQU8sSUFDWixJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCxzQkFBc0IsQ0FBQyxPQUFlLEVBQUUsSUFBa0I7UUFDeEQsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsdUJBQXVCLGtCQUNyRSxNQUFNLEVBQUUsT0FBTyxJQUNaLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELHNCQUFzQixDQUFDLE9BQWUsRUFBRSxJQUFrQjtRQUN4RCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyx1QkFBdUIsa0JBQ3JFLE1BQU0sRUFBRSxPQUFPLElBQ1osSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBQ0QseUJBQXlCLENBQ3ZCLGdCQUF3QixFQUN4QixJQUFrQjtRQUVsQixNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsK0JBQStCLGtCQUV6QyxNQUFNLEVBQUUsZ0JBQWdCLElBQ3JCLElBQUksRUFFVixDQUFDO0lBQ0osQ0FBQztJQUVELG9CQUFvQixDQUFDLGdCQUF3QixFQUFFLElBQWtCO1FBQy9ELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQywwQkFBMEIsa0JBRXBDLE1BQU0sRUFBRSxnQkFBZ0IsSUFDckIsSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsZ0JBQWdCLENBQUMsSUFBa0I7UUFDakMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLHNCQUFzQixFQUNsQyxJQUFJLENBQ0wsQ0FBQztJQUNKLENBQUM7SUFFRCxpQkFBaUIsQ0FBQyxJQUFrQjtRQUNsQyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyx1QkFBdUIsa0JBQ3JFLE1BQU0sRUFBRSwwREFBMEQsSUFDL0QsSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBRUQscUJBQXFCLENBQUMsSUFBa0I7UUFDdEMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLDJCQUEyQixrQkFFckMsTUFBTSxFQUFFLHlCQUF5QixJQUM5QixJQUFJLEVBRVYsQ0FBQztJQUNKLENBQUM7SUFFRCxzQkFBc0IsQ0FBQyxVQUFrQixFQUFFLElBQWtCO1FBQzNELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQyw2QkFBNkIsa0JBRXZDLE1BQU0sRUFBRSxVQUFVLElBQ2YsSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsMEJBQTBCLENBQ3hCLGNBQXNCLEVBQ3RCLElBQWtCO1FBRWxCLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQyxpQ0FBaUMsa0JBRTNDLE1BQU0sRUFBRSxjQUFjLElBQ25CLElBQUksRUFFVixDQUFDO0lBQ0osQ0FBQztJQUVELDZCQUE2QixDQUMzQixNQUEyQixFQUMzQixJQUFrQjtRQUVsQixNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsbUNBQW1DLGtCQUU3QyxNQUFNLEVBQUUsdUNBQXVDLE1BQU0sRUFBRSxJQUNwRCxJQUFJLEVBRVYsQ0FBQztJQUNKLENBQUM7SUFFRCw0QkFBNEIsQ0FBQyxJQUFrQjtRQUM3QyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsbUNBQW1DLGtCQUU3QyxNQUFNLEVBQUUsaUNBQWlDLElBQ3RDLElBQUksRUFFVixDQUFDO0lBQ0osQ0FBQztJQUVELHdCQUF3QixDQUFDLE9BQWUsRUFBRSxJQUFrQjtRQUMxRCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsOEJBQThCLGtCQUV4QyxNQUFNLEVBQUUsT0FBTyxJQUNaLElBQUksRUFFVixDQUFDO0lBQ0osQ0FBQztJQUVELCtCQUErQixDQUFDLElBQWtCO1FBQ2hELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQyxzQ0FBc0Msa0JBRWhELE1BQU0sRUFBRSx1Q0FBdUMsSUFDNUMsSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBQ0QsaUJBQWlCLENBQUMsRUFBVSxFQUFFLElBQWtCO1FBQzlDLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLHVCQUF1QixrQkFDckUsTUFBTSxFQUFFLEVBQUUsSUFDUCxJQUFJLEVBQ1AsQ0FBQztJQUNMLENBQUM7SUFFRCxpQkFBaUIsQ0FBQyxJQUFrQjtRQUNsQyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyx1QkFBdUIsb0JBQ2xFLElBQUksRUFDUCxDQUFDO0lBQ0wsQ0FBQztJQUVELDJCQUEyQixDQUFDLElBQWtCO1FBQzVDLE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQ2pDLFdBQVcsQ0FBQyxtQ0FBbUMsb0JBRTFDLElBQUksRUFFVixDQUFDO0lBQ0osQ0FBQztJQUVELDJCQUEyQixDQUN6QixTQUFpQixFQUNqQixLQUFhLEVBQ2IsSUFBa0I7UUFFbEIsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLGtDQUFrQyxrQkFFNUMsTUFBTSxFQUFFLFNBQVMsU0FBUyxhQUFhLEtBQUssRUFBRSxJQUMzQyxJQUFJLEVBRVYsQ0FBQztJQUNKLENBQUM7SUFFRCx5QkFBeUIsQ0FBQyxLQUFhLEVBQUUsSUFBa0I7UUFDekQsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLCtCQUErQixrQkFFekMsTUFBTSxFQUFFLEtBQUssQ0FBQyxRQUFRLEVBQUUsSUFDckIsSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsNkJBQTZCLENBQUMsSUFBa0I7UUFDOUMsTUFBTSxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FDakMsV0FBVyxDQUFDLG9DQUFvQyxvQkFFM0MsSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsb0JBQW9CLENBQUMsU0FBaUIsRUFBRSxJQUFrQjtRQUN4RCxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUNqQyxXQUFXLENBQUMsMkJBQTJCLGtCQUVyQyxNQUFNLEVBQUUsU0FBUyxTQUFTLGFBQWEsSUFDcEMsSUFBSSxFQUVWLENBQUM7SUFDSixDQUFDO0lBRUQsY0FBYyxDQUFDLEVBQVUsRUFBRSxJQUFrQjtRQUMzQyxNQUFNLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxvQkFBb0Isa0JBQ2xFLE1BQU0sRUFBRSxFQUFFLElBQ1AsSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0lBRUQsZ0JBQWdCLENBQUMsTUFBYyxFQUFFLElBQWtCO1FBQ2pELE1BQU0sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLHNCQUFzQixrQkFDcEUsTUFBTSxFQUFFLEdBQUcsTUFBTSxxQkFBcUIsSUFDbkMsSUFBSSxFQUNQLENBQUM7SUFDTCxDQUFDO0NBQ0YifQ==