import dayjs, { extend } from 'dayjs';
import customParseFormat from 'dayjs/plugin/customParseFormat.js';
import isBetween from 'dayjs/plugin/isBetween';
import isSameOrAfter from 'dayjs/plugin/isSameOrAfter';
import isSameOrBefore from 'dayjs/plugin/isSameOrBefore';
import relativeTime from 'dayjs/plugin/relativeTime.js';
import utc from 'dayjs/plugin/utc.js';
import timezone from 'dayjs/plugin/timezone.js';
import { isDateMonthFormat } from '../../lib/dateTimeHelper';
import { buildFilterTree } from '../../lib/filterHelpers';
import { parseProp } from '../../lib/helperFunctions';
import UITypes from '../../lib/UITypes';
import { getLookupColumnType } from '../../lib/columnHelper/utils/get-lookup-column-type';
import { getNodejsTimezone } from '../../lib/timezoneUtils';
import { ColumnHelper } from '../../lib/columnHelper/column-helper';
import { CURRENT_USER_TOKEN } from '../../lib/globals';
import { getMetaWithCompositeKey } from '../../lib/helpers/metaHelpers';
extend(utc);
extend(timezone);
extend(relativeTime);
extend(customParseFormat);
extend(isSameOrBefore);
extend(isSameOrAfter);
extend(isBetween);
const ncToString = (value) => {
    var _a;
    return ((_a = value === null || value === void 0 ? void 0 : value.toString) === null || _a === void 0 ? void 0 : _a.call(value)) || '';
};
export function validateRowFilters(params) {
    var _a, _b, _c, _d, _e, _f, _g;
    const { filters: _filters, data = {}, columns = [], client, metas, baseId, } = params;
    if (!_filters.length) {
        return true;
    }
    const filters = buildFilterTree(_filters);
    let isValid = null;
    for (const filter of filters) {
        let res;
        if (filter.is_group && ((_a = filter.children) === null || _a === void 0 ? void 0 : _a.length)) {
            res = validateRowFilters({
                filters: filter.children,
                data: data,
                columns: columns,
                client: client,
                metas: metas,
                baseId: baseId,
            });
        }
        else {
            const column = columns.find((c) => c.id === filter.fk_column_id);
            if (!column) {
                continue;
            }
            const field = column.title;
            let val = data[field];
            if ([
                UITypes.Date,
                UITypes.DateTime,
                UITypes.CreatedTime,
                UITypes.LastModifiedTime,
            ].includes(column.uidt) &&
                !['empty', 'blank', 'notempty', 'notblank'].includes(filter.comparison_op)) {
                const getTimezone = () => {
                    var _a;
                    return getNodejsTimezone(parseProp(filter.meta).timezone, parseProp(column.meta).timezone, (_a = params.options) === null || _a === void 0 ? void 0 : _a.timezone);
                };
                const dateFormat = client === 'mysql2' ? 'YYYY-MM-DD HH:mm:ss' : 'YYYY-MM-DD HH:mm:ssZ';
                let now = dayjs.tz(new Date(), getTimezone());
                const dateFormatFromMeta = (_b = parseProp(column.meta)) === null || _b === void 0 ? void 0 : _b.date_format;
                const dataVal = val;
                let filterVal = filter.value;
                if (dateFormatFromMeta && isDateMonthFormat(dateFormatFromMeta)) {
                    // reset to 1st
                    now = now.date(1);
                    if (val)
                        val = dayjs.tz(val, getTimezone()).date(1);
                }
                if (filterVal)
                    res = dayjs.tz(filterVal, getTimezone()).isSame(dataVal, 'day');
                // handle sub operation
                switch (filter.comparison_sub_op) {
                    case 'today':
                        filterVal = now;
                        break;
                    case 'tomorrow':
                        filterVal = now.add(1, 'day');
                        break;
                    case 'yesterday':
                        filterVal = now.add(-1, 'day');
                        break;
                    case 'oneWeekAgo':
                        filterVal = now.add(-1, 'week');
                        break;
                    case 'oneWeekFromNow':
                        filterVal = now.add(1, 'week');
                        break;
                    case 'oneMonthAgo':
                        filterVal = now.add(-1, 'month');
                        break;
                    case 'oneMonthFromNow':
                        filterVal = now.add(1, 'month');
                        break;
                    case 'daysAgo':
                        if (!filterVal)
                            return null;
                        filterVal = now.add(-filterVal, 'day');
                        break;
                    case 'daysFromNow':
                        if (!filterVal)
                            return null;
                        filterVal = now.add(filterVal, 'day');
                        break;
                    case 'exactDate':
                        if (!filterVal)
                            return null;
                        break;
                    // sub-ops for `isWithin` comparison
                    case 'pastWeek':
                        filterVal = now.add(-1, 'week');
                        break;
                    case 'pastMonth':
                        filterVal = now.add(-1, 'month');
                        break;
                    case 'pastYear':
                        filterVal = now.add(-1, 'year');
                        break;
                    case 'nextWeek':
                        filterVal = now.add(1, 'week');
                        break;
                    case 'nextMonth':
                        filterVal = now.add(1, 'month');
                        break;
                    case 'nextYear':
                        filterVal = now.add(1, 'year');
                        break;
                    case 'pastNumberOfDays':
                        if (!filterVal)
                            return null;
                        filterVal = now.add(-filterVal, 'day');
                        break;
                    case 'nextNumberOfDays':
                        if (!filterVal)
                            return null;
                        filterVal = now.add(filterVal, 'day');
                        break;
                }
                if (dataVal) {
                    switch (filter.comparison_op) {
                        case 'eq':
                        case 'gb_eq':
                            res = dayjs
                                .utc(dataVal)
                                .tz(getTimezone())
                                .isSame(filterVal, 'day');
                            break;
                        case 'neq':
                            res = !dayjs
                                .utc(dataVal)
                                .tz(getTimezone())
                                .isSame(filterVal, 'day');
                            break;
                        case 'gt':
                            res = dayjs
                                .utc(dataVal)
                                .tz(getTimezone())
                                .isAfter(filterVal, 'day');
                            break;
                        case 'lt':
                            res = dayjs
                                .utc(dataVal)
                                .tz(getTimezone())
                                .isBefore(filterVal, 'day');
                            break;
                        case 'lte':
                        case 'le':
                            res = dayjs
                                .utc(dataVal)
                                .tz(getTimezone())
                                .isSameOrBefore(filterVal, 'day');
                            break;
                        case 'gte':
                        case 'ge':
                            res = dayjs
                                .utc(dataVal)
                                .tz(getTimezone())
                                .isSameOrAfter(filterVal, 'day');
                            break;
                        case 'empty':
                        case 'blank':
                            res = dataVal === '' || dataVal === null || dataVal === undefined;
                            break;
                        case 'notempty':
                        case 'notblank':
                            res = !(dataVal === '' ||
                                dataVal === null ||
                                dataVal === undefined);
                            break;
                        case 'isWithin': {
                            let now = dayjs
                                .tz(new Date(), getTimezone())
                                .format(dateFormat)
                                .toString();
                            now = column.uidt === UITypes.Date ? now.substring(0, 10) : now;
                            switch (filter.comparison_sub_op) {
                                case 'pastWeek':
                                case 'pastMonth':
                                case 'pastYear':
                                case 'pastNumberOfDays': {
                                    // the 'today' need to be included, hence we don't use isBetween
                                    const dataValDayjs = dayjs.utc(dataVal).tz(getTimezone());
                                    res =
                                        dataValDayjs.isSameOrAfter(filterVal, 'day') &&
                                            dataValDayjs.isSameOrBefore(now, 'day');
                                    break;
                                }
                                case 'nextWeek':
                                case 'nextMonth':
                                case 'nextYear':
                                case 'nextNumberOfDays': {
                                    // the 'today' need to be included, hence we don't use isBetween
                                    const dataValDayjs = dayjs.utc(dataVal).tz(getTimezone());
                                    res =
                                        dataValDayjs.isSameOrAfter(now, 'day') &&
                                            dataValDayjs.isSameOrBefore(filterVal, 'day');
                                    break;
                                }
                            }
                        }
                    }
                }
            }
            else {
                switch (typeof filter.value) {
                    case 'boolean':
                        val = !!data[field];
                        break;
                    case 'number':
                        val = +data[field];
                        break;
                }
                if ([UITypes.User, UITypes.CreatedBy, UITypes.LastModifiedBy].includes(column.uidt) ||
                    (column.uidt === UITypes.Lookup &&
                        [UITypes.User, UITypes.CreatedBy, UITypes.LastModifiedBy].includes(getLookupColumnType({
                            col: column,
                            meta: { columns, base_id: baseId },
                            metas: metas,
                            baseId: baseId,
                        })))) {
                    const userIds = Array.isArray(data[field])
                        ? data[field].map((user) => user.id)
                        : ((_c = data[field]) === null || _c === void 0 ? void 0 : _c.id)
                            ? [data[field].id]
                            : [];
                    const filterValues = (ncToString(filter.value).split(',') || []).map((v) => {
                        var _a, _b, _c;
                        let result = v.trim();
                        if (result === CURRENT_USER_TOKEN) {
                            result = (_c = (_b = (_a = params.options) === null || _a === void 0 ? void 0 : _a.currentUser) === null || _b === void 0 ? void 0 : _b.id) !== null && _c !== void 0 ? _c : result;
                        }
                        return result;
                    });
                    switch (filter.comparison_op) {
                        case 'anyof':
                            res = userIds.some((id) => filterValues.includes(id));
                            break;
                        case 'nanyof':
                            res = !userIds.some((id) => filterValues.includes(id));
                            break;
                        case 'allof':
                            res = filterValues.every((id) => userIds.includes(id));
                            break;
                        case 'nallof':
                            res = !filterValues.every((id) => userIds.includes(id));
                            break;
                        case 'empty':
                        case 'blank':
                            res = userIds.length === 0;
                            break;
                        case 'notempty':
                        case 'notblank':
                            res = userIds.length > 0;
                            break;
                        default:
                            res = false; // Unsupported operation for User fields
                    }
                }
                else if (column.uidt === UITypes.LinkToAnotherRecord) {
                    let linkData = data[field];
                    linkData = Array.isArray(linkData) ? linkData : [linkData];
                    const colOptions = column.colOptions;
                    const relatedModelId = colOptions === null || colOptions === void 0 ? void 0 : colOptions.fk_related_model_id;
                    const relatedMeta = getMetaWithCompositeKey(metas, baseId, relatedModelId);
                    if (!(relatedMeta === null || relatedMeta === void 0 ? void 0 : relatedMeta.columns)) {
                        res = false;
                    }
                    else {
                        // Find the child column in the related table
                        const childColumn = relatedMeta.columns.find((col) => col.pv);
                        if (!childColumn) {
                            res = false;
                        }
                        else {
                            const childFieldName = childColumn.title;
                            const childValues = linkData
                                .map((item) => {
                                return ncToString(item === null || item === void 0 ? void 0 : item[childFieldName]);
                            })
                                .filter((val) => val !== '');
                            switch (filter.comparison_op) {
                                case 'eq':
                                    res = childValues.includes(ncToString(filter.value));
                                    break;
                                case 'neq':
                                    res = !childValues.includes(ncToString(filter.value));
                                    break;
                                case 'like':
                                    res = childValues.some((val) => val
                                        .toLowerCase()
                                        .includes(ncToString(filter.value).toLowerCase()));
                                    break;
                                case 'nlike':
                                    res = !childValues.some((val) => val
                                        .toLowerCase()
                                        .includes(ncToString(filter.value).toLowerCase()));
                                    break;
                                case 'anyof': {
                                    const filterValues = ncToString(filter.value)
                                        .split(',')
                                        .map((v) => v.trim()) || [];
                                    res = childValues.some((val) => filterValues.includes(val));
                                    break;
                                }
                                case 'nanyof': {
                                    const filterValues2 = ncToString(filter.value)
                                        .split(',')
                                        .map((v) => v.trim()) || [];
                                    res = !childValues.some((val) => filterValues2.includes(val));
                                    break;
                                }
                                case 'allof': {
                                    const filterValues3 = ncToString(filter.value)
                                        .split(',')
                                        .map((v) => v.trim()) || [];
                                    res = filterValues3.every((filterVal) => childValues.includes(filterVal));
                                    break;
                                }
                                case 'nallof': {
                                    const filterValues4 = ncToString(filter.value)
                                        .split(',')
                                        .map((v) => v.trim()) || [];
                                    res = !filterValues4.every((filterVal) => childValues.includes(filterVal));
                                    break;
                                }
                                case 'empty':
                                case 'blank':
                                    res = linkData.length === 0;
                                    break;
                                case 'notempty':
                                case 'notblank':
                                    res = linkData.length > 0;
                                    break;
                                default:
                                    res = false;
                            }
                        }
                    }
                }
                else if ([UITypes.JSON, UITypes.Time].includes(column.uidt) &&
                    ['eq'].includes(filter.comparison_op)) {
                    res = ColumnHelper.getColumn(column.uidt).equalityComparison(val, filter.value, {
                        col: column,
                    });
                }
                else {
                    switch (filter.comparison_op) {
                        case 'eq':
                        case 'gb_eq':
                            res = val == filter.value;
                            break;
                        case 'neq':
                            res = val != filter.value;
                            break;
                        case 'like':
                            res =
                                ncToString(data[field])
                                    .toLowerCase()
                                    .indexOf(ncToString(filter.value).toLowerCase()) > -1;
                            break;
                        case 'nlike':
                            res =
                                ncToString(data[field])
                                    .toLowerCase()
                                    .indexOf(ncToString(filter.value).toLowerCase()) === -1;
                            break;
                        case 'empty':
                        case 'blank':
                            res =
                                data[field] === '' ||
                                    data[field] === null ||
                                    data[field] === undefined;
                            break;
                        case 'notempty':
                        case 'notblank':
                            res = !(data[field] === '' ||
                                data[field] === null ||
                                data[field] === undefined);
                            break;
                        case 'checked':
                            res = !!data[field];
                            break;
                        case 'notchecked':
                            res = !data[field];
                            break;
                        case 'null':
                            res = data[field] === null;
                            break;
                        case 'notnull':
                            res = data[field] !== null;
                            break;
                        case 'allof':
                            res = ((_d = ncToString(filter.value)
                                .split(',')
                                .map((item) => item.trim())) !== null && _d !== void 0 ? _d : []).every((item) => { var _a; return ((_a = ncToString(data[field]).split(',')) !== null && _a !== void 0 ? _a : []).includes(item); });
                            break;
                        case 'anyof':
                            res = ((_e = ncToString(filter.value)
                                .split(',')
                                .map((item) => item.trim())) !== null && _e !== void 0 ? _e : []).some((item) => { var _a; return ((_a = ncToString(data[field]).split(',')) !== null && _a !== void 0 ? _a : []).includes(item); });
                            break;
                        case 'nallof':
                            res = !((_f = ncToString(filter.value)
                                .split(',')
                                .map((item) => item.trim())) !== null && _f !== void 0 ? _f : []).every((item) => { var _a; return ((_a = ncToString(data[field]).split(',')) !== null && _a !== void 0 ? _a : []).includes(item); });
                            break;
                        case 'nanyof':
                            res = !((_g = ncToString(filter.value)
                                .split(',')
                                .map((item) => item.trim())) !== null && _g !== void 0 ? _g : []).some((item) => { var _a; return ((_a = ncToString(data[field]).split(',')) !== null && _a !== void 0 ? _a : []).includes(item); });
                            break;
                        case 'lt':
                            res = +data[field] < +filter.value;
                            break;
                        case 'lte':
                        case 'le':
                            res = +data[field] <= +filter.value;
                            break;
                        case 'gt':
                            res = +data[field] > +filter.value;
                            break;
                        case 'gte':
                        case 'ge':
                            res = +data[field] >= +filter.value;
                            break;
                    }
                }
            }
        }
        switch (filter.logical_op) {
            case 'or':
                isValid = isValid || !!res;
                break;
            case 'not':
                isValid = isValid && !res;
                break;
            case 'and':
            default:
                isValid = (isValid !== null && isValid !== void 0 ? isValid : true) && res;
                break;
        }
    }
    return isValid;
}
//# sourceMappingURL=data:application/json;base64,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