import { validateRowFilters } from './validate-row-filters';
import UITypes from '../../lib/UITypes';
import dayjs from 'dayjs';
import utc from 'dayjs/plugin/utc.js';
import timezone from 'dayjs/plugin/timezone.js';
import customParseFormat from 'dayjs/plugin/customParseFormat.js';
import isBetween from 'dayjs/plugin/isBetween';
import isSameOrAfter from 'dayjs/plugin/isSameOrAfter';
import isSameOrBefore from 'dayjs/plugin/isSameOrBefore';
import { CURRENT_USER_TOKEN } from '../../lib';
dayjs.extend(utc);
dayjs.extend(timezone);
dayjs.extend(customParseFormat);
dayjs.extend(isSameOrBefore);
dayjs.extend(isSameOrAfter);
dayjs.extend(isBetween);
const mockColumns = [
    { id: '1', title: 'Name', uidt: UITypes.SingleLineText },
    { id: '2', title: 'Age', uidt: UITypes.Number },
    { id: '3', title: 'IsActive', uidt: UITypes.Checkbox },
    {
        id: '4',
        title: 'CreatedAt',
        uidt: UITypes.DateTime,
    },
    { id: '5', title: 'CreatedBy', uidt: UITypes.User },
    {
        id: '6',
        title: 'RelatedRecords',
        uidt: UITypes.LinkToAnotherRecord,
        colOptions: {
            fk_related_model_id: 'relatedModel',
        },
    },
    { id: '7', title: 'JsonData', uidt: UITypes.JSON },
    { id: '8', title: 'TimeData', uidt: UITypes.Time },
];
const mockMetas = {
    relatedModel: {
        columns: [
            { id: 'r1', title: 'Primary', pv: true, uidt: UITypes.SingleLineText },
        ],
    },
};
const mockClient = 'pg'; // or 'mysql2'
describe('validateRowFilters', () => {
    it('should return true if no filters are provided', () => {
        const filters = [];
        const data = { Name: 'Test' };
        const result = validateRowFilters({
            filters,
            data,
            columns: mockColumns,
            client: mockClient,
            metas: mockMetas,
        });
        expect(result).toBe(true);
    });
    // Test cases for basic comparisons (eq, neq, like, nlike)
    // FIXME: not reviewed
    describe.skip('Basic comparisons', () => {
        it('should correctly evaluate "eq" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'eq', value: 'Alice' },
            ];
            const data = { Name: 'Alice' };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: 'Bob' };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "neq" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'neq', value: 'Alice' },
            ];
            const data = { Name: 'Bob' };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: 'Alice' };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "like" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'like', value: 'ali' },
            ];
            const data = { Name: 'Alice' };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: 'Bob' };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "nlike" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'nlike', value: 'ali' },
            ];
            const data = { Name: 'Bob' };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: 'Alice' };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "eq" for number', () => {
            const filters = [
                { fk_column_id: '2', comparison_op: 'eq', value: 30 },
            ];
            const data = { Age: 30 };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Age: 25 };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "gt" for number', () => {
            const filters = [
                { fk_column_id: '2', comparison_op: 'gt', value: 25 },
            ];
            const data = { Age: 30 };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Age: 20 };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "lte" for number', () => {
            const filters = [
                { fk_column_id: '2', comparison_op: 'lte', value: 30 },
            ];
            const data = { Age: 30 };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Age: 35 };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "checked" for checkbox', () => {
            const filters = [
                { fk_column_id: '3', comparison_op: 'checked' },
            ];
            const data = { IsActive: true };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { IsActive: false };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "notchecked" for checkbox', () => {
            const filters = [
                { fk_column_id: '3', comparison_op: 'notchecked' },
            ];
            const data = { IsActive: false };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { IsActive: true };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
    });
    // Test cases for empty/blank/null checks
    // FIXME: not reviewed
    describe.skip('Empty/Blank/Null checks', () => {
        it('should correctly evaluate "empty" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'empty' },
            ];
            const data = { Name: '' };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: null };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data3 = { Name: undefined };
            expect(validateRowFilters({
                filters,
                data: data3,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data4 = { Name: 'Test' };
            expect(validateRowFilters({
                filters,
                data: data4,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "notempty" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'notempty' },
            ];
            const data = { Name: 'Test' };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: '' };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "null" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'null' },
            ];
            const data = { Name: null };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: 'Test' };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "notnull" for string', () => {
            const filters = [
                { fk_column_id: '1', comparison_op: 'notnull' },
            ];
            const data = { Name: 'Test' };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { Name: null };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
    });
    // Test cases for Date/DateTime filters
    describe('Date/DateTime filters', () => {
        const today = dayjs()
            .tz('Etc/UTC')
            .startOf('day')
            .format('YYYY-MM-DD HH:mm:ssZ');
        const todayEnd = dayjs()
            .tz('Etc/UTC')
            .endOf('day')
            .format('YYYY-MM-DD HH:mm:ssZ');
        const yesterday = dayjs()
            .tz('Etc/UTC')
            .subtract(1, 'day')
            .format('YYYY-MM-DD HH:mm:ssZ');
        const tzAsiaKolkata = 'Asia/Kolkata';
        const todayKolkata = dayjs()
            .tz(tzAsiaKolkata)
            .startOf('day')
            .format('YYYY-MM-DD HH:mm:ssZ');
        const todayKolkataEnd = dayjs()
            .tz(tzAsiaKolkata)
            .endOf('day')
            .format('YYYY-MM-DD HH:mm:ssZ');
        it('should correctly evaluate "eq" with "today" sub-op', () => {
            const filters = [
                { fk_column_id: '4', comparison_op: 'eq', comparison_sub_op: 'today' },
            ];
            const data = { CreatedAt: today };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            expect(validateRowFilters({
                filters,
                data: { CreatedAt: todayEnd },
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            const data2 = { CreatedAt: yesterday };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
        it('should correctly evaluate "eq" with "today" sub-op for other timezone', () => {
            const filters = [
                { fk_column_id: '4', comparison_op: 'eq', comparison_sub_op: 'today' },
            ];
            const data = { CreatedAt: todayKolkata };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: tzAsiaKolkata },
            })).toBe(true);
            expect(validateRowFilters({
                filters,
                data: { CreatedAt: todayKolkataEnd },
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: tzAsiaKolkata },
            })).toBe(true);
            const data2 = { CreatedAt: yesterday };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
        it('should correctly evaluate "eq" with "yesterday" sub-op', () => {
            const filters = [
                {
                    fk_column_id: '4',
                    comparison_op: 'eq',
                    comparison_sub_op: 'yesterday',
                },
            ];
            const data = { CreatedAt: yesterday };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            const data2 = { CreatedAt: today };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
        it('should correctly evaluate "gt" with "daysAgo" sub-op', () => {
            const filters = [
                {
                    fk_column_id: '4',
                    comparison_op: 'gt',
                    comparison_sub_op: 'daysAgo',
                    value: 2,
                },
            ];
            const data = { CreatedAt: yesterday }; // 1 day ago
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            const data2 = {
                CreatedAt: dayjs()
                    .tz('Etc/UTC')
                    .subtract(3, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            }; // 3 days ago
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
        it('should correctly evaluate "isWithin" with "pastWeek" sub-op', () => {
            const filters = [
                {
                    fk_column_id: '4',
                    comparison_op: 'isWithin',
                    comparison_sub_op: 'pastWeek',
                },
            ];
            const data = {
                CreatedAt: dayjs()
                    .tz('Etc/UTC')
                    .subtract(3, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            }; // 3 days ago
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            const data2 = {
                CreatedAt: dayjs()
                    .tz('Etc/UTC')
                    .subtract(8, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            }; // 8 days ago
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
        it('should correctly evaluate "isWithin" with "nextNumberOfDays" sub-op', () => {
            const filters = [
                {
                    fk_column_id: '4',
                    comparison_op: 'isWithin',
                    comparison_sub_op: 'nextNumberOfDays',
                    value: 5,
                },
            ];
            const data = {
                CreatedAt: dayjs()
                    .tz('Etc/UTC')
                    .add(3, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            }; // 3 days from now
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(true);
            const data2 = {
                CreatedAt: dayjs()
                    .tz('Etc/UTC')
                    .add(6, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            }; // 6 days from now
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(false);
        });
    });
    // Test cases for User/CreatedBy/LastModifiedBy filters
    // FIXME: not reviewed
    describe.skip('User/CreatedBy/LastModifiedBy filters', () => {
        const currentUser = { id: 'user1', email: 'user1@example.com' };
        it('should correctly evaluate "anyof" for single user', () => {
            const filters = [
                { fk_column_id: '5', comparison_op: 'anyof', value: 'user1' },
            ];
            const data = { CreatedBy: { id: 'user1', email: 'user1@example.com' } };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data2 = { CreatedBy: { id: 'user2', email: 'user2@example.com' } };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(false);
        });
        it('should correctly evaluate "anyof" for multiple users', () => {
            const filters = [
                { fk_column_id: '5', comparison_op: 'anyof', value: 'user1, user3' },
            ];
            const data = { CreatedBy: [{ id: 'user1' }, { id: 'user2' }] };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data2 = { CreatedBy: [{ id: 'user4' }] };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(false);
        });
        it('should correctly evaluate "anyof" with CURRENT_USER_TOKEN', () => {
            const filters = [
                {
                    fk_column_id: '5',
                    comparison_op: 'anyof',
                    value: CURRENT_USER_TOKEN,
                },
            ];
            const data = { CreatedBy: { id: 'user1', email: 'user1@example.com' } };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data2 = { CreatedBy: { id: 'user2', email: 'user2@example.com' } };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(false);
        });
        it('should correctly evaluate "allof" for multiple users', () => {
            const filters = [
                { fk_column_id: '5', comparison_op: 'allof', value: 'user1, user2' },
            ];
            const data = {
                CreatedBy: [{ id: 'user1' }, { id: 'user2' }, { id: 'user3' }],
            };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data2 = { CreatedBy: [{ id: 'user1' }, { id: 'user3' }] };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(false);
        });
        it('should correctly evaluate "empty" for user field', () => {
            const filters = [
                { fk_column_id: '5', comparison_op: 'empty' },
            ];
            const data = { CreatedBy: [] };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data2 = { CreatedBy: null };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(true);
            const data3 = { CreatedBy: { id: 'user1' } };
            expect(validateRowFilters({
                filters,
                data: data3,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { currentUser },
            })).toBe(false);
        });
    });
    // Test cases for LinkToAnotherRecord filters
    // FIXME: not reviewed
    describe.skip('LinkToAnotherRecord filters', () => {
        it('should correctly evaluate "eq" for linked record primary value', () => {
            const filters = [
                { fk_column_id: '6', comparison_op: 'eq', value: 'RecordA' },
            ];
            const data = { RelatedRecords: [{ Primary: 'RecordA' }] };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { RelatedRecords: [{ Primary: 'RecordB' }] };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "like" for linked record primary value', () => {
            const filters = [
                { fk_column_id: '6', comparison_op: 'like', value: 'record' },
            ];
            const data = { RelatedRecords: [{ Primary: 'RecordA' }] };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { RelatedRecords: [{ Primary: 'SomethingElse' }] };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "anyof" for linked record primary value', () => {
            const filters = [
                {
                    fk_column_id: '6',
                    comparison_op: 'anyof',
                    value: 'RecordA, RecordC',
                },
            ];
            const data = {
                RelatedRecords: [{ Primary: 'RecordA' }, { Primary: 'RecordB' }],
            };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { RelatedRecords: [{ Primary: 'RecordB' }] };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
        it('should correctly evaluate "empty" for linked record', () => {
            const filters = [
                { fk_column_id: '6', comparison_op: 'empty' },
            ];
            const data = { RelatedRecords: [] };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(true);
            const data2 = { RelatedRecords: [{ Primary: 'RecordA' }] };
            expect(validateRowFilters({
                filters,
                data: data2,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(false);
        });
    });
    // Edge cases
    // FIXME: not reviewed
    describe.skip('Edge cases', () => {
        it('should return null if a date sub-op value is missing', () => {
            const filters = [
                {
                    fk_column_id: '4',
                    comparison_op: 'gt',
                    comparison_sub_op: 'daysAgo',
                    value: null,
                },
            ];
            const data = {
                CreatedAt: dayjs()
                    .tz('Etc/UTC')
                    .subtract(1, 'day')
                    .format('YYYY-MM-DD HH:mm:ss'),
            };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
                options: { timezone: 'Etc/UTC' },
            })).toBe(null);
        });
        it('should handle missing column gracefully', () => {
            const filters = [
                { fk_column_id: '99', comparison_op: 'eq', value: 'test' },
            ]; // Non-existent column
            const data = { Name: 'Test' };
            expect(validateRowFilters({
                filters,
                data,
                columns: mockColumns,
                client: mockClient,
                metas: mockMetas,
            })).toBe(null);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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