import { QueryFilterParser } from '../lib/parser/queryFilter/query-filter-parser';
import UITypes from './UITypes';
import { InvalidFilterError } from './error/invalid-filter.error';
import { COMPARISON_SUB_OPS, IS_WITHIN_COMPARISON_SUB_OPS, } from './filterHelpers';
import { arrFlatMap } from './arrayHelpers';
import { parseLexingError, parseParsingError, } from './parser/queryFilter/error-message-parser';
import { getNodejsTimezone } from './timezoneUtils';
import { parseProp } from './helperFunctions';
export { COMPARISON_OPS, COMPARISON_SUB_OPS, GROUPBY_COMPARISON_OPS, IS_WITHIN_COMPARISON_SUB_OPS, } from '../lib/parser/queryFilter/query-filter-lexer';
export function extractFilterFromXwhere(context, { str, aliasColObjMap, throwErrorIfInvalid = false, errors = [], }) {
    if (!str) {
        return { filters: [] };
    }
    for (const columnName of Object.keys(aliasColObjMap)) {
        const column = aliasColObjMap[columnName];
        aliasColObjMap[column.id] = column;
    }
    return innerExtractFilterFromXwhere(context, {
        str,
        aliasColObjMap,
        throwErrorIfInvalid,
        errors,
    });
}
function innerExtractFilterFromXwhere(context, { str, aliasColObjMap, throwErrorIfInvalid = false, errors = [], }) {
    if (!str) {
        return { filters: [] };
    } // if array treat it as `and` group
    else if (Array.isArray(str)) {
        // calling recursively for nested query
        const nestedFilters = [].concat(...str.map((s) => extractFilterFromXwhere(context, {
            str: s,
            aliasColObjMap,
            throwErrorIfInvalid,
        })));
        // extract and flatten filters
        const filters = nestedFilters.reduce((acc, { filters }) => {
            if (!filters)
                return acc;
            return [...acc, ...filters];
        }, []);
        // extract and flatten errors
        const collectedErrors = nestedFilters.reduce((acc, { errors }) => {
            if (!errors)
                return acc;
            return [...acc, ...errors];
        }, []);
        // If errors exist, return them
        if (collectedErrors.length > 0) {
            return { errors: collectedErrors };
        }
        // If there's only one filter, return it directly
        if (filters.length === 1) {
            return { filters: nestedFilters };
        }
        // Otherwise, wrap it in an AND group
        return {
            filters: [
                {
                    is_group: true,
                    logical_op: 'and',
                    children: filters,
                },
            ],
        };
    }
    else if (typeof str !== 'string') {
        const message = 'Invalid filter format. Expected string or array of strings';
        if (throwErrorIfInvalid) {
            throw new InvalidFilterError({ message });
        }
        else {
            errors.push({ message });
            return { errors };
        }
    }
    const parseResult = QueryFilterParser.parse(str);
    if (parseResult.lexErrors.length > 0 || parseResult.parseErrors.length > 0) {
        if (throwErrorIfInvalid)
            throw new InvalidFilterError({
                lexingError: parseResult.lexErrors,
                parsingError: parseResult.parseErrors,
            });
        else {
            if (parseResult.lexErrors.length > 0) {
                errors.push({
                    message: parseResult.lexErrors
                        .map((k) => parseLexingError(k))
                        .join(', '),
                });
            }
            else if (parseResult.parseErrors.length > 0) {
                errors.push({
                    message: parseResult.parseErrors
                        .map((k) => parseParsingError(k))
                        .join(', '),
                });
            }
            return { errors };
        }
    }
    const filterSubType = parseResult.parsedCst;
    const { filter, errors: parseErrors } = mapFilterGroupSubType(context, {
        filter: filterSubType,
        aliasColObjMap,
        throwErrorIfInvalid,
        errors,
    });
    if ((parseErrors === null || parseErrors === void 0 ? void 0 : parseErrors.length) > 0) {
        return { errors: parseErrors };
    }
    return { filters: [filter] };
}
function mapFilterGroupSubType(context, { filter, aliasColObjMap, throwErrorIfInvalid = false, errors = [], }) {
    const children = filter.children
        .map((k) => k.is_group
        ? mapFilterGroupSubType(context, {
            filter: k,
            aliasColObjMap,
            throwErrorIfInvalid,
            errors,
        })
        : mapFilterClauseSubType(context, {
            filter: k,
            aliasColObjMap,
            throwErrorIfInvalid,
            errors,
        }))
        .filter((k) => k);
    if (children.length === 1) {
        return children[0];
    }
    else {
        return {
            filter: {
                is_group: filter.is_group,
                logical_op: filter.logical_op,
                children: children.map((k) => k.filter),
            },
            errors: arrFlatMap(children.map((k) => k.errors || [])).filter((k) => k),
        };
    }
}
function mapFilterClauseSubType(context, { filter, aliasColObjMap, throwErrorIfInvalid = false, errors = [], }) {
    const aliasCol = aliasColObjMap[filter.field];
    if (!aliasCol) {
        if (throwErrorIfInvalid) {
            throw new InvalidFilterError({
                message: `Invalid filter field '${filter.field}' not found`,
            });
        }
        else {
            errors.push({
                message: `Invalid filter field '${filter.field}' not found`,
            });
            return { errors };
        }
    }
    const result = {
        fk_column_id: aliasCol.id,
        is_group: false,
        logical_op: filter.logical_op,
        comparison_op: filter.comparison_op,
        comparison_sub_op: undefined,
        value: filter.value,
    };
    return handleDataTypes(context, {
        filterType: result,
        column: aliasCol,
        throwErrorIfInvalid,
        errors,
    });
}
function handleDataTypes(context, { filterType, column, throwErrorIfInvalid = false, errors = [], }) {
    var _a, _b;
    if (filterType.value === null) {
        return { filter: filterType };
    }
    if ([
        UITypes.Date,
        UITypes.DateTime,
        UITypes.CreatedTime,
        UITypes.LastModifiedTime,
    ].includes(column.uidt) &&
        filterType.value) {
        const [subOp, ...value] = Array.isArray(filterType.value)
            ? filterType.value
            : filterType.value.split(',').map((k) => k.trim());
        filterType.comparison_sub_op = subOp;
        filterType.value = value.join('');
        if (filterType.comparison_sub_op) {
            if (!COMPARISON_SUB_OPS.includes(filterType.comparison_sub_op)) {
                if (throwErrorIfInvalid)
                    throw new InvalidFilterError({
                        message: `Invalid filter '${filterType.comparison_sub_op}' is not supported`,
                    });
                else {
                    errors.push({
                        message: `Invalid filter '${filterType.comparison_sub_op}' is not supported`,
                    });
                    return { errors };
                }
            }
        }
        if ((filterType.comparison_op === 'isWithin' &&
            !IS_WITHIN_COMPARISON_SUB_OPS.includes(filterType.comparison_sub_op)) ||
            (filterType.comparison_op !== 'isWithin' &&
                IS_WITHIN_COMPARISON_SUB_OPS.includes(filterType.comparison_sub_op))) {
            if (throwErrorIfInvalid)
                throw new InvalidFilterError({
                    message: `Invalid filter '${filterType.comparison_sub_op}' is not supported for '${filterType.comparison_op}'`,
                });
            else {
                errors.push({
                    message: `Invalid filter '${filterType.comparison_sub_op}' is not supported for '${filterType.comparison_op}'`,
                });
                return { errors };
            }
        }
        if (filterType.value === '') {
            filterType.value = undefined;
        }
        filterType.meta = (_a = filterType.meta) !== null && _a !== void 0 ? _a : {};
        // use || to also include falsy values
        filterType.meta.timezone = getNodejsTimezone((_b = parseProp(column.meta)) === null || _b === void 0 ? void 0 : _b.timezone, context.timezone);
    }
    return { filter: filterType };
}
//# sourceMappingURL=data:application/json;base64,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