import { validateDateWithUnknownFormat } from '../dateTimeHelper';
import { FormulaDataTypes, FormulaErrorType, JSEPNode } from './enums';
import { FormulaError } from './error';
import { isSystemColumn, isVirtualCol, UITypes } from '../../lib';
export const API_DOC_PREFIX = 'https://nocodb.com/docs/product-docs/fields';
const customValidationArray = (_argTypes, parsedTree) => {
    var _a, _b, _c, _d, _e;
    if (
    // check if first parameter exists
    !((_a = parsedTree.arguments) === null || _a === void 0 ? void 0 : _a[0]) ||
        (!((_c = (_b = parsedTree.arguments) === null || _b === void 0 ? void 0 : _b[0]) === null || _c === void 0 ? void 0 : _c.isDataArray) &&
            // data type array is backward compatibility
            ((_e = (_d = parsedTree.arguments) === null || _d === void 0 ? void 0 : _d[0]) === null || _e === void 0 ? void 0 : _e.dataType) !== FormulaDataTypes.ARRAY)) {
        throw new FormulaError(FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.firstParamArray' }, 'First parameter need to be array. Either it is a link with Has Many / Many to Many relation, or either a Formula or Lookup that reference the link');
    }
};
export const formulas = {
    AVG: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#avg`,
        validation: {
            args: {
                min: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Average of input parameters',
        syntax: 'AVG(value1, [value2, ...])',
        examples: [
            'AVG(10, 5) => 7.5',
            'AVG({column1}, {column2})',
            'AVG({column1}, {column2}, {column3})',
        ],
        returnType: FormulaDataTypes.NUMERIC,
    },
    ADD: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#add`,
        validation: {
            args: {
                min: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Sum of input parameters',
        syntax: 'ADD(value1, [value2, ...])',
        examples: [
            'ADD(5, 5) => 10',
            'ADD({column1}, {column2})',
            'ADD({column1}, {column2}, {column3})',
        ],
        returnType: FormulaDataTypes.NUMERIC,
    },
    DATEADD: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/date-functions#dateadd`,
        validation: {
            args: {
                rqd: 3,
                type: FormulaDataTypes.DATE,
            },
            custom: (_argTypes, parsedTree) => {
                if (parsedTree.arguments[0].type === JSEPNode.LITERAL) {
                    if (!validateDateWithUnknownFormat(parsedTree.arguments[0].value)) {
                        throw new FormulaError(FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.firstParamDateAddHaveDate' }, 'First parameter of DATEADD should be a date');
                    }
                }
                if (parsedTree.arguments[1].type === JSEPNode.LITERAL) {
                    if (typeof parsedTree.arguments[1].value !== 'number') {
                        throw new FormulaError(FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.secondParamDateAddHaveNumber' }, 'Second parameter of DATEADD should be a number');
                    }
                }
                if (parsedTree.arguments[2].type === JSEPNode.LITERAL) {
                    if (![
                        'day',
                        'week',
                        'month',
                        'year',
                        'hour',
                        'minute',
                        'second',
                    ].includes(parsedTree.arguments[2].value)) {
                        throw new FormulaError(FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.thirdParamDateAddHaveDate' }, "Third parameter of DATEADD should be one of 'day', 'week', 'month', 'year', 'hour', 'minute', 'second'");
                    }
                }
            },
        },
        description: 'Adds "count" units to Datetime.',
        syntax: 'DATEADD(date | datetime, count, ["day" | "week" | "month" | "year" | "hour" | "minute" | "second"])',
        examples: [
            'DATEADD({column1}, 2, "day")',
            'DATEADD({column1}, -2, "day")',
            'DATEADD({column1}, 2, "week")',
            'DATEADD({column1}, -2, "week")',
            'DATEADD({column1}, 2, "month")',
            'DATEADD({column1}, -2, "month")',
            'DATEADD({column1}, 2, "year")',
            'DATEADD({column1}, -2, "year")',
        ],
        returnType: FormulaDataTypes.DATE,
    },
    DATESTR: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/date-functions#datestr`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        syntax: 'DATESTR(date | datetime)',
        description: 'Formats input field into a string in "YYYY-MM-DD" format',
        examples: ['DATESTR({column1})'],
        returnType: FormulaDataTypes.STRING,
    },
    DAY: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/date-functions#day`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        syntax: 'DAY(date | datetime)',
        description: 'Extract day from a date field (1-31)',
        examples: ['DAY({column1})'],
        returnType: FormulaDataTypes.STRING,
    },
    MONTH: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/date-functions#month`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        syntax: 'MONTH(date | datetime)',
        description: 'Extract month from a date field (1-12)',
        examples: ['MONTH({column1})'],
        returnType: FormulaDataTypes.STRING,
    },
    YEAR: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/date-functions#year`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        syntax: 'YEAR(date | datetime)',
        description: 'Extract year from a date field',
        examples: ['YEAR({column1})'],
        returnType: FormulaDataTypes.STRING,
    },
    HOUR: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/date-functions#hour`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        syntax: 'DAY(time | datetime)',
        description: 'Extract hour from a time field (0-23)',
        examples: ['HOUR({column1})'],
        returnType: FormulaDataTypes.STRING,
    },
    DATETIME_DIFF: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/date-functions#datetime_diff`,
        validation: {
            args: {
                min: 2,
                max: 3,
                type: FormulaDataTypes.DATE,
            },
            custom: (_argTypes, parsedTree) => {
                if (parsedTree.arguments[0].type === JSEPNode.LITERAL) {
                    if (!validateDateWithUnknownFormat(parsedTree.arguments[0].value)) {
                        throw new FormulaError(FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.firstParamDateDiffHaveDate' }, 'First parameter of DATETIME_DIFF should be a date');
                    }
                }
                if (parsedTree.arguments[1].type === JSEPNode.LITERAL) {
                    if (!validateDateWithUnknownFormat(parsedTree.arguments[1].value)) {
                        throw new FormulaError(FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.secondParamDateDiffHaveDate' }, 'Second parameter of DATETIME_DIFF should be a date');
                    }
                }
                if (parsedTree.arguments[2] &&
                    parsedTree.arguments[2].type === JSEPNode.LITERAL) {
                    if (![
                        'milliseconds',
                        'ms',
                        'seconds',
                        's',
                        'minutes',
                        'm',
                        'hours',
                        'h',
                        'days',
                        'd',
                        'weeks',
                        'w',
                        'months',
                        'M',
                        'quarters',
                        'Q',
                        'years',
                        'y',
                    ].includes(parsedTree.arguments[2].value)) {
                        throw new FormulaError(FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.thirdParamDateDiffHaveDate' }, "Third parameter of DATETIME_DIFF should be one of 'milliseconds', 'ms', 'seconds', 's', 'minutes', 'm', 'hours', 'h', 'days', 'd', 'weeks', 'w', 'months', 'M', 'quarters', 'Q', 'years', 'y'");
                    }
                }
            },
        },
        description: 'Calculate the difference of two given date / datetime fields in specified units.',
        syntax: 'DATETIME_DIFF(date | datetime, date | datetime, ["milliseconds" | "ms" | "seconds" | "s" | "minutes" | "m" | "hours" | "h" | "days" | "d" | "weeks" | "w" | "months" | "M" | "quarters" | "Q" | "years" | "y"])',
        examples: [
            'DATEDIFF({column1}, {column2})',
            'DATEDIFF({column1}, {column2}, "seconds")',
            'DATEDIFF({column1}, {column2}, "s")',
            'DATEDIFF({column1}, {column2}, "years")',
            'DATEDIFF({column1}, {column2}, "y")',
            'DATEDIFF({column1}, {column2}, "minutes")',
            'DATEDIFF({column1}, {column2}, "m")',
            'DATEDIFF({column1}, {column2}, "days")',
            'DATEDIFF({column1}, {column2}, "d")',
        ],
        returnType: FormulaDataTypes.NUMERIC,
    },
    AND: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/conditional-expressions#and`,
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Result is TRUE if all conditions are met',
        syntax: 'AND(expr1, [expr2, ...])',
        examples: ['AND(5 > 2, 5 < 10) => 1', 'AND({column1} > 2, {column2} < 10)'],
        returnType: FormulaDataTypes.COND_EXP,
    },
    OR: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/conditional-expressions#or`,
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Result is TRUE if at least one condition is met',
        syntax: 'OR(expr1, [expr2, ...])',
        examples: ['OR(5 > 2, 5 < 10) => 1', 'OR({column1} > 2, {column2} < 10)'],
        returnType: FormulaDataTypes.COND_EXP,
    },
    CONCAT: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#concat`,
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Concatenate input parameters into a single string',
        syntax: 'CONCAT(str1, [str2, ...])',
        examples: [
            'CONCAT("AA", "BB", "CC") => "AABBCC"',
            'CONCAT({column1}, {column2}, {column3})',
        ],
        returnType: FormulaDataTypes.STRING,
    },
    TRIM: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#trim`,
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.STRING,
            },
        },
        description: 'Remove trailing and leading whitespaces from input parameter',
        syntax: 'TRIM(str)',
        examples: [
            'TRIM("         HELLO WORLD  ") => "HELLO WORLD"',
            'TRIM({column1})',
        ],
        returnType: FormulaDataTypes.STRING,
    },
    UPPER: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#upper`,
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.STRING,
            },
        },
        description: 'Converts the input parameter to an upper-case string.',
        syntax: 'UPPER(str)',
        examples: ['UPPER("nocodb") => "NOCODB"', 'UPPER({column1})'],
        returnType: FormulaDataTypes.STRING,
    },
    LOWER: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#lower`,
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.STRING,
            },
        },
        description: 'Converts the input parameter to an lower-case string.',
        syntax: 'LOWER(str)',
        examples: ['LOWER("NOCODB") => "nocodb"', 'LOWER({column1})'],
        returnType: FormulaDataTypes.STRING,
    },
    LEN: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#len`,
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.STRING,
            },
        },
        description: 'Calculate the character length of the input parameter.',
        syntax: 'LEN(value)',
        examples: ['LEN("NocoDB") => 6', 'LEN({column1})'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    MIN: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#min`,
        validation: {
            args: {
                min: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Find the minimum value among the input parameters.',
        syntax: 'MIN(value1, [value2, ...])',
        examples: ['MIN(1000, 2000) => 1000', 'MIN({column1}, {column2})'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    MAX: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#max`,
        validation: {
            args: {
                min: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Find the maximum value among the input parameters.',
        syntax: 'MAX(value1, [value2, ...])',
        examples: ['MAX(1000, 2000) => 2000', 'MAX({column1}, {column2})'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    CEILING: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#ceiling`,
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds the input parameter to the next largest integer value.',
        syntax: 'CEILING(value)',
        examples: ['CEILING(1.01) => 2', 'CEILING({column1})'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    FLOOR: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#floor`,
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Round down the input parameter to the nearest integer.',
        syntax: 'FLOOR(value)',
        examples: ['FLOOR(3.1415) => 3', 'FLOOR({column1})'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    ROUND: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#round`,
        validation: {
            args: {
                min: 1,
                max: 2,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds the number to a specified decimal places or the nearest integer if precision is not specified',
        syntax: 'ROUND(value, precision), ROUND(value)',
        examples: [
            'ROUND(3.1415) => 3',
            'ROUND(3.1415, 2) => 3.14',
            'ROUND({column1}, 3)',
        ],
        returnType: FormulaDataTypes.NUMERIC,
    },
    MOD: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#mod`,
        validation: {
            args: {
                rqd: 2,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Calculate the remainder resulting from integer division of input parameters.',
        syntax: 'MOD(value1, value2)',
        examples: ['MOD(1024, 1000) => 24', 'MOD({column}, 2)'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    REPEAT: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#repeat`,
        validation: {
            args: {
                rqd: 2,
            },
            custom(argTypes, parsedTree) {
                var _a, _b;
                if (argTypes[1] !== FormulaDataTypes.NUMERIC) {
                    throw new FormulaError(FormulaErrorType.INVALID_ARG, {
                        key: 'msg.formula.typeIsExpected',
                        type: 'Numeric',
                        calleeName: (_b = (_a = parsedTree.callee) === null || _a === void 0 ? void 0 : _a.name) === null || _b === void 0 ? void 0 : _b.toUpperCase(),
                        position: 2,
                    }, 'The REPEAT function requires a numeric as the parameter at position 2');
                }
            },
        },
        description: 'Concatenate the specified number of copies of the input parameter string.',
        syntax: 'REPEAT(str, count)',
        examples: ['REPEAT("A", 5) => "AAAAA"', 'REPEAT({column}, 5)'],
        returnType: FormulaDataTypes.STRING,
    },
    LOG: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#log`,
        validation: {
            args: {
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Compute the logarithm of the input parameter to the specified base (default = e).',
        syntax: 'LOG([base], value)',
        examples: ['LOG(2, 1024) => 10', 'LOG(2, {column1})'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    EXP: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#exp`,
        validation: {
            args: {
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Compute the exponential value of the input parameter (e raised to the power specified)',
        syntax: 'EXP(power)',
        examples: ['EXP(1) => 2.718281828459045', 'EXP({column1})'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    POWER: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#power`,
        validation: {
            args: {
                rqd: 2,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Compute base raised to the exponent power.',
        syntax: 'POWER(base, exponent)',
        examples: ['POWER(2, 10) => 1024', 'POWER({column1}, 10)'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    SQRT: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#sqrt`,
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Calculate the square root of the input parameter.',
        syntax: 'SQRT(value)',
        examples: ['SQRT(100) => 10', 'SQRT({column1})'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    ABS: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#abs`,
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Obtain the absolute value of the input parameter.',
        syntax: 'ABS(value)',
        examples: ['ABS({column1})'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    NOW: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/date-functions#now`,
        validation: {
            args: {
                rqd: 0,
                type: FormulaDataTypes.DATE,
            },
        },
        description: 'Retrieve the current time and day.',
        syntax: 'NOW()',
        examples: ['NOW() => 2022-05-19 17:20:43'],
        returnType: FormulaDataTypes.DATE,
    },
    REPLACE: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#replace`,
        validation: {
            args: {
                rqd: 3,
                type: FormulaDataTypes.STRING,
            },
        },
        description: 'Replace all occurrences of "searchStr" with "replaceStr" in the given string.',
        syntax: 'REPLACE(str, searchStr, replaceStr)',
        examples: [
            'REPLACE("AABBCC", "AA", "BB") => "BBBBCC"',
            'REPLACE({column1}, {column2}, {column3})',
        ],
        returnType: FormulaDataTypes.STRING,
    },
    SEARCH: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#search`,
        validation: {
            args: {
                rqd: 2,
                type: FormulaDataTypes.STRING,
            },
        },
        description: 'Retrieve the index of the specified "searchStr" if found; otherwise, returns 0.',
        syntax: 'SEARCH(str, searchStr)',
        examples: [
            'SEARCH("HELLO WORLD", "WORLD") => 7',
            'SEARCH({column1}, "abc")',
        ],
        returnType: FormulaDataTypes.NUMERIC,
    },
    INT: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#int`,
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Obtain the integer value of the input parameter',
        syntax: 'INT(value)',
        examples: ['INT(3.1415) => 3', 'INT({column1})'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    RIGHT: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#right`,
        validation: {
            args: {
                rqd: 2,
                type: [FormulaDataTypes.STRING, FormulaDataTypes.NUMERIC],
            },
        },
        description: 'Retrieve the last n characters from the input string.',
        syntax: 'RIGHT(str, n)',
        examples: ['RIGHT("HELLO WORLD", 5) => WORLD', 'RIGHT({column1}, 3)'],
        returnType: FormulaDataTypes.STRING,
    },
    LEFT: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#left`,
        validation: {
            args: {
                rqd: 2,
                type: [FormulaDataTypes.STRING, FormulaDataTypes.NUMERIC],
            },
        },
        description: 'Retrieve the first n characters from the input string.',
        syntax: 'LEFT(str, n)',
        examples: ['LEFT({column1}, 2)', 'LEFT("ABCD", 2) => "AB"'],
        returnType: FormulaDataTypes.STRING,
    },
    SUBSTR: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#substr`,
        validation: {
            args: {
                min: 2,
                max: 3,
                type: [
                    FormulaDataTypes.STRING,
                    FormulaDataTypes.NUMERIC,
                    FormulaDataTypes.NUMERIC,
                ],
            },
        },
        description: 'Extracts a substring of length "n" from the input string, starting from the specified position.',
        syntax: '	SUBTR(str, position, [n])',
        examples: [
            'SUBSTR("HELLO WORLD", 7) => WORLD',
            'SUBSTR("HELLO WORLD", 7, 3) => WOR',
            'SUBSTR({column1}, 7, 5)',
        ],
        returnType: FormulaDataTypes.STRING,
    },
    MID: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#mid`,
        validation: {
            args: {
                rqd: 3,
                type: [
                    FormulaDataTypes.STRING,
                    FormulaDataTypes.NUMERIC,
                    FormulaDataTypes.NUMERIC,
                ],
            },
        },
        description: 'Extracts a substring; an alias for SUBSTR.',
        syntax: 'MID(str, position, [count])',
        examples: ['MID("NocoDB", 3, 2) => "co"', 'MID({column1}, 3, 2)'],
        returnType: FormulaDataTypes.STRING,
    },
    ISBLANK: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#isblank`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        description: 'Check if the input parameter is blank.',
        syntax: 'ISBLANK(value)',
        examples: ['ISBLANK({column1}) => false', 'ISBLANK("") => true'],
        returnType: FormulaDataTypes.BOOLEAN,
    },
    ISNOTBLANK: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#isnotblank`,
        validation: {
            args: {
                rqd: 1,
            },
        },
        description: 'Check if the input parameter is not blank.',
        syntax: 'ISNOTBLANK(value)',
        examples: ['ISNOTBLANK({column1}) => true', 'ISNOTBLANK("") => false'],
        returnType: FormulaDataTypes.BOOLEAN,
    },
    IF: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/conditional-expressions#if`,
        validation: {
            args: {
                min: 2,
                max: 3,
            },
        },
        description: 'Evaluate successCase if the expression is TRUE, else the failureCase.',
        syntax: 'IF(expr, successCase, failureCase)',
        examples: [
            'IF(5 > 1, "YES", "NO") => "YES"',
            'IF({column} > 1, "YES", "NO")',
        ],
        returnType: (argTypes) => {
            // extract all return types except NULL, since null can be returned by any type
            const returnValueTypes = new Set(argTypes.slice(1).filter((type) => type !== FormulaDataTypes.NULL));
            // if there are more than one return types or if there is a string return type
            // return type as string else return the type
            if (returnValueTypes.size > 1 ||
                returnValueTypes.has(FormulaDataTypes.STRING)) {
                return FormulaDataTypes.STRING;
            }
            else if (returnValueTypes.has(FormulaDataTypes.NUMERIC)) {
                return FormulaDataTypes.NUMERIC;
            }
            else if (returnValueTypes.has(FormulaDataTypes.BOOLEAN)) {
                return FormulaDataTypes.BOOLEAN;
            }
            else if (returnValueTypes.has(FormulaDataTypes.DATE)) {
                return FormulaDataTypes.DATE;
            }
            // if none of the above conditions are met, return the first return argument type
            return argTypes[1];
        },
    },
    SWITCH: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/conditional-expressions#switch`,
        validation: {
            args: {
                min: 3,
            },
            custom: (_argTypes, _parseTree) => {
                // Todo: Add validation for switch
            },
        },
        description: 'Evaluate case value based on expression output; if no match is found, evaluate default case.',
        syntax: 'SWITCH(expr, [pattern, value, ..., default])',
        examples: [
            'SWITCH(1, 1, "One", 2, "Two", "N/A") => "One""',
            'SWITCH(2, 1, "One", 2, "Two", "N/A") => "Two"',
            'SWITCH(3, 1, "One", 2, "Two", "N/A") => "N/A"',
            'SWITCH({column1}, 1, "One", 2, "Two", "N/A")',
        ],
        returnType: (argTypes) => {
            // extract all return types except NULL, since null can be returned by any type
            const returnValueTypes = new Set(argTypes.slice(2).filter((_, i) => i % 2 === 0));
            // if there are more than one return types or if there is a string return type
            // return type as string else return the type
            if (returnValueTypes.size > 1 ||
                returnValueTypes.has(FormulaDataTypes.STRING)) {
                return FormulaDataTypes.STRING;
            }
            else if (returnValueTypes.has(FormulaDataTypes.NUMERIC)) {
                return FormulaDataTypes.NUMERIC;
            }
            else if (returnValueTypes.has(FormulaDataTypes.BOOLEAN)) {
                return FormulaDataTypes.BOOLEAN;
            }
            else if (returnValueTypes.has(FormulaDataTypes.DATE)) {
                return FormulaDataTypes.DATE;
            }
            // if none of the above conditions are met, return the first return argument type
            return argTypes[1];
        },
    },
    URL: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#url`,
        validation: {
            args: {
                min: 1,
                max: 2,
                type: [FormulaDataTypes.STRING, FormulaDataTypes.STRING],
            },
        },
        description: 'Verify and convert to a hyperlink if the input is a valid URL.',
        syntax: 'URL(string, [label])',
        examples: [
            'URL("https://github.com/nocodb/nocodb")',
            'URL({column1})',
            'URL("https://github.com/nocodb/nocodb", "NocoDB")',
            'URL({column1}, {column1})',
        ],
        returnType: FormulaDataTypes.STRING,
    },
    URLENCODE: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#urlencode`,
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.STRING,
            },
        },
        description: 'Percent-encode the input parameter for use in URLs',
        syntax: 'URLENCODE(str)',
        examples: [
            'URLENCODE("Hello, world") => "Hello%2C%20world"',
            'URLENCODE({column1})',
        ],
        returnType: FormulaDataTypes.STRING,
    },
    WEEKDAY: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/date-functions#weekday`,
        validation: {
            args: {
                min: 1,
                max: 2,
                type: FormulaDataTypes.NUMERIC,
            },
            custom(_argTypes, parsedTree) {
                if (parsedTree.arguments[0].type === JSEPNode.LITERAL) {
                    if (!validateDateWithUnknownFormat(parsedTree.arguments[0].value)) {
                        throw new FormulaError(FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.firstParamWeekDayHaveDate' }, 'First parameter of WEEKDAY should be a date');
                    }
                }
                // if second argument is present and literal then validate it
                if (parsedTree.arguments[1] &&
                    parsedTree.arguments[1].type === JSEPNode.LITERAL) {
                    const value = parsedTree.arguments[1].value;
                    if (typeof value !== 'string' ||
                        ![
                            'sunday',
                            'monday',
                            'tuesday',
                            'wednesday',
                            'thursday',
                            'friday',
                            'saturday',
                        ].includes(value.toLowerCase())) {
                        throw new FormulaError(FormulaErrorType.TYPE_MISMATCH, { key: 'msg.formula.secondParamWeekDayHaveDate' }, 'Second parameter of WEEKDAY should be day of week string');
                    }
                }
            },
        },
        description: 'Retrieve the day of the week as an integer (0-6), starting from Monday by default.',
        syntax: 'WEEKDAY(date, [startDayOfWeek])',
        examples: ['WEEKDAY("2021-06-09")', 'WEEKDAY(NOW(), "sunday")'],
        returnType: FormulaDataTypes.NUMERIC,
    },
    TRUE: {
        validation: {
            args: {
                max: 0,
            },
        },
        description: 'Returns 1',
        syntax: 'TRUE()',
        examples: ['TRUE()'],
        returnType: FormulaDataTypes.NUMERIC,
        // TODO: Add docs url
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/logical-functions#true`,
    },
    FALSE: {
        validation: {
            args: {
                max: 0,
            },
        },
        description: 'Returns 0',
        syntax: 'FALSE()',
        examples: ['FALSE()'],
        returnType: FormulaDataTypes.NUMERIC,
        // TODO: Add docs url
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/logical-functions#false`,
    },
    ARRAYUNIQUE: {
        validation: {
            args: {
                rqd: 1,
            },
            custom: customValidationArray,
        },
        description: 'Return unique items from the given array',
        syntax: 'ARRAYUNIQUE(value)',
        examples: ['ARRAYUNIQUE({column})'],
        returnType: FormulaDataTypes.ARRAY,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/array-functions#arrayunique`,
    },
    ARRAYSORT: {
        validation: {
            args: {
                min: 1,
                max: 2,
            },
            custom: customValidationArray,
        },
        description: 'Sort an array result',
        syntax: 'ARRAYSORT(value, [direction])',
        examples: ['ARRAYSORT({column}, "desc")'],
        returnType: FormulaDataTypes.ARRAY,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/array-functions#arraysort`,
    },
    ARRAYCOMPACT: {
        validation: {
            args: {
                rqd: 1,
            },
            custom: customValidationArray,
        },
        description: 'Removes empty strings and null values from the array',
        syntax: 'ARRAYCOMPACT(value)',
        examples: ['ARRAYCOMPACT({column})'],
        returnType: FormulaDataTypes.ARRAY,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/array-functions#arraycompact`,
    },
    ARRAYSLICE: {
        validation: {
            args: {
                min: 2,
                max: 3,
            },
            custom: customValidationArray,
        },
        description: 'Removes empty strings and null values from the array',
        syntax: 'ARRAYSLICE(value, start, [end])',
        examples: ['ARRAYSLICE({column})'],
        returnType: FormulaDataTypes.ARRAY,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/array-functions#arrayslice`,
    },
    REGEX_MATCH: {
        validation: {
            args: {
                rqd: 2,
                type: FormulaDataTypes.STRING,
            },
        },
        description: 'Verifies whether the input text matches a regular expression, returning 1 for a match and 0 otherwise.',
        syntax: 'REGEX_MATCH(string, regex)',
        examples: ['REGEX_MATCH({title}, "abc.*")'],
        returnType: FormulaDataTypes.NUMERIC,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#regex_match`,
    },
    REGEX_EXTRACT: {
        validation: {
            args: {
                rqd: 2,
                type: FormulaDataTypes.STRING,
            },
        },
        description: 'Retrieve the first match of a regular expression in a string.',
        syntax: 'REGEX_EXTRACT(string, regex)',
        examples: ['REGEX_EXTRACT({title}, "abc.*")'],
        returnType: FormulaDataTypes.STRING,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#regex_extract`,
    },
    REGEX_REPLACE: {
        validation: {
            args: {
                rqd: 3,
                type: FormulaDataTypes.STRING,
            },
        },
        description: 'Replace all occurrences of a regular expression in a string with a specified replacement string.',
        syntax: 'REGEX_MATCH(string, regex, replacement)',
        examples: ['REGEX_EXTRACT({title}, "abc.*", "abcd")'],
        returnType: FormulaDataTypes.STRING,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/string-functions#regex_replace`,
    },
    BLANK: {
        validation: {
            args: {
                rqd: 0,
            },
        },
        description: 'Yields a null value.',
        syntax: 'BLANK()',
        examples: ['BLANK()'],
        returnType: FormulaDataTypes.NULL,
        // TODO: Add docs url
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/logical-functions#blank`,
    },
    XOR: {
        validation: {
            args: {
                min: 1,
            },
            // todo: validation for boolean
        },
        description: 'Verifies whether an odd number of arguments are true, returning true if so, and false otherwise.',
        syntax: 'XOR(expression, [exp2, ...])',
        examples: ['XOR(TRUE(), FALSE(), TRUE())'],
        returnType: FormulaDataTypes.BOOLEAN,
        // TODO: Add docs url
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/logical-functions#xor`,
    },
    EVEN: {
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds up the specified value to the nearest even integer that is greater than or equal to the specified value',
        syntax: 'EVEN(value)',
        examples: ['EVEN({column})'],
        returnType: FormulaDataTypes.NUMERIC,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#even`,
    },
    ODD: {
        validation: {
            args: {
                rqd: 1,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds up the specified value to the nearest odd integer that is greater than or equal to the specified value',
        syntax: 'ODD(value)',
        examples: ['ODD({column})'],
        returnType: FormulaDataTypes.NUMERIC,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#odd`,
    },
    RECORD_ID: {
        validation: {
            args: {
                rqd: 0,
            },
        },
        description: 'Retrieve the record ID of the current record.',
        syntax: 'RECORD_ID()',
        examples: ['RECORD_ID()'],
        // todo: resolve return type based on the args
        returnType: () => {
            return FormulaDataTypes.STRING;
        },
    },
    COUNTA: {
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Counts non-empty arguments',
        syntax: 'COUNTA(value1, [value2, ...])',
        examples: ['COUNTA({field1}, {field2})'],
        returnType: FormulaDataTypes.NUMERIC,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#counta`,
    },
    COUNT: {
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Counts numerical arguments',
        syntax: 'COUNT(value1, [value2, ...])',
        examples: ['COUNT({field1}, {field2})'],
        returnType: FormulaDataTypes.NUMERIC,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#count`,
    },
    COUNTALL: {
        validation: {
            args: {
                min: 1,
            },
        },
        description: 'Counts the number of arguments',
        syntax: 'COUNTALL(value1, [value2, ...])',
        examples: ['COUNTALL({field1}, {field2})'],
        returnType: FormulaDataTypes.NUMERIC,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#countall`,
    },
    ROUNDDOWN: {
        validation: {
            args: {
                min: 1,
                max: 2,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds down the value after the decimal point to the specified number of decimal places given by "precision" (default is 0).',
        syntax: 'ROUNDDOWN(value, [precision])',
        examples: ['ROUNDDOWN({field1})', 'ROUNDDOWN({field1}, 2)'],
        returnType: FormulaDataTypes.NUMERIC,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#rounddown`,
    },
    ROUNDUP: {
        validation: {
            args: {
                min: 1,
                max: 2,
                type: FormulaDataTypes.NUMERIC,
            },
        },
        description: 'Rounds up the value after the decimal point to the specified number of decimal places given by "precision" (default is 0).',
        syntax: 'ROUNDUP(value, [precision])',
        examples: ['ROUNDUP({field1})', 'ROUNDUP({field1}, 2)'],
        returnType: FormulaDataTypes.NUMERIC,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#roundup`,
    },
    VALUE: {
        validation: {
            args: {
                rqd: 1,
            },
        },
        description: 'Extracts the numeric value from a string, handling % or - appropriately, and returns the resulting numeric value.',
        syntax: 'VALUE(value)',
        examples: ['VALUE({field})', 'VALUE("abc10000%")', 'VALUE("$10000")'],
        returnType: FormulaDataTypes.NUMERIC,
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/numeric-functions#value`,
    },
    JSON_EXTRACT: {
        docsUrl: `${API_DOC_PREFIX}/field-types/formula/json-functions#json_extract`,
        validation: {
            args: {
                min: 2,
                max: 2,
                type: [FormulaDataTypes.STRING, FormulaDataTypes.STRING],
            },
        },
        description: 'Extracts a value from a JSON string using a jq-like syntax',
        syntax: 'JSON_EXTRACT(json_string, path)',
        examples: [
            'JSON_EXTRACT(\'{"a": {"b": "c"}}\', \'.a.b\') => "c"',
            "JSON_EXTRACT({json_column}, '.key')",
        ],
        returnType: FormulaDataTypes.STRING,
    },
    // Disabling these functions for now; these act as alias for CreatedAt & UpdatedAt fields;
    // Issue: Error noticed if CreatedAt & UpdatedAt fields are removed from the table after creating these formulas
    //
    // CREATED_TIME: {
    //   validation: {
    //     args: {
    //       rqd: 0,
    //     },
    //   },
    //   description: 'Returns the created time of the current record if it exists',
    //   syntax: 'CREATED_TIME()',
    //   examples: ['CREATED_TIME()'],
    // },
    LAST_MODIFIED_TIME: {
        validation: {
            args: {
                min: 0,
            },
            custom(_argTypes, parsedTree, columns) {
                if (parsedTree.arguments.length) {
                    // check if meta column exists or not, if not throw error when user provides args}
                    if (!columns.find((column) => column.uidt === UITypes.Meta)) {
                        throw new FormulaError(FormulaErrorType.INVALID_ARG, {}, 'LAST_MODIFIED_TIME with field arguments is not supported');
                    }
                }
                for (const arg of parsedTree.arguments) {
                    if (arg.type !== JSEPNode.IDENTIFIER) {
                        throw new FormulaError(FormulaErrorType.INVALID_ARG, {}, 'Only column references are allowed as arguments for LAST_MODIFIED_TIME');
                    }
                    // now check if the column allows LAST_MODIFIED_TIME
                    const virtualOrSystemColRef = columns.find((c) => (c.title === arg.name || c.id === arg.name) &&
                        (isSystemColumn(c) || isVirtualCol(c)));
                    if (virtualOrSystemColRef) {
                        throw new FormulaError(FormulaErrorType.INVALID_ARG, {}, `LAST_MODIFIED_TIME() is not supported for ${virtualOrSystemColRef.system
                            ? 'System'
                            : virtualOrSystemColRef.uidt} field.`);
                    }
                }
            },
        },
        description: 'Returns the last modified time of the current record or selection if it exists',
        syntax: 'LAST_MODIFIED_TIME([field1, ...])',
        examples: [
            'LAST_MODIFIED_TIME()',
            'LAST_MODIFIED_TIME({Status}, {Priority})',
        ],
        returnType: FormulaDataTypes.DATE,
    },
};
//# sourceMappingURL=data:application/json;base64,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