import { arrUniq } from '../arrayHelpers';
import UITypes from '../UITypes';
import { FormulaDataTypes } from './enums';
import { arrFlatMap } from '../arrayHelpers';
const IMPURE_OPR_UIDT_MAP = new Map([
    [UITypes.SingleSelect, UITypes.SingleLineText],
    [UITypes.MultiSelect, UITypes.SingleLineText],
    [UITypes.Attachment, UITypes.SingleLineText],
    [UITypes.User, UITypes.SingleLineText],
    [UITypes.Email, UITypes.SingleLineText],
    [UITypes.URL, UITypes.SingleLineText],
    [UITypes.Number, UITypes.Decimal],
    [UITypes.Rating, UITypes.Decimal],
    [UITypes.Year, UITypes.Number],
    [UITypes.Year, UITypes.Number],
]);
/**
 * Filters and refines referenced column information based on allowed UI data types (UIDTs).
 * This function ensures that the data types associated with referenced columns and their candidates
 * are valid and consistent with a set of allowed UIDTs for a specific operation or formula.
 *
 * @param {object} params - The parameters for filtering.
 * @param {ReferencedInfo} params.referencedInfo - An object containing details about a referenced column,
 *   including the `referencedColumn` itself (with its ID and UIDT) and a list of `uidtCandidates` (potential UIDTs).
 * @param {UITypes[]} params.allowedUidts - An array of `UITypes` that are considered valid for the current context.
 * @param {UITypes} params.defaultUidt - A fallback `UIType` to use if no valid `uidtCandidates` are found after filtering.
 * @param {boolean} [params.isPureOperation=false] - A flag indicating if the operation is "pure."
 *   A pure operation (e.g., TRIM, ARRAYUNIQUE) doesn't fundamentally change the display value's
 *   underlying type, even if it modifies the value itself. Impure operations might map certain
 *   UI types (like SingleSelect) to a more generic type (like SingleLineText).
 * @returns {ReferencedInfo} A new `ReferencedInfo` object with the updated `referencedColumn`,
 *   `uidtCandidates`, and `invalidForReferenceColumn` status, ensuring consistency with formula requirements.
 */
const filterReferencedInfoByUidt = ({ referencedInfo, allowedUidts, defaultUidt, isPureOperation = false, }) => {
    var _a, _b;
    let referencedColumn = referencedInfo.referencedColumn;
    if (referencedColumn && !isPureOperation) {
        referencedColumn =
            allowedUidts.includes(referencedInfo.referencedColumn.uidt) &&
                !IMPURE_OPR_UIDT_MAP.has(referencedInfo.referencedColumn.uidt)
                ? referencedInfo.referencedColumn
                : undefined;
    }
    let uidtCandidates = (_b = (_a = referencedInfo.uidtCandidates) === null || _a === void 0 ? void 0 : _a.filter((uidt) => allowedUidts.includes(uidt))) !== null && _b !== void 0 ? _b : [];
    if (!isPureOperation) {
        uidtCandidates = uidtCandidates
            .map((c) => { var _a; return (_a = IMPURE_OPR_UIDT_MAP.get(c)) !== null && _a !== void 0 ? _a : c; })
            .filter((k) => k);
    }
    return Object.assign(Object.assign({}, referencedInfo), { referencedColumn, uidtCandidates: uidtCandidates.length > 0 ? arrUniq(uidtCandidates) : [defaultUidt], 
        // Determine if the referenced column is invalid. This can happen if it was already marked as invalid,
        // or if the `referencedColumn` was present in the original `referencedInfo` but got filtered out
        // during the current processing (i.e., `referencedColumn` is now undefined while `referencedInfo.referencedColumn` was not).
        invalidForReferenceColumn: referencedInfo.invalidForReferenceColumn ||
            !!referencedColumn !== !!referencedInfo.referencedColumn });
};
/**
 * Extracts and refines information about referenced columns from an array of parsed formula nodes.
 * This function is crucial for ensuring type compatibility and validity of referenced columns
 * within formula arguments based on the expected data type of the formula's output.
 *
 * @param {ParsedFormulaNode[]} nodes - An array of parsed formula nodes, representing the arguments
 *   of a formula. Each node may contain information about referenced columns and their potential
 *   UI data types (UIDTs).
 * @param {FormulaDataTypes} dataType - The expected output data type of the formula. This is used
 *   to filter and validate the UIDTs of the referenced columns.
 * @param {{ isPureOperation?: boolean }} [option] - An optional object containing additional
 *   configuration.
 * @param {boolean} [option.isPureOperation=false] - A flag indicating if the operation is "pure."
 *   A pure operation (e.g., TRIM, ARRAYUNIQUE) does not fundamentally change the display value's
 *   underlying type, even if it modifies the value itself. Impure operations might map certain
 *   UI types (like SingleSelect) to a more generic type (like SingleLineText).
 * @returns {ReferencedInfo} A `ReferencedInfo` object containing the aggregated and filtered
 *   information about referenced columns, their UIDT candidates, and a flag indicating if the
 *   reference is invalid.
 */
export const getReferencedInfoFromArgs = (nodes, dataType, option = { isPureOperation: false }) => {
    let referencedColumn;
    let invalidForReferenceColumn = nodes.some((c) => c.invalidForReferenceColumn);
    const referencedColumns = invalidForReferenceColumn
        ? []
        : nodes.map((k) => k.referencedColumn).filter((k) => k);
    const uniqueLength = arrUniq(referencedColumns.map((k) => k.id)).length;
    if (uniqueLength === 1) {
        referencedColumn = referencedColumns[0];
    }
    else if (uniqueLength > 1) {
        invalidForReferenceColumn = true;
    }
    const uidtCandidates = arrFlatMap(nodes.map((k) => k.uidtCandidates));
    const referencedInfo = {
        referencedColumn,
        uidtCandidates,
        invalidForReferenceColumn,
    };
    switch (dataType) {
        case FormulaDataTypes.BOOLEAN: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [UITypes.Checkbox], defaultUidt: UITypes.Checkbox }, option));
        }
        case FormulaDataTypes.COND_EXP: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [UITypes.Checkbox], defaultUidt: UITypes.Checkbox }, option));
        }
        case FormulaDataTypes.STRING: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [
                    UITypes.SingleLineText,
                    UITypes.LongText,
                    UITypes.Email,
                    UITypes.URL,
                ], defaultUidt: UITypes.SingleLineText }, option));
        }
        case FormulaDataTypes.NUMERIC: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [
                    UITypes.Number,
                    UITypes.Decimal,
                    UITypes.Currency,
                    UITypes.Percent,
                    UITypes.Rating,
                ], defaultUidt: UITypes.Decimal }, option));
        }
        case FormulaDataTypes.DATE: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [UITypes.Date, UITypes.DateTime], defaultUidt: UITypes.DateTime }, option));
        }
        case FormulaDataTypes.INTERVAL: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [UITypes.Time, UITypes.Duration], defaultUidt: UITypes.Time }, option));
        }
        case FormulaDataTypes.ARRAY: {
            return filterReferencedInfoByUidt(Object.assign({ referencedInfo, allowedUidts: [UITypes.LinkToAnotherRecord, UITypes.Lookup], defaultUidt: UITypes.SingleLineText }, option));
        }
        default: {
            return {
                uidtCandidates: [UITypes.SingleLineText],
                invalidForReferenceColumn,
            };
        }
    }
};
/**
 * get referenced column info for call expression
 */
export const extractBinaryExpReferencedInfo = ({ parsedTree, left, right, }) => {
    return getReferencedInfoFromArgs([left, right], parsedTree.dataType, {
        isPureOperation: false,
    });
};
/**
 * get referenced column info for call expression
 */
export const extractCallExpressionReferencedInfo = ({ parsedTree, }) => {
    var _a;
    let nodeArgs = (_a = parsedTree.arguments) !== null && _a !== void 0 ? _a : [];
    let isPureOperation = false;
    switch (parsedTree.callee.name) {
        case 'TRIM':
        case 'MAX':
        case 'MIN':
            isPureOperation = true;
            break;
        case 'DATEADD':
        case 'ROUND':
        case 'MOD':
        case 'POWER':
        case 'REPEAT':
        case 'REPLACE':
        case 'REGEX_EXTRACT':
        case 'REGEX_REPLACE':
        case 'MID':
        case 'LEFT':
        case 'RIGHT':
        case 'ROUNDDOWN':
        case 'ROUNDUP':
        case 'JSON_EXTRACT':
        case 'SUBSTR': {
            nodeArgs = nodeArgs.slice(0, 1);
            break;
        }
        case 'LOG': {
            nodeArgs = nodeArgs.slice(1, 2);
            break;
        }
        // URL won't need the referenced column info
        case 'URL':
        case 'REGEX_MATCH':
        case 'ISBLANK':
        case 'ISNOTBLANK':
        case 'DAY':
        case 'MONTH':
        case 'YEAR':
        case 'HOUR':
        case 'WEEKDAY':
        case 'LEN':
        case 'DATESTR':
        case 'AND':
        case 'OR':
        case 'XOR':
        case 'COUNTA':
        case 'COUNT':
        case 'COUNTALL':
        case 'SEARCH': {
            nodeArgs = [];
            break;
        }
        case 'ARRAYUNIQUE':
        case 'ARRAYSORT':
        case 'ARRAYCOMPACT':
        case 'ARRAYSLICE': {
            nodeArgs = nodeArgs.slice(0, 1);
            isPureOperation = true;
            break;
        }
        case 'IF': {
            nodeArgs = nodeArgs.slice(1);
            // is pure operation if the 2nd and 3rd arguments only has data type NULL or single referenced column
            const uniqueReferencedColumnIds = arrUniq(nodeArgs.map((n) => { var _a; return (_a = n.referencedColumn) === null || _a === void 0 ? void 0 : _a.id; }).filter((k) => k));
            const notNullDataTypes = nodeArgs
                .filter((n) => !n.referencedColumn && n.dataType !== FormulaDataTypes.NULL)
                .map((n) => n.dataType);
            isPureOperation =
                notNullDataTypes.length === 0 && uniqueReferencedColumnIds.length <= 1;
            break;
        }
        case 'SWITCH': {
            nodeArgs = nodeArgs.slice(1).filter((_val, index) => index % 2 === 1);
            // is pure operation if resulting arguments only has data type NULL or single referenced column
            // is pure operation if the 2nd and 3rd arguments only has data type NULL or single referenced column
            const uniqueReferencedColumnIds = arrUniq(nodeArgs.map((n) => { var _a; return (_a = n.referencedColumn) === null || _a === void 0 ? void 0 : _a.id; }).filter((k) => k));
            const notNullDataTypes = nodeArgs
                .filter((n) => !n.referencedColumn && n.dataType !== FormulaDataTypes.NULL)
                .map((n) => n.dataType);
            isPureOperation =
                notNullDataTypes.length === 0 && uniqueReferencedColumnIds.length <= 1;
            break;
        }
    }
    return getReferencedInfoFromArgs(nodeArgs, parsedTree.dataType, {
        isPureOperation,
    });
};
//# sourceMappingURL=data:application/json;base64,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