var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import UITypes from '../../lib/UITypes';
import { FormulaDataTypes, FormulaErrorType, JSEPNode, } from '../../lib/formula/enums';
import { FormulaError } from '../../lib/formula/error';
import { handleFormulaError } from '../../lib/formula/handle-formula-error';
import { formulas } from '../../lib/formula/formulas';
import { formulaJsep, formulaJsepWithIndex } from '../../lib/formula/jsepInstances';
import { SqlUiFactory } from '../../lib/sqlUi';
import { extractBinaryExpReferencedInfo, extractCallExpressionReferencedInfo, } from '../../lib/formula/referenced-info-extractor';
import { unifiedMeta } from '../../lib/unifiedMeta';
import { getColOptions } from '../../lib/unifiedMeta/getColOptions';
function extractColumnIdentifierType(_a) {
    return __awaiter(this, arguments, void 0, function* ({ col, columns, getMeta, clientOrSqlUi, }) {
        var _b, _c, _d, _e, _f, _g;
        const res = {};
        const sqlUI = typeof clientOrSqlUi === 'string'
            ? SqlUiFactory.create({ client: clientOrSqlUi })
            : clientOrSqlUi;
        switch (col === null || col === void 0 ? void 0 : col.uidt) {
            // string
            case UITypes.SingleLineText:
            case UITypes.LongText:
            case UITypes.MultiSelect:
            case UITypes.SingleSelect:
            case UITypes.PhoneNumber:
            case UITypes.Email:
            case UITypes.URL:
            case UITypes.User:
            case UITypes.CreatedBy:
            case UITypes.LastModifiedBy:
                res.dataType = FormulaDataTypes.STRING;
                break;
            // numeric
            case UITypes.Year:
            case UITypes.Number:
            case UITypes.Decimal:
            case UITypes.Rating:
            case UITypes.Count:
            case UITypes.AutoNumber:
                res.dataType = FormulaDataTypes.NUMERIC;
                break;
            // date
            case UITypes.Date:
            case UITypes.DateTime:
            case UITypes.CreatedTime:
            case UITypes.LastModifiedTime:
                res.dataType = FormulaDataTypes.DATE;
                break;
            case UITypes.Currency:
            case UITypes.Percent:
            case UITypes.Duration:
            case UITypes.Links:
                res.dataType = FormulaDataTypes.NUMERIC;
                break;
            case UITypes.Rollup:
                {
                    const rollupFunction = (yield unifiedMeta.getColOptions(unifiedMeta.getContextFromObject(col), {
                        column: col,
                    })).rollup_function;
                    if ([
                        'count',
                        'avg',
                        'sum',
                        'countDistinct',
                        'sumDistinct',
                        'avgDistinct',
                    ].includes(rollupFunction)) {
                        // these functions produce a numeric value, which can be used in numeric functions
                        res.dataType = FormulaDataTypes.NUMERIC;
                    }
                    else {
                        const rollupColOptions = yield getColOptions(unifiedMeta.getContextFromObject(col), { column: col });
                        const relationColumn = columns.find((column) => column.id === rollupColOptions.fk_relation_column_id);
                        const relationColumnOpt = yield getColOptions(unifiedMeta.getContextFromObject(col), { column: relationColumn });
                        // the value is based on the foreign rollup column type
                        const refTableMeta = yield getMeta(unifiedMeta.getContextFromObject(Object.assign(Object.assign({}, col), { base_id: (_b = relationColumnOpt.fk_related_base_id) !== null && _b !== void 0 ? _b : col.base_id })), {
                            id: relationColumnOpt.fk_related_model_id,
                        });
                        const refTableColumns = refTableMeta.columns;
                        const childFieldColumn = refTableColumns.find((col) => col.id === rollupColOptions.fk_rollup_column_id);
                        // extract type and add to res
                        Object.assign(res, yield extractColumnIdentifierType({
                            col: childFieldColumn,
                            columns: refTableColumns,
                            getMeta,
                            clientOrSqlUi,
                        }));
                    }
                }
                break;
            case UITypes.Attachment:
                res.dataType = FormulaDataTypes.STRING;
                break;
            case UITypes.Checkbox:
                if (col.dt === 'boolean' || col.dt === 'bool') {
                    res.dataType = FormulaDataTypes.BOOLEAN;
                }
                else {
                    res.dataType = FormulaDataTypes.NUMERIC;
                }
                break;
            case UITypes.Time:
                res.dataType = FormulaDataTypes.INTERVAL;
                break;
            case UITypes.ID:
            case UITypes.ForeignKey:
            case UITypes.SpecificDBType:
                {
                    if (sqlUI) {
                        const abstractType = sqlUI.getAbstractType(col);
                        if (['integer', 'float', 'decimal'].includes(abstractType)) {
                            res.dataType = FormulaDataTypes.NUMERIC;
                        }
                        else if (['boolean'].includes(abstractType)) {
                            res.dataType = FormulaDataTypes.BOOLEAN;
                        }
                        else if (['date', 'datetime', 'time', 'year'].includes(abstractType)) {
                            res.dataType = FormulaDataTypes.DATE;
                        }
                        else {
                            res.dataType = FormulaDataTypes.STRING;
                        }
                    }
                    else {
                        res.dataType = FormulaDataTypes.UNKNOWN;
                    }
                }
                break;
            // not supported
            case UITypes.Lookup: {
                const colContext = unifiedMeta.getContextFromObject(col);
                const lookupColOption = yield unifiedMeta.getColOptions(colContext, {
                    column: col,
                });
                const lookupInfo = yield unifiedMeta.getLookupRelatedInfo(colContext, {
                    colOptions: lookupColOption,
                    columns,
                    getMeta,
                });
                const relationColumn = lookupInfo.relationColumn;
                const lookupColumn = lookupInfo.lookupColumn;
                const lookupColumnIdentifierType = yield extractColumnIdentifierType({
                    col: lookupColumn,
                    clientOrSqlUi,
                    columns: yield unifiedMeta.getColumns(unifiedMeta.getContextFromObject(lookupInfo.relatedTable), { model: lookupInfo.relatedTable }),
                    getMeta,
                });
                res.dataType = lookupColumnIdentifierType.dataType;
                res.isDataArray = lookupColumnIdentifierType.isDataArray;
                if (!res.isDataArray) {
                    const relationColOptions = yield unifiedMeta.getColOptions(colContext, {
                        column: relationColumn,
                    });
                    res.isDataArray = ['hm', 'mm'].includes(relationColOptions.type);
                }
                res.referencedColumn = {
                    id: (_c = lookupColumnIdentifierType === null || lookupColumnIdentifierType === void 0 ? void 0 : lookupColumnIdentifierType.referencedColumn) === null || _c === void 0 ? void 0 : _c.id,
                    // if array, we present it as lookup column
                    uidt: (_d = lookupColumnIdentifierType === null || lookupColumnIdentifierType === void 0 ? void 0 : lookupColumnIdentifierType.referencedColumn) === null || _d === void 0 ? void 0 : _d.uidt,
                    intermediaryUidt: UITypes.Lookup,
                    intermediaryId: col.id,
                };
                break;
            }
            case UITypes.LinkToAnotherRecord: {
                const colOptions = yield unifiedMeta.getColOptions(unifiedMeta.getContextFromObject(col), {
                    column: col,
                });
                const relatedTable = yield unifiedMeta.getLTARRelatedTable(unifiedMeta.getContextFromObject(col), {
                    colOptions,
                    getMeta,
                });
                const relatedTableColumns = yield unifiedMeta.getColumns(unifiedMeta.getContextFromObject(relatedTable), {
                    model: relatedTable,
                });
                const relatedTableDisplayColumn = relatedTableColumns.find((col) => col.pv);
                const relatedColumnIdentifierType = yield extractColumnIdentifierType({
                    col: relatedTableDisplayColumn,
                    clientOrSqlUi,
                    columns: relatedTableColumns,
                    getMeta,
                });
                res.dataType = relatedColumnIdentifierType.dataType;
                res.isDataArray =
                    relatedColumnIdentifierType.isDataArray ||
                        ['hm', 'mm'].includes(colOptions.type);
                res.referencedColumn = {
                    id: (_e = relatedColumnIdentifierType === null || relatedColumnIdentifierType === void 0 ? void 0 : relatedColumnIdentifierType.referencedColumn) === null || _e === void 0 ? void 0 : _e.id,
                    uidt: (_f = relatedColumnIdentifierType === null || relatedColumnIdentifierType === void 0 ? void 0 : relatedColumnIdentifierType.referencedColumn) === null || _f === void 0 ? void 0 : _f.uidt,
                    intermediaryUidt: UITypes.LinkToAnotherRecord,
                    intermediaryId: col.id,
                };
                break;
            }
            case UITypes.Formula: {
                const colOptions = yield unifiedMeta.getColOptions(unifiedMeta.getContextFromObject(col), {
                    column: col,
                });
                const parsedTree = yield unifiedMeta.getParsedTree(unifiedMeta.getContextFromObject(col), { colOptions, getMeta });
                // parsedTree may not exists when formula column create / update
                if (parsedTree) {
                    res.isDataArray = parsedTree.isDataArray;
                    res.referencedColumn = parsedTree.referencedColumn;
                }
                break;
            }
            case UITypes.Barcode:
            case UITypes.Button:
            case UITypes.Collaborator:
            case UITypes.QrCode:
            default:
                res.dataType = FormulaDataTypes.UNKNOWN;
                break;
        }
        res.referencedColumn = Object.assign({ id: col.id, uidt: col.uidt }, ((_g = res.referencedColumn) !== null && _g !== void 0 ? _g : {}));
        return res;
    });
}
function handleBinaryExpressionForDateAndTime(params) {
    const { sourceBinaryNode } = params;
    let res;
    if ([FormulaDataTypes.DATE, FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.left.dataType) &&
        [FormulaDataTypes.DATE, FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.right.dataType) &&
        sourceBinaryNode.operator === '-') {
        // when it's interval and interval, we return diff in minute (numeric)
        if ([FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.left.dataType) &&
            [FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.right.dataType)) {
            res = {
                type: JSEPNode.CALL_EXP,
                arguments: [
                    sourceBinaryNode.left,
                    sourceBinaryNode.right,
                    {
                        type: 'Literal',
                        value: 'minutes',
                        raw: '"minutes"',
                        dataType: 'string',
                    },
                ],
                callee: {
                    type: 'Identifier',
                    name: 'DATETIME_DIFF',
                },
                dataType: FormulaDataTypes.NUMERIC,
            };
        }
        // when it's date - date, show the difference in minute
        else if ([FormulaDataTypes.DATE].includes(sourceBinaryNode.left.dataType) &&
            [FormulaDataTypes.DATE].includes(sourceBinaryNode.right.dataType)) {
            res = {
                type: JSEPNode.CALL_EXP,
                arguments: [
                    sourceBinaryNode.left,
                    sourceBinaryNode.right,
                    {
                        type: 'Literal',
                        value: 'minutes',
                        raw: '"minutes"',
                        dataType: 'string',
                    },
                ],
                callee: {
                    type: 'Identifier',
                    name: 'DATETIME_DIFF',
                },
                dataType: FormulaDataTypes.NUMERIC,
            };
        }
        // else interval and date can be addedd seamlessly A - B
        // with result as DATE
        // may be changed if we find other db use case
        else if ([FormulaDataTypes.INTERVAL, FormulaDataTypes.DATE].includes(sourceBinaryNode.left.dataType) &&
            [FormulaDataTypes.INTERVAL, FormulaDataTypes.DATE].includes(sourceBinaryNode.right.dataType) &&
            sourceBinaryNode.left.dataType != sourceBinaryNode.right.dataType) {
            res = {
                type: JSEPNode.BINARY_EXP,
                left: sourceBinaryNode.left,
                right: sourceBinaryNode.right,
                operator: '-',
                dataType: FormulaDataTypes.DATE,
            };
        }
    }
    else if ([FormulaDataTypes.DATE, FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.left.dataType) &&
        [FormulaDataTypes.DATE, FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.right.dataType) &&
        sourceBinaryNode.operator === '+') {
        // when it's interval and interval, we return addition in minute (numeric)
        if ([FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.left.dataType) &&
            [FormulaDataTypes.INTERVAL].includes(sourceBinaryNode.right.dataType)) {
            const left = {
                type: JSEPNode.CALL_EXP,
                arguments: [
                    sourceBinaryNode.left,
                    {
                        type: 'Literal',
                        value: '00:00:00',
                        raw: '"00:00:00"',
                        dataType: FormulaDataTypes.INTERVAL,
                    },
                    {
                        type: 'Literal',
                        value: 'minutes',
                        raw: '"minutes"',
                        dataType: 'string',
                    },
                ],
                callee: {
                    type: 'Identifier',
                    name: 'DATETIME_DIFF',
                },
                dataType: FormulaDataTypes.NUMERIC,
            };
            const right = {
                type: JSEPNode.CALL_EXP,
                arguments: [
                    sourceBinaryNode.right,
                    {
                        type: 'Literal',
                        value: '00:00:00',
                        raw: '"00:00:00"',
                        dataType: FormulaDataTypes.INTERVAL,
                    },
                    {
                        type: 'Literal',
                        value: 'minutes',
                        raw: '"minutes"',
                        dataType: 'string',
                    },
                ],
                callee: {
                    type: 'Identifier',
                    name: 'DATETIME_DIFF',
                },
                dataType: FormulaDataTypes.NUMERIC,
            };
            return {
                type: JSEPNode.BINARY_EXP,
                left,
                right,
                operator: '+',
                dataType: FormulaDataTypes.NUMERIC,
            };
        }
        // else interval and date can be addedd seamlessly A + B
        // with result as DATE
        // may be changed if we find other db use case
        else if ([FormulaDataTypes.INTERVAL, FormulaDataTypes.DATE].includes(sourceBinaryNode.left.dataType) &&
            [FormulaDataTypes.INTERVAL, FormulaDataTypes.DATE].includes(sourceBinaryNode.right.dataType) &&
            sourceBinaryNode.left.dataType != sourceBinaryNode.right.dataType) {
            res = {
                type: JSEPNode.BINARY_EXP,
                left: sourceBinaryNode.left,
                right: sourceBinaryNode.right,
                operator: '+',
                dataType: FormulaDataTypes.DATE,
            };
        }
    }
    return res;
}
function checkForCircularFormulaRef(formulaCol, parsedTree, columns, getMeta) {
    return __awaiter(this, void 0, void 0, function* () {
        // Extract formula references
        const formulaPaths = yield columns.reduce((promiseRes, c) => __awaiter(this, void 0, void 0, function* () {
            const res = yield promiseRes;
            if (c.id !== formulaCol.id && c.uidt === UITypes.Formula) {
                const neighbours = [
                    ...new Set((c.colOptions.formula.match(/cl?_?\w{14,15}/g) || []).filter((colId) => columns.some((col) => col.id === colId && col.uidt === UITypes.Formula))),
                ];
                if (neighbours.length)
                    res.push({ [c.id]: neighbours });
            }
            else if (c.uidt === UITypes.Lookup ||
                c.uidt === UITypes.LinkToAnotherRecord) {
                const neighbours = yield processLookupOrLTARColumn(c);
                if (neighbours === null || neighbours === void 0 ? void 0 : neighbours.length)
                    res.push({ [c.id]: neighbours });
            }
            return res;
        }), Promise.resolve([]));
        function processLookupFormula(col, columns) {
            return __awaiter(this, void 0, void 0, function* () {
                const neighbours = [];
                if (formulaCol.fk_model_id === col.fk_model_id) {
                    return [col.id];
                }
                // Extract columns used in the formula and check for cycles
                const referencedColumns = col.colOptions.formula.match(/cl?_?\w{14,15}/g) || [];
                for (const refColId of referencedColumns) {
                    const refCol = columns.find((c) => c.id === refColId);
                    // if refColId is not valid column, maybe just a concidence value of CONCAT
                    // we ignore
                    if (refCol) {
                        if (refCol.uidt === UITypes.Formula) {
                            neighbours.push(...(yield processLookupFormula(refCol, columns)));
                        }
                        else if (refCol.uidt === UITypes.Lookup ||
                            refCol.uidt === UITypes.LinkToAnotherRecord) {
                            neighbours.push(...(yield processLookupOrLTARColumn(refCol)));
                        }
                    }
                }
                return neighbours;
            });
        }
        // Function to process lookup columns recursively
        function processLookupOrLTARColumn(lookupOrLTARCol) {
            return __awaiter(this, void 0, void 0, function* () {
                var _a;
                const neighbours = [];
                let ltarColumn;
                let lookupFilterFn;
                if (lookupOrLTARCol.uidt === UITypes.Lookup) {
                    const relationColId = lookupOrLTARCol.colOptions
                        .fk_relation_column_id;
                    const lookupColId = lookupOrLTARCol.colOptions
                        .fk_lookup_column_id;
                    ltarColumn = columns.find((c) => c.id === relationColId);
                    lookupFilterFn = (column) => column.id === lookupColId;
                }
                else if (lookupOrLTARCol.uidt === UITypes.LinkToAnotherRecord) {
                    ltarColumn = lookupOrLTARCol;
                    lookupFilterFn = (column) => !!column.pv;
                }
                if (ltarColumn) {
                    const relatedTableMeta = yield getMeta(unifiedMeta.getContextFromObject(Object.assign(Object.assign({}, ltarColumn), { base_id: (_a = ltarColumn.colOptions
                            .fk_related_base_id) !== null && _a !== void 0 ? _a : ltarColumn.base_id })), {
                        id: ltarColumn.colOptions
                            .fk_related_model_id,
                    });
                    const lookupTarget = (yield unifiedMeta.getColumns(unifiedMeta.getContextFromObject(relatedTableMeta), {
                        model: relatedTableMeta,
                    })).find(lookupFilterFn);
                    if (lookupTarget) {
                        if (lookupTarget.uidt === UITypes.Formula) {
                            neighbours.push(...(yield processLookupFormula(lookupTarget, relatedTableMeta.columns)));
                        }
                        else if (lookupTarget.uidt === UITypes.Lookup ||
                            lookupTarget.uidt === UITypes.LinkToAnotherRecord) {
                            neighbours.push(...(yield processLookupOrLTARColumn(lookupTarget)));
                        }
                    }
                }
                return [...new Set(neighbours)];
            });
        }
        // include target formula column (i.e. the one to be saved if applicable)
        const targetFormulaCol = columns.find((c) => c.title === parsedTree.name &&
            [UITypes.Formula, UITypes.LinkToAnotherRecord, UITypes.Lookup].includes(c.uidt));
        if (targetFormulaCol && (formulaCol === null || formulaCol === void 0 ? void 0 : formulaCol.id)) {
            formulaPaths.push({
                [formulaCol === null || formulaCol === void 0 ? void 0 : formulaCol.id]: [targetFormulaCol.id],
            });
        }
        const vertices = formulaPaths.length;
        if (vertices > 0) {
            // perform kahn's algo for cycle detection
            const adj = new Map();
            const inDegrees = new Map();
            // init adjacency list & indegree
            for (const [_, v] of Object.entries(formulaPaths)) {
                const src = Object.keys(v)[0];
                const neighbours = v[src];
                inDegrees.set(src, inDegrees.get(src) || 0);
                for (const neighbour of neighbours) {
                    adj.set(src, (adj.get(src) || new Set()).add(neighbour));
                    inDegrees.set(neighbour, (inDegrees.get(neighbour) || 0) + 1);
                }
            }
            const queue = [];
            // put all vertices with in-degree = 0 (i.e. no incoming edges) to queue
            inDegrees.forEach((inDegree, col) => {
                if (inDegree === 0) {
                    // in-degree = 0 means we start traversing from this node
                    queue.push(col);
                }
            });
            // init count of visited vertices
            let visited = 0;
            // BFS
            while (queue.length !== 0) {
                // remove a vertex from the queue
                const src = queue.shift();
                // if this node has neighbours, increase visited by 1
                const neighbours = adj.get(src) || new Set();
                if (neighbours.size > 0) {
                    visited += 1;
                }
                // iterate each neighbouring nodes
                neighbours.forEach((neighbour) => {
                    // decrease in-degree of its neighbours by 1
                    inDegrees.set(neighbour, inDegrees.get(neighbour) - 1);
                    // if in-degree becomes 0
                    if (inDegrees.get(neighbour) === 0) {
                        // then put the neighboring node to the queue
                        queue.push(neighbour);
                    }
                });
            }
            // vertices not same as visited = cycle found
            if (vertices !== visited) {
                throw new FormulaError(FormulaErrorType.CIRCULAR_REFERENCE, {
                    key: 'msg.formula.cantSaveCircularReference',
                }, 'Circular reference detected');
            }
        }
    });
}
export function validateFormulaAndExtractTreeWithType(_a) {
    return __awaiter(this, arguments, void 0, function* ({ formula, column, columns, clientOrSqlUi, getMeta, trackPosition, }) {
        var _b, _c, _d;
        // extract column list from meta since columns array might not have all columns(system columns)
        const meta = yield getMeta(unifiedMeta.getContextFromObject((_c = (_b = ((column === null || column === void 0 ? void 0 : column.base_id) && column)) !== null && _b !== void 0 ? _b : columns === null || columns === void 0 ? void 0 : columns[0]) !== null && _c !== void 0 ? _c : {}), { id: (column === null || column === void 0 ? void 0 : column.fk_model_id) || ((_d = columns === null || columns === void 0 ? void 0 : columns[0]) === null || _d === void 0 ? void 0 : _d.fk_model_id) || '' });
        const allColumns = (meta === null || meta === void 0 ? void 0 : meta.columns) || columns;
        const sqlUI = typeof clientOrSqlUi === 'string'
            ? SqlUiFactory.create({ client: clientOrSqlUi })
            : clientOrSqlUi;
        const colAliasToColMap = {};
        const colIdToColMap = {};
        for (const col of columns) {
            colAliasToColMap[col.title] = col;
            colIdToColMap[col.id] = col;
        }
        const validateAndExtract = (parsedTree) => __awaiter(this, void 0, void 0, function* () {
            var _a, _b, _c, _d, _e, _f, _g;
            const res = Object.assign({}, parsedTree);
            if (parsedTree.type === JSEPNode.CALL_EXP) {
                const calleeName = parsedTree.callee.name.toUpperCase();
                // validate function name
                if (!formulas[calleeName]) {
                    throw new FormulaError(FormulaErrorType.INVALID_FUNCTION_NAME, {
                        calleeName,
                        position: parsedTree.indexStart >= 0
                            ? {
                                index: parsedTree.indexStart,
                                length: parsedTree.nodeLength,
                            }
                            : undefined,
                    }, `Function ${calleeName} is not available`);
                }
                else if (sqlUI === null || sqlUI === void 0 ? void 0 : sqlUI.getUnsupportedFnList().includes(calleeName)) {
                    throw new FormulaError(FormulaErrorType.INVALID_FUNCTION_NAME, {
                        calleeName,
                        position: parsedTree.indexStart >= 0
                            ? {
                                index: parsedTree.indexStart,
                                length: parsedTree.nodeLength,
                            }
                            : undefined,
                    }, `Function ${calleeName} is unavailable for your database`);
                }
                // validate arguments
                const validation = formulas[calleeName] && formulas[calleeName].validation;
                if (validation && validation.args) {
                    if (validation.args.rqd !== undefined &&
                        validation.args.rqd !== parsedTree.arguments.length) {
                        throw new FormulaError(FormulaErrorType.INVALID_ARG, {
                            key: 'msg.formula.requiredArgumentsFormula',
                            requiredArguments: validation.args.rqd,
                            calleeName,
                            position: parsedTree.indexStart >= 0
                                ? {
                                    index: parsedTree.indexStart,
                                    length: parsedTree.nodeLength,
                                }
                                : undefined,
                        }, 'Required arguments missing');
                    }
                    else if (validation.args.min !== undefined &&
                        validation.args.min > parsedTree.arguments.length) {
                        throw new FormulaError(FormulaErrorType.MIN_ARG, {
                            key: 'msg.formula.minRequiredArgumentsFormula',
                            minRequiredArguments: validation.args.min,
                            calleeName,
                            position: parsedTree.indexStart >= 0
                                ? {
                                    index: parsedTree.indexStart,
                                    length: parsedTree.nodeLength,
                                }
                                : undefined,
                        }, 'Minimum arguments required');
                    }
                    else if (validation.args.max !== undefined &&
                        validation.args.max < parsedTree.arguments.length) {
                        throw new FormulaError(FormulaErrorType.INVALID_ARG, {
                            key: 'msg.formula.maxRequiredArgumentsFormula',
                            maxRequiredArguments: validation.args.max,
                            calleeName,
                            position: parsedTree.indexStart >= 0
                                ? {
                                    index: parsedTree.indexStart,
                                    length: parsedTree.nodeLength,
                                }
                                : undefined,
                        }, 'Maximum arguments missing');
                    }
                }
                // get args type and validate
                const validateResult = (res.arguments =
                    yield Promise.all(parsedTree.arguments.map((arg) => {
                        return validateAndExtract(arg);
                    })));
                const argTypes = validateResult.map((v) => v.dataType);
                // if validation function is present, call it
                if ((_a = formulas[calleeName].validation) === null || _a === void 0 ? void 0 : _a.custom) {
                    (_b = formulas[calleeName].validation) === null || _b === void 0 ? void 0 : _b.custom(argTypes, 
                    // need to use res rather than parsedTree
                    // because post-processing like referencedColumn is needed
                    res, allColumns);
                }
                // validate against expected arg types if present
                else if ((_d = (_c = formulas[calleeName].validation) === null || _c === void 0 ? void 0 : _c.args) === null || _d === void 0 ? void 0 : _d.type) {
                    for (let i = 0; i < validateResult.length; i++) {
                        const argPt = validateResult[i];
                        // if type
                        const expectedArgType = Array.isArray(formulas[calleeName].validation.args.type)
                            ? formulas[calleeName].validation.args.type[i]
                            : formulas[calleeName].validation.args.type;
                        if (argPt.dataType !== expectedArgType &&
                            argPt.dataType !== FormulaDataTypes.NULL &&
                            argPt.dataType !== FormulaDataTypes.UNKNOWN &&
                            expectedArgType !== FormulaDataTypes.STRING) {
                            if (argPt.type === JSEPNode.IDENTIFIER) {
                                const name = ((_e = columns === null || columns === void 0 ? void 0 : columns.find((c) => c.id === argPt.name || c.title === argPt.name)) === null || _e === void 0 ? void 0 : _e.title) || argPt.name;
                                throw new FormulaError(FormulaErrorType.INVALID_ARG, {
                                    key: 'msg.formula.columnWithTypeFoundButExpected',
                                    columnName: name,
                                    columnType: argPt.dataType,
                                    expectedType: expectedArgType,
                                }, `Field ${name} with ${argPt.dataType} type is found but ${expectedArgType} type is expected`);
                            }
                            else {
                                let key = '';
                                const position = i + 1;
                                let type = '';
                                if (expectedArgType === FormulaDataTypes.NUMERIC) {
                                    key = 'msg.formula.typeIsExpected';
                                    type = 'numeric';
                                }
                                else if (expectedArgType === FormulaDataTypes.BOOLEAN) {
                                    key = 'msg.formula.typeIsExpected';
                                    type = 'boolean';
                                }
                                else if (expectedArgType === FormulaDataTypes.DATE) {
                                    key = 'msg.formula.typeIsExpected';
                                    type = 'date';
                                }
                                throw new FormulaError(FormulaErrorType.INVALID_ARG, {
                                    type,
                                    key,
                                    position,
                                    calleeName,
                                }, `${calleeName === null || calleeName === void 0 ? void 0 : calleeName.toUpperCase()} requires a ${type || expectedArgType} at position ${position}`);
                            }
                        }
                        // if expected type is string and arg type is not string, then cast it to string
                        if (expectedArgType === FormulaDataTypes.STRING &&
                            expectedArgType !== argPt.dataType) {
                            argPt.cast = FormulaDataTypes.STRING;
                        }
                    }
                }
                if (typeof formulas[calleeName].returnType === 'function') {
                    res.dataType = (_g = (_f = formulas[calleeName]).returnType) === null || _g === void 0 ? void 0 : _g.call(_f, argTypes);
                }
                else if (formulas[calleeName].returnType) {
                    res.dataType = formulas[calleeName].returnType;
                }
                if (calleeName.toUpperCase().startsWith('ARRAY')) {
                    res.inArrayFormat = true;
                }
                Object.assign(res, extractCallExpressionReferencedInfo({
                    parsedTree: res,
                }));
            }
            else if (parsedTree.type === JSEPNode.IDENTIFIER) {
                const identifierName = parsedTree.name;
                const col = (colIdToColMap[identifierName] ||
                    colAliasToColMap[identifierName]);
                if (!col) {
                    if (formulas[identifierName]) {
                        throw new FormulaError(FormulaErrorType.INVALID_SYNTAX, {
                            key: 'msg.formula.formulaMissingParentheses',
                            calleeName: identifierName,
                            position: parsedTree.indexStart >= 0
                                ? {
                                    index: parsedTree.indexEnd,
                                    length: 1,
                                }
                                : undefined,
                        }, `Missing parentheses after function name "${JSON.stringify(identifierName)}"`);
                    }
                    throw new FormulaError(FormulaErrorType.INVALID_COLUMN, {
                        key: 'msg.formula.columnNotAvailable',
                        columnName: identifierName,
                        position: parsedTree.indexStart >= 0
                            ? {
                                index: parsedTree.indexStart,
                                length: parsedTree.nodeLength,
                            }
                            : undefined,
                    }, `Invalid column name/id ${JSON.stringify(identifierName)} in formula`);
                }
                res.name = col.id;
                if ((col === null || col === void 0 ? void 0 : col.uidt) === UITypes.Formula) {
                    if (column) {
                        // check for circular reference when column is present(only available when calling root formula)
                        yield checkForCircularFormulaRef(column, parsedTree, columns, getMeta);
                    }
                    const formulaRes = col.colOptions.parsed_tree ||
                        (yield validateFormulaAndExtractTreeWithType(
                        // formula may include double curly brackets in previous version
                        // convert to single curly bracket here for compatibility
                        {
                            formula: col.colOptions.formula
                                .replaceAll('{{', '{')
                                .replaceAll('}}', '}'),
                            columns,
                            clientOrSqlUi,
                            getMeta,
                        }));
                    res.dataType = formulaRes === null || formulaRes === void 0 ? void 0 : formulaRes.dataType;
                    res.inArrayFormat = formulaRes.inArrayFormat;
                }
                else {
                    if ((col === null || col === void 0 ? void 0 : col.uidt) === UITypes.Lookup ||
                        (col === null || col === void 0 ? void 0 : col.uidt) === UITypes.LinkToAnotherRecord) {
                        // check for circular reference when column is present(only available when calling root formula)
                        if (column) {
                            yield checkForCircularFormulaRef(column, parsedTree, columns, getMeta);
                        }
                    }
                }
                // extract type and add to res
                Object.assign(res, yield extractColumnIdentifierType({
                    col,
                    columns,
                    getMeta,
                    clientOrSqlUi,
                }));
            }
            else if (parsedTree.type === JSEPNode.LITERAL) {
                if (typeof parsedTree.value === 'number') {
                    res.dataType = FormulaDataTypes.NUMERIC;
                }
                else if (typeof parsedTree.value === 'string') {
                    res.dataType = FormulaDataTypes.STRING;
                }
                else if (typeof parsedTree.value === 'boolean') {
                    res.dataType = FormulaDataTypes.BOOLEAN;
                }
                else {
                    res.dataType = FormulaDataTypes.STRING;
                }
            }
            else if (parsedTree.type === JSEPNode.UNARY_EXP) {
                // only support -ve values
                if (['-'].includes(parsedTree.operator) &&
                    parsedTree.argument.type === JSEPNode.LITERAL &&
                    typeof parsedTree.argument.value === 'number') {
                    res.dataType = FormulaDataTypes.NUMERIC;
                }
                else {
                    throw new FormulaError(FormulaErrorType.NOT_SUPPORTED, {}, `Unary expression '${parsedTree.operator}' is not supported`);
                }
            }
            else if (parsedTree.type === JSEPNode.BINARY_EXP) {
                const argsLeft = yield validateAndExtract(parsedTree.left);
                const argsRight = yield validateAndExtract(parsedTree.right);
                res.left = argsLeft;
                res.right = argsRight;
                const dateAndTimeParsedNode = handleBinaryExpressionForDateAndTime({
                    sourceBinaryNode: res,
                });
                if (dateAndTimeParsedNode) {
                    Object.assign(res, handleBinaryExpressionForDateAndTime({ sourceBinaryNode: res }));
                    if (res.type !== JSEPNode.BINARY_EXP) {
                        res.left = undefined;
                        res.right = undefined;
                        res.operator = undefined;
                    }
                }
                else if (['==', '<', '>', '<=', '>=', '!='].includes(parsedTree.operator)) {
                    res.dataType = FormulaDataTypes.COND_EXP;
                }
                else if (parsedTree.operator === '+') {
                    res.dataType = FormulaDataTypes.NUMERIC;
                    // if any side is string/date/other type, then the result will be concatenated string
                    // e.g. 1 + '2' = '12'
                    if ([
                        res.left,
                        res.right,
                    ].some((r) => ![
                        FormulaDataTypes.NUMERIC,
                        FormulaDataTypes.BOOLEAN,
                        FormulaDataTypes.NULL,
                        FormulaDataTypes.UNKNOWN,
                    ].includes(r.dataType))) {
                        res.dataType = FormulaDataTypes.STRING;
                    }
                }
                else if (['&'].includes(parsedTree.operator)) {
                    res.dataType = FormulaDataTypes.STRING;
                }
                else {
                    res.dataType = FormulaDataTypes.NUMERIC;
                }
                Object.assign(res, extractBinaryExpReferencedInfo({
                    parsedTree: res,
                    left: argsLeft,
                    right: argsRight,
                }));
            }
            else if (parsedTree.type === JSEPNode.MEMBER_EXP) {
                throw new FormulaError(FormulaErrorType.NOT_SUPPORTED, {}, 'Bracket notation is not supported');
            }
            else if (parsedTree.type === JSEPNode.ARRAY_EXP) {
                throw new FormulaError(FormulaErrorType.NOT_SUPPORTED, {}, 'Array is not supported');
            }
            else if (parsedTree.type === JSEPNode.COMPOUND) {
                throw new FormulaError(FormulaErrorType.NOT_SUPPORTED, {}, 'Compound statement is not supported');
            }
            return res;
        });
        try {
            const jsepInstance = trackPosition ? formulaJsepWithIndex : formulaJsep;
            const parsedFormula = jsepInstance(formula);
            // TODO: better jsep expression handling
            const result = yield validateAndExtract(parsedFormula);
            return result;
        }
        catch (ex) {
            if (trackPosition) {
                handleFormulaError({ formula, error: ex });
            }
            throw ex;
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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