var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { validateFormulaAndExtractTreeWithType } from './validate-extract-tree';
import { FormulaDataTypes, FormulaErrorType, JSEPNode } from './enums';
import { FormulaError } from './error';
import UITypes from '../UITypes';
import { SqlUiFactory } from '../../lib/sqlUi';
// Mock dependencies
const mockColumns = [
    {
        id: 'col1',
        title: 'Column1',
        uidt: UITypes.SingleLineText,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
    },
    {
        id: 'col2',
        title: 'Column2',
        uidt: UITypes.Number,
        dt: 'int',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
    },
    {
        id: 'col3',
        title: 'Column3',
        uidt: UITypes.Date,
        dt: 'date',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
    },
    {
        id: 'col4',
        title: 'Column4',
        uidt: UITypes.Formula,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            formula: '{Column1}',
            formula_raw: '{Column1}',
            fk_column_id: 'col4',
            error: null,
            parsed_tree: {
                type: JSEPNode.IDENTIFIER,
                name: 'col1',
                dataType: FormulaDataTypes.STRING,
                referencedColumn: { id: 'col1', uidt: UITypes.SingleLineText },
            },
            getParsedTree: () => ({
                type: JSEPNode.IDENTIFIER,
                name: 'col1',
                dataType: FormulaDataTypes.STRING,
                referencedColumn: { id: 'col1', uidt: UITypes.SingleLineText },
            }),
        },
    },
    {
        id: 'col5',
        title: 'Column5',
        uidt: UITypes.Lookup,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            fk_relation_column_id: 'col6',
            fk_lookup_column_id: 'col7',
            fk_column_id: 'col5',
        },
    },
    {
        id: 'col6',
        title: 'Column6',
        uidt: UITypes.LinkToAnotherRecord,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            fk_related_model_id: 'model2',
            type: 'oo',
            fk_column_id: 'col6',
            id: 'col6',
        },
    },
    {
        id: 'col7',
        title: 'Column7',
        uidt: UITypes.SingleLineText,
        dt: 'varchar',
        pv: true,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model2',
        deleted: false,
    },
    {
        id: 'col8',
        title: 'Column8',
        uidt: UITypes.Checkbox,
        dt: 'boolean',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
    },
    {
        id: 'col9',
        title: 'Column9',
        uidt: UITypes.Time,
        dt: 'time',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
    },
    {
        id: 'col10',
        title: 'Column10',
        uidt: UITypes.Formula,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            formula: '{Column4}', // Circular reference: Column10 -> Column4 -> Column1
            formula_raw: '{Column4}',
            fk_column_id: 'col10',
            error: null,
            parsed_tree: {
                type: JSEPNode.IDENTIFIER,
                name: 'col4',
                dataType: FormulaDataTypes.STRING,
                referencedColumn: { id: 'col4', uidt: UITypes.Formula },
            },
            getParsedTree: () => ({
                type: JSEPNode.IDENTIFIER,
                name: 'col4',
                dataType: FormulaDataTypes.STRING,
                referencedColumn: { id: 'col4', uidt: UITypes.Formula },
            }),
        },
    },
    {
        id: 'cdYAQE3SFa1F4ys',
        title: 'Column11',
        uidt: UITypes.Formula,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            formula: '{c3hOMA9YnfkD4WW}',
            formula_raw: '{Column12}',
            fk_column_id: 'cdYAQE3SFa1F4ys',
            error: null,
            parsed_tree: {
                type: JSEPNode.IDENTIFIER,
                name: 'c3hOMA9YnfkD4WW',
                dataType: FormulaDataTypes.STRING,
                referencedColumn: { id: 'c3hOMA9YnfkD4WW', uidt: UITypes.Formula },
            },
            getParsedTree: () => ({
                type: JSEPNode.IDENTIFIER,
                name: 'c3hOMA9YnfkD4WW',
                dataType: FormulaDataTypes.STRING,
                referencedColumn: { id: 'c3hOMA9YnfkD4WW', uidt: UITypes.Formula },
            }),
        },
    },
    {
        id: 'c3hOMA9YnfkD4WW',
        title: 'Column12',
        uidt: UITypes.Formula,
        dt: 'varchar',
        pv: false,
        base_id: 'base1',
        fk_workspace_id: 'ws1',
        fk_model_id: 'model1',
        deleted: false,
        colOptions: {
            formula: '{cdYAQE3SFa1F4ys}',
            formula_raw: '{Column11}',
            fk_column_id: 'col12',
            error: null,
            parsed_tree: {
                type: JSEPNode.IDENTIFIER,
                name: 'cdYAQE3SFa1F4ys',
                dataType: FormulaDataTypes.STRING,
                referencedColumn: { id: 'cdYAQE3SFa1F4ys', uidt: UITypes.Formula },
            },
            getParsedTree: () => ({
                type: JSEPNode.IDENTIFIER,
                name: 'cdYAQE3SFa1F4ys',
                dataType: FormulaDataTypes.STRING,
                referencedColumn: { id: 'cdYAQE3SFa1F4ys', uidt: UITypes.Formula },
            }),
        },
    },
    // case Function to rollup
    {
        id: 'ciro4tf7xfab6pj',
        title: 'Max(Number) from Table-L1',
        uidt: UITypes.SingleLineText,
        base_id: 'base1',
    },
];
const mockGetMeta = jest.fn((_context, options) => __awaiter(void 0, void 0, void 0, function* () {
    if (options.id === 'model2') {
        return {
            id: 'model2',
            title: 'Model2',
            base_id: 'base1',
            columns: [
                {
                    id: 'col7',
                    title: 'Column7',
                    uidt: UITypes.SingleLineText,
                    dt: 'varchar',
                    pv: true,
                    base_id: 'base1',
                    fk_workspace_id: 'ws1',
                    fk_model_id: 'model2',
                    deleted: false,
                },
            ],
        };
    }
    return {
        id: 'model1',
        title: 'Model1',
        base_id: 'base1',
        columns: mockColumns,
    };
}));
const mockClientOrSqlUi = SqlUiFactory.create({ client: 'pg' });
const caseIdentifierRollup = {
    column: {
        id: 'c4o52mufchcfgi8',
        source_id: 'bv4r9c7t9enq9vs',
        base_id: 'p4zgm45rxfc6izi',
        fk_model_id: 'mroakgv13nikn55',
        title: 'Formula',
        column_name: 'Formula',
        uidt: 'Formula',
        system: false,
        order: 10,
        meta: {
            display_column_meta: {
                meta: {},
                custom: {},
            },
            defaultViewColOrder: 10,
            defaultViewColVisibility: true,
        },
        readonly: false,
        fk_workspace_id: 'w5x8zatd',
        colOptions: {
            parsed_tree: {
                type: 'Identifier',
                name: 'cpqnd2bdmzg00x1',
                raw: '{{cpqnd2bdmzg00x1}}',
                dataType: 'string',
                isDataArray: true,
                referencedColumn: {
                    id: 'czb6z4e94l7kiyr',
                    uidt: 'SingleLineText',
                    intermediaryUidt: 'LinkToAnotherRecord',
                    intermediaryId: 'cpqnd2bdmzg00x1',
                },
            },
            id: 'fhl7s1abjhwj6e0',
            fk_column_id: 'c4o52mufchcfgi8',
            formula: '{{cpqnd2bdmzg00x1}}',
            formula_raw: '{Table-L1}',
            base_id: 'p4zgm45rxfc6izi',
            fk_workspace_id: 'w5x8zatd',
        },
        extra: {
            display_type: 'SingleLineText',
        },
    },
    columns: [
        {
            id: 'cizch3j3b3inbiv',
            source_id: 'bv4r9c7t9enq9vs',
            base_id: 'p4zgm45rxfc6izi',
            fk_model_id: 'mroakgv13nikn55',
            title: 'Id',
            column_name: 'id',
            uidt: 'ID',
            order: 1,
            meta: {
                defaultViewColOrder: 2,
                defaultViewColVisibility: true,
            },
            readonly: false,
            fk_workspace_id: 'w5x8zatd',
        },
        {
            id: 'cpxst4a7s3wf3pb',
            source_id: 'bv4r9c7t9enq9vs',
            base_id: 'p4zgm45rxfc6izi',
            fk_model_id: 'mroakgv13nikn55',
            title: 'Title',
            column_name: 'title',
            uidt: 'SingleLineText',
            pv: true,
            system: false,
            order: 7,
            meta: {
                defaultViewColOrder: 1,
                defaultViewColVisibility: true,
            },
            readonly: false,
            fk_workspace_id: 'w5x8zatd',
            extra: {},
        },
        {
            id: 'cpqnd2bdmzg00x1',
            source_id: 'bv4r9c7t9enq9vs',
            base_id: 'p4zgm45rxfc6izi',
            fk_model_id: 'mroakgv13nikn55',
            title: 'Table-L1',
            uidt: 'LinkToAnotherRecord',
            virtual: true,
            order: 8,
            meta: {
                plural: 'Table-L1s',
                singular: 'Table-L1',
                custom: false,
                defaultViewColOrder: 8,
                defaultViewColVisibility: true,
            },
            readonly: false,
            fk_workspace_id: 'w5x8zatd',
            colOptions: {
                virtual: true,
                id: 'lurlzpshvoroi9a',
                type: 'hm',
                fk_column_id: 'cpqnd2bdmzg00x1',
                fk_related_model_id: 'mr6xbpmnuuusa9y',
                fk_child_column_id: 'c1tmwp5y0mx5036',
                fk_parent_column_id: 'cizch3j3b3inbiv',
                base_id: 'p4zgm45rxfc6izi',
                fk_workspace_id: 'w5x8zatd',
            },
            extra: {},
        },
        {
            id: 'ccuxnqvpoueby3r',
            source_id: 'bv4r9c7t9enq9vs',
            base_id: 'p4zgm45rxfc6izi',
            fk_model_id: 'mroakgv13nikn55',
            title: 'Max(Number) from Table-L1',
            column_name: 'Max(Number) from Table-L1',
            uidt: 'Rollup',
            system: false,
            order: 9,
            meta: {
                precision: 0,
                isLocaleString: false,
                defaultViewColOrder: 9,
                defaultViewColVisibility: true,
            },
            readonly: false,
            fk_workspace_id: 'w5x8zatd',
            colOptions: {
                id: 'rl6o2v6aesvywmry',
                fk_column_id: 'ccuxnqvpoueby3r',
                fk_relation_column_id: 'cpqnd2bdmzg00x1',
                fk_rollup_column_id: 'ciro4tf7xfab6pj',
                rollup_function: 'max',
                base_id: 'p4zgm45rxfc6izi',
                fk_workspace_id: 'w5x8zatd',
            },
            extra: {},
        },
    ],
    formula: '{Max(Number) from Table-L1}',
};
describe('validateFormulaAndExtractTreeWithType', () => {
    beforeEach(() => {
        jest.clearAllMocks();
    });
    // Test cases for function name validation
    it('should throw INVALID_FUNCTION_NAME for unknown function', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'UNKNOWN_FUNC()',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'UNKNOWN_FUNC()',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.INVALID_FUNCTION_NAME);
    }));
    it('should throw INVALID_FUNCTION_NAME for unsupported function by database', () => __awaiter(void 0, void 0, void 0, function* () {
        const mocked = jest
            .spyOn(mockClientOrSqlUi, 'getUnsupportedFnList')
            .mockReturnValue(['UPPER']);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'UPPER("test")',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'UPPER("test")',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.INVALID_FUNCTION_NAME);
        mocked.mockRestore();
    }));
    // Test cases for argument validation
    it('should throw INVALID_ARG for missing required arguments', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'CONCAT()', // CONCAT requires at least 1 argument
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'CONCAT()',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.MIN_ARG);
    }));
    it('should throw INVALID_ARG for too many arguments', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'UPPER("a", "b")', // NOT takes exactly 1 argument
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'UPPER("a", "b")',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.INVALID_ARG);
    }));
    it('should throw INVALID_ARG for incorrect argument type (identifier)', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'MIN({Column1},{Column2})', // Column2 is numeric, UPPER expects string
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'MIN({Column1},{Column2})', // Column2 is numeric, UPPER expects string
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.INVALID_ARG);
    }));
    it('should throw INVALID_ARG for incorrect argument type (literal)', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'MIN("day", 1)',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'MIN("day", 1)',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.INVALID_ARG);
    }));
    // Test cases for identifier resolution
    it('should resolve column by title', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column1}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col1');
        expect(result.dataType).toBe(FormulaDataTypes.STRING);
    }));
    it('should resolve column by id', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{col1}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col1');
        expect(result.dataType).toBe(FormulaDataTypes.STRING);
    }));
    it('should throw INVALID_COLUMN for unknown column', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: '{UNKNOWN_COL}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: '{UNKNOWN_COL}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.INVALID_COLUMN);
    }));
    it('should handle formula column correctly', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column4}', // Column4 is a formula column that references Column1
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col4');
        expect(result.dataType).toBe(FormulaDataTypes.STRING);
    }));
    it('should handle lookup column correctly', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column5}', // Column5 is a lookup column
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col5');
        expect(result.dataType).toBe(FormulaDataTypes.STRING);
    }));
    it('should handle LTAR column correctly', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column6}', // Column6 is an LTAR column
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col6');
        expect(result.dataType).toBe(FormulaDataTypes.STRING);
    }));
    it('should detect circular reference in formula columns', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: '{Column11}',
            column: mockColumns.find((c) => c.id === 'c3hOMA9YnfkD4WW'), // col12 references col11, col11 references col12
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: '{Column11}',
            column: mockColumns.find((c) => c.id === 'c3hOMA9YnfkD4WW'),
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.CIRCULAR_REFERENCE);
    }));
    // Test cases for literal handling
    it('should identify numeric literal', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '123',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.LITERAL);
        expect(result.dataType).toBe(FormulaDataTypes.NUMERIC);
    }));
    it('should identify string literal', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '"hello"',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.LITERAL);
        expect(result.dataType).toBe(FormulaDataTypes.STRING);
    }));
    it('should identify boolean CALL_EXP', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: 'ISBLANK("")',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.CALL_EXP);
        expect(result.dataType).toBe(FormulaDataTypes.BOOLEAN);
    }));
    // Test cases for unary expression
    it('should handle negative numeric literal', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '-123',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.UNARY_EXP);
        expect(result.dataType).toBe(FormulaDataTypes.NUMERIC);
    }));
    it('should throw NOT_SUPPORTED for unsupported unary expression', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: '!TRUE', // NOT is a function, not a unary operator in this context
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: '!TRUE',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.NOT_SUPPORTED);
    }));
    // Test cases for binary expression
    it('should handle date subtraction (DATE - DATE)', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column3} - {Column3}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.CALL_EXP);
        expect(result.callee.name).toBe('DATETIME_DIFF');
        expect(result.dataType).toBe(FormulaDataTypes.NUMERIC);
    }));
    it('should handle time subtraction (INTERVAL - INTERVAL)', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column9} - {Column9}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.CALL_EXP);
        expect(result.callee.name).toBe('DATETIME_DIFF');
        expect(result.dataType).toBe(FormulaDataTypes.NUMERIC);
    }));
    it('should handle date and interval subtraction (DATE - INTERVAL)', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column3} - {Column9}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('-');
        expect(result.dataType).toBe(FormulaDataTypes.DATE);
    }));
    it('should handle date and interval addition (DATE + INTERVAL)', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column3} + {Column9}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('+');
        expect(result.dataType).toBe(FormulaDataTypes.DATE);
    }));
    it('should handle interval addition (INTERVAL + INTERVAL)', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column9} + {Column9}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('+');
        expect(result.dataType).toBe(FormulaDataTypes.NUMERIC);
    }));
    it('should handle comparison operators', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column2} > 10',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('>');
        expect(result.dataType).toBe(FormulaDataTypes.COND_EXP);
    }));
    it('should handle arithmetic operators', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column2} * 2',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('*');
        expect(result.dataType).toBe(FormulaDataTypes.NUMERIC);
    }));
    it('should handle string concatenation with &', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column1} & " suffix"',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('&');
        expect(result.dataType).toBe(FormulaDataTypes.STRING);
    }));
    it('should handle string concatenation with + if one operand is string', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column2} + " suffix"',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.BINARY_EXP);
        expect(result.operator).toBe('+');
        expect(result.dataType).toBe(FormulaDataTypes.STRING);
    }));
    // Test cases for unsupported nodes
    it('should throw NOT_SUPPORTED for member expression', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'obj.property',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'obj.property',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.NOT_SUPPORTED);
    }));
    it('should throw NOT_SUPPORTED for array expression', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: '[1, 2, 3]',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: '[1, 2, 3]',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.NOT_SUPPORTED);
    }));
    it('should throw NOT_SUPPORTED for compound statement', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: '1; 2',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: '1; 2',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.NOT_SUPPORTED);
    }));
    it('should handle nested formulas correctly', () => __awaiter(void 0, void 0, void 0, function* () {
        const formulaCol10 = mockColumns.find((c) => c.id === 'col10');
        formulaCol10.colOptions.formula =
            '{Column4}'; // col10 -> col4 -> col1
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: '{Column10}',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.IDENTIFIER);
        expect(result.name).toBe('col10');
        expect(result.dataType).toBe(FormulaDataTypes.STRING);
    }));
    it('should cast non-string arguments to string if expected type is string', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: 'CONCAT({Column1}, {Column2})', // Column2 is numeric, should be cast to string
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.type).toBe(JSEPNode.CALL_EXP);
        expect(result.callee.name).toBe('CONCAT');
        expect(result.arguments[1].dataType).toBe(FormulaDataTypes.NUMERIC);
    }));
    it('should throw INVALID_SYNTAX for missing parentheses after function name', () => __awaiter(void 0, void 0, void 0, function* () {
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'UPPER',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toThrow(FormulaError);
        yield expect(validateFormulaAndExtractTreeWithType({
            formula: 'UPPER',
            columns: mockColumns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        })).rejects.toHaveProperty('type', FormulaErrorType.INVALID_SYNTAX);
    }));
    it('should return correct when identifier to rollup', () => __awaiter(void 0, void 0, void 0, function* () {
        const result = yield validateFormulaAndExtractTreeWithType({
            formula: caseIdentifierRollup.formula,
            columns: caseIdentifierRollup.columns,
            clientOrSqlUi: mockClientOrSqlUi,
            getMeta: mockGetMeta,
        });
        expect(result.referencedColumn.uidt).toBe('SingleLineText');
    }));
});
//# sourceMappingURL=data:application/json;base64,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