import UITypes, { isLinksOrLTAR, isNumericCol } from './UITypes';
import { RelationTypes } from './globals';
import { ClientType } from './enums';
import { IntegrationsType, } from './Api';
import { FormulaDataTypes } from './formula/enums';
import { ncIsNull, ncIsUndefined } from '../lib/is';
// import {RelationTypes} from "./globals";
const filterOutSystemColumns = (columns) => {
    return (columns && columns.filter((c) => !isSystemColumn(c))) || [];
};
const getSystemColumnsIds = (columns) => {
    return ((columns && columns.filter(isSystemColumn)) || []).map((c) => c.id);
};
const getSystemColumns = (columns) => columns.filter(isSystemColumn) || [];
const isSystemColumn = (col) => !!(col &&
    (col.uidt === UITypes.ForeignKey ||
        (col.pk && (col.ai || col.cdf)) ||
        (col.pk && col.meta && col.meta.ag) ||
        col.system));
const isSelfReferencingTableColumn = (col) => {
    var _a, _b, _c;
    return (col &&
        (col.uidt === UITypes.Links || col.uidt === UITypes.LinkToAnotherRecord) &&
        ((col === null || col === void 0 ? void 0 : col.fk_model_id) || ((_a = col === null || col === void 0 ? void 0 : col.colOptions) === null || _a === void 0 ? void 0 : _a.fk_model_id)) &&
        ((_b = col === null || col === void 0 ? void 0 : col.colOptions) === null || _b === void 0 ? void 0 : _b.fk_related_model_id) &&
        ((col === null || col === void 0 ? void 0 : col.fk_model_id) || ((_c = col === null || col === void 0 ? void 0 : col.colOptions) === null || _c === void 0 ? void 0 : _c.fk_model_id)) ===
            col.colOptions.fk_related_model_id);
};
const extractRolesObj = (roles) => {
    if (!roles)
        return null;
    if (typeof roles === 'object' && !Array.isArray(roles))
        return roles;
    if (typeof roles === 'string') {
        roles = roles.split(',');
    }
    if (roles.length === 0)
        return null;
    return roles.reduce((acc, role) => {
        acc[role] = true;
        return acc;
    }, {});
};
const stringifyRolesObj = (roles) => {
    if (!roles)
        return '';
    const rolesArr = Object.keys(roles).filter((r) => roles[r]);
    return rolesArr.join(',');
};
const getAvailableRollupForColumn = (column) => {
    var _a, _b;
    if ([UITypes.Formula].includes(column.uidt)) {
        return getAvailableRollupForFormulaType((_b = (_a = column.colOptions.parsed_tree) === null || _a === void 0 ? void 0 : _a.dataType) !== null && _b !== void 0 ? _b : FormulaDataTypes.UNKNOWN);
    }
    else {
        return getAvailableRollupForUiType(column.uidt);
    }
};
const getAvailableRollupForUiType = (type) => {
    if ([
        UITypes.Year,
        UITypes.Time,
        UITypes.Date,
        UITypes.DateTime,
        UITypes.CreatedTime,
        UITypes.LastModifiedTime,
    ].includes(type)) {
        return ['count', 'min', 'max', 'countDistinct'];
    }
    if (isNumericCol(type)) {
        // Number, Currency, Percent, Duration, Rating, Decimal
        return [
            'sum',
            'count',
            'min',
            'max',
            'avg',
            'countDistinct',
            'sumDistinct',
            'avgDistinct',
        ];
    }
    if ([
        UITypes.SingleLineText,
        UITypes.LongText,
        UITypes.User,
        UITypes.Email,
        UITypes.PhoneNumber,
        UITypes.URL,
        UITypes.JSON,
    ].includes(type)) {
        return ['count', 'countDistinct'];
    }
    if ([UITypes.Checkbox].includes(type)) {
        return ['count', 'sum'];
    }
    if ([UITypes.Attachment].includes(type)) {
        return [];
    }
    if ([UITypes.SingleSelect, UITypes.MultiSelect].includes(type)) {
        return ['count', 'countDistinct'];
    }
    return [
        'sum',
        'count',
        'min',
        'max',
        'avg',
        'countDistinct',
        'sumDistinct',
        'avgDistinct',
    ];
};
const getAvailableRollupForFormulaType = (type) => {
    switch (type) {
        case FormulaDataTypes.DATE:
        case FormulaDataTypes.INTERVAL: {
            return ['count', 'min', 'max', 'countDistinct'];
        }
        case FormulaDataTypes.NUMERIC: {
            return [
                'sum',
                'count',
                'min',
                'max',
                'avg',
                'countDistinct',
                'sumDistinct',
                'avgDistinct',
            ];
        }
        case FormulaDataTypes.BOOLEAN: {
            return ['count', 'sum'];
        }
        case FormulaDataTypes.STRING: {
            return ['count', 'countDistinct'];
        }
        case FormulaDataTypes.UNKNOWN:
        default: {
            return ['count'];
        }
    }
};
const getRenderAsTextFunForUiType = (type) => {
    if ([
        UITypes.Year,
        UITypes.Time,
        UITypes.Date,
        UITypes.DateTime,
        UITypes.CreatedTime,
        UITypes.LastModifiedTime,
        UITypes.Currency,
        UITypes.Duration,
    ].includes(type)) {
        return ['count', 'countDistinct'];
    }
    return [
        'sum',
        'count',
        'avg',
        'min',
        'max',
        'countDistinct',
        'sumDistinct',
        'avgDistinct',
    ];
};
const getFileName = ({ name, count, ext }) => `${name}${count ? `(${count})` : ''}${ext ? `${ext}` : ''}`;
// add count before extension if duplicate name found
function populateUniqueFileName(fileName, attachments) {
    return fileName.replace(/^(.+?)(?:\((\d+)\))?(\.(?:tar|min)\.(?:\w{2,4})|\.\w+)$/, (fileName, name, count, ext) => {
        let genFileName = fileName;
        let c = count || 1;
        // iterate until a unique name
        while (attachments.some((fn) => fn === genFileName)) {
            genFileName = getFileName({
                name,
                ext,
                count: c++,
            });
        }
        return genFileName;
    });
}
function roundUpToPrecision(number, precision = 0) {
    precision =
        precision == null
            ? 0
            : precision >= 0
                ? Math.min(precision, 292)
                : Math.max(precision, -292);
    if (precision) {
        // Shift with exponential notation to avoid floating-point issues.
        // See [MDN](https://mdn.io/round#Examples) for more details.
        let pair = `${number}e`.split('e');
        const value = Math.round(Number(`${pair[0]}e${+pair[1] + precision}`));
        pair = `${value}e`.split('e');
        return (+`${pair[0]}e${+pair[1] - precision}`).toFixed(precision);
    }
    return Math.round(number).toFixed(precision);
}
export { filterOutSystemColumns, getSystemColumnsIds, getSystemColumns, isSystemColumn, isSelfReferencingTableColumn, extractRolesObj, stringifyRolesObj, getAvailableRollupForColumn, getAvailableRollupForUiType, getAvailableRollupForFormulaType, getRenderAsTextFunForUiType, populateUniqueFileName, roundUpToPrecision, };
const testDataBaseNames = {
    [ClientType.MYSQL]: null,
    mysql: null,
    [ClientType.PG]: 'postgres',
    oracledb: 'xe',
    [ClientType.SQLITE]: 'a.sqlite',
};
export const getTestDatabaseName = (db) => {
    var _a;
    if (db.client === ClientType.PG || db.client === ClientType.SNOWFLAKE)
        return (_a = db.connection) === null || _a === void 0 ? void 0 : _a.database;
    return testDataBaseNames[db.client];
};
export const integrationCategoryNeedDefault = (category) => {
    return [IntegrationsType.Ai].includes(category);
};
export function parseProp(v, fallbackVal = {}) {
    var _a;
    if (ncIsUndefined(v) || ncIsNull(v))
        return {};
    try {
        return typeof v === 'string' ? (_a = JSON.parse(v)) !== null && _a !== void 0 ? _a : fallbackVal : v;
    }
    catch (_b) {
        return fallbackVal;
    }
}
export function stringifyProp(v, fallbackVal = '{}') {
    var _a;
    if (ncIsUndefined(v) || ncIsNull(v))
        return '{}';
    try {
        return typeof v === 'string' ? v : (_a = JSON.stringify(v)) !== null && _a !== void 0 ? _a : fallbackVal;
    }
    catch (_b) {
        return fallbackVal;
    }
}
export function parseHelper(v) {
    try {
        return typeof v === 'string' ? JSON.parse(v) : v;
    }
    catch (_a) {
        return v;
    }
}
export function stringifyHelper(v) {
    try {
        return typeof v === 'string' ? v : JSON.stringify(v);
    }
    catch (_a) {
        return v;
    }
}
export function toSafeInteger(value) {
    return Math.max(Number.MIN_SAFE_INTEGER, Math.min(value, Number.MAX_SAFE_INTEGER));
}
export function isCrossBaseLink(col) {
    var _a, _b, _c;
    return (col &&
        isLinksOrLTAR(col) &&
        ((_a = col.colOptions) === null || _a === void 0 ? void 0 : _a.fk_related_base_id) &&
        ((_b = col.colOptions) === null || _b === void 0 ? void 0 : _b.fk_related_base_id) !==
            ((_c = col.colOptions) === null || _c === void 0 ? void 0 : _c.base_id));
}
export function lookupCanHaveRecursiveEvaluation(param) {
    const { isEeUI, dbClientType, relationType, relationCol } = param;
    return (isEeUI &&
        dbClientType === ClientType.PG &&
        isSelfReferencingTableColumn(relationCol) &&
        [RelationTypes.HAS_MANY, RelationTypes.BELONGS_TO].includes(relationType));
}
export function formatBytes(bytes, decimals = 2, base = 1000) {
    if (bytes === 0)
        return '0 Bytes';
    const k = base;
    const dm = Math.max(0, decimals);
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return `${(bytes / Math.pow(k, i)).toFixed(dm)} ${sizes[i]}`;
}
//# sourceMappingURL=data:application/json;base64,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