/**
 * Checks if a value is an object (excluding null).
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is an object, false otherwise.
 *
 * @example
 * ```typescript
 * const value = { key: 'value' };
 * console.log(ncIsObject(value)); // true
 * ```
 */
export declare function ncIsObject(value: any): value is object;
/**
 * Checks if a value is an empty object.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is an empty object, false otherwise.
 *
 * @example
 * ```typescript
 * const value = {};
 * console.log(ncIsEmptyObject(value)); // true
 * ```
 */
export declare function ncIsEmptyObject(value: any): boolean;
/**
 * Checks whether the given value is an object and contains all the specified properties.
 *
 * Supports checking either a **single property** or **multiple properties** at once.
 *
 * @template T - The expected object type.
 * @param value - The value to check.
 * @param keyOrKeys - A property key or an array of property keys that should exist in the object.
 *                     Can be a single string, a single `keyof T`, an array of strings, or an array of `keyof T`.
 * @returns {value is T} - Returns `true` if `value` is an object containing all specified keys, otherwise `false`.
 *
 * @example
 * ```typescript
 * type User = { name: string; age: number };
 *
 * const obj = { name: "Alice", age: 25 };
 *
 * // Check multiple keys
 * if (ncHasProperties<User>(obj, ["name", "age"])) {
 *   console.log(obj.name); // ✅ TypeScript ensures obj.name is safe to access
 * }
 *
 * // Check single key
 * if (ncHasProperties<User>(obj, "age")) {
 *   console.log(obj.age); // ✅ Safe to access
 * }
 * ```
 *
 * @example
 * ```typescript
 * const obj = { title: "Hello", value: "World" };
 *
 * // Using string array
 * if (ncHasProperties(obj, ["title", "value"])) {
 *   console.log(obj["title"]); // ✅ Safe to access without explicit type
 * }
 *
 * // Using single string
 * if (ncHasProperties(obj, "title")) {
 *   console.log(obj.title); // ✅ Works
 * }
 * ```
 */
export declare function ncHasProperties<T extends object = object>(value: any, key: keyof T): value is T;
export declare function ncHasProperties<T extends object = object>(value: any, key: string | number | symbol): value is T;
export declare function ncHasProperties<T extends object = object>(value: any, keys: readonly (keyof T)[]): value is T;
export declare function ncHasProperties<T extends object = object>(value: any, keys: readonly (string | number | symbol)[]): value is T;
/**
 * Checks if a value is an array.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is an array, false otherwise.
 *
 * @example
 * ```typescript
 * const value = [1, 2, 3];
 * console.log(ncIsArray(value)); // true
 * ```
 */
export declare function ncIsArray(value: any): value is any[];
/**
 * Checks if a value is an empty array.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is an empty array, false otherwise.
 *
 * @example
 * ```typescript
 * const value = [];
 * console.log(ncIsEmptyArray(value)); // true
 *
 * const nonEmptyArray = [1, 2, 3];
 * console.log(ncIsEmptyArray(nonEmptyArray)); // false
 * ```
 */
export declare function ncIsEmptyArray(value: any): boolean;
/**
 * Checks if a value is a string.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is a string, false otherwise.
 *
 * @example
 * ```typescript
 * const value = 'Hello, world!';
 * console.log(ncIsString(value)); // true
 * ```
 */
export declare function ncIsString(value: any): value is string;
/**
 * Checks if a value is a number.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is a number, false otherwise.
 *
 * @example
 * ```typescript
 * const value = 42;
 * console.log(ncIsNumber(value)); // true
 * ```
 */
export declare function ncIsNumber(value: any): value is number;
/**
 * Checks if a value is NaN (Not-a-Number).
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is NaN, false otherwise.
 *
 * @example
 * ```typescript
 * console.log(ncIsNaN(NaN)); // true
 * console.log(ncIsNaN("abc")); // true
 * console.log(ncIsNaN(42)); // false
 * console.log(ncIsNaN("42")); // false
 * ```
 */
export declare function ncIsNaN(value: any): boolean;
/**
 * Checks if a value is a boolean.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is a boolean, false otherwise.
 *
 * @example
 * ```typescript
 * const value = true;
 * console.log(ncIsBoolean(value)); // true
 * ```
 */
export declare function ncIsBoolean(value: any): value is boolean;
/**
 * Checks if a value is undefined.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is undefined, false otherwise.
 *
 * @example
 * ```typescript
 * const value = undefined;
 * console.log(ncIsUndefined(value)); // true
 * ```
 */
export declare function ncIsUndefined(value: any): value is undefined;
/**
 * Checks if a value is null.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is null, false otherwise.
 *
 * @example
 * ```typescript
 * const value = null;
 * console.log(ncIsNull(value)); // true
 * ```
 */
export declare function ncIsNull(value: any): value is null;
/**
 * Checks if a value is a function.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is a function, false otherwise.
 *
 * @example
 * ```typescript
 * const value = () => {};
 * console.log(ncIsFunction(value)); // true
 * ```
 */
export declare function ncIsFunction(value: any): value is Function;
/**
 * Checks if a value is a promise.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is a Promise, false otherwise.
 *
 * @example
 * ```typescript
 * const value = new Promise((resolve) => resolve(true));
 * console.log(ncIsPromise(value)); // true
 * ```
 */
export declare function ncIsPromise(value: any): boolean;
/**
 * Checks whether an array includes a specific value.
 *
 * If an `objectKey` is provided and the array consists of objects, it will check
 * whether the value of the specified `objectKey` in any object matches the given value.
 *
 * @param {T[]} array - The array to check.
 * @param {any} value - The value to search for in the array.
 * @param {keyof T} [objectKey] - The key to check in objects, if the array contains objects.
 *
 * @returns {boolean} - Returns `true` if the value or object with matching `objectKey` is found, otherwise `false`.
 *
 * @example
 * // For primitive arrays
 * ncIsArrayIncludes([1, 2, 3], 2) // true
 *
 * // For arrays with objects
 * ncIsArrayIncludes([{ id: 1 }, { id: 2 }], 2, 'id') // true
 */
export declare function ncIsArrayIncludes<T>(array: T[], value: any, objectKey?: keyof T): boolean;
export declare function isPrimitiveValue(value: any): value is string | number | boolean | null | undefined;
/**
 * Checks if a value is null or undefined.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is null or undefined, false otherwise.
 *
 * @example
 * ```typescript
 * console.log(ncIsNullOrUndefined(null)); // true
 * console.log(ncIsNullOrUndefined(undefined)); // true
 * console.log(ncIsNullOrUndefined(0)); // false
 * console.log(ncIsNullOrUndefined('')); // false
 * ```
 */
export declare function ncIsNullOrUndefined(value: any): value is null | undefined;
/**
 * Checks if a value is valid.
 *
 * @param val - The value to check.
 * @returns {boolean} - True if the value is valid, false otherwise.
 */
export declare const isValidValue: (val: unknown) => boolean;
/**
 * Checks if a value is a valid UUID.
 *
 * @param value - The value to check.
 * @param version - Optional UUID version to check (1, 2, 3, 4, or 5). If not provided, checks for any valid UUID.
 * @returns {boolean} - True if the value is a valid UUID (of the specified version if provided), false otherwise.
 *
 * @example
 * ```typescript
 * console.log(ncIsUUID('550e8400-e29b-41d4-a716-446655440000')); // true (any version)
 * console.log(ncIsUUID('550e8400-e29b-41d4-a716-446655440000', 4)); // true (v4)
 * console.log(ncIsUUID('6ba7b810-9dad-11d1-80b4-00c04fd430c8', 4)); // false (not v4)
 * console.log(ncIsUUID('invalid-uuid')); // false
 * console.log(ncIsUUID(null)); // false
 * ```
 */
export declare function ncIsUUID(value: any, version?: 1 | 2 | 3 | 4 | 5): value is string;
