/**
 * Checks if a value is an object (excluding null).
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is an object, false otherwise.
 *
 * @example
 * ```typescript
 * const value = { key: 'value' };
 * console.log(ncIsObject(value)); // true
 * ```
 */
export function ncIsObject(value) {
    return value !== null && typeof value === 'object' && !ncIsArray(value);
}
/**
 * Checks if a value is an empty object.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is an empty object, false otherwise.
 *
 * @example
 * ```typescript
 * const value = {};
 * console.log(ncIsEmptyObject(value)); // true
 * ```
 */
export function ncIsEmptyObject(value) {
    return ncIsObject(value) && Object.keys(value).length === 0;
}
// Implementation signature (broad enough)
export function ncHasProperties(value, keyOrKeys) {
    const keys = ncIsArray(keyOrKeys) ? keyOrKeys : [keyOrKeys];
    return ncIsObject(value) && keys.every((key) => key in value);
}
/**
 * Checks if a value is an array.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is an array, false otherwise.
 *
 * @example
 * ```typescript
 * const value = [1, 2, 3];
 * console.log(ncIsArray(value)); // true
 * ```
 */
export function ncIsArray(value) {
    return Array.isArray(value);
}
/**
 * Checks if a value is an empty array.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is an empty array, false otherwise.
 *
 * @example
 * ```typescript
 * const value = [];
 * console.log(ncIsEmptyArray(value)); // true
 *
 * const nonEmptyArray = [1, 2, 3];
 * console.log(ncIsEmptyArray(nonEmptyArray)); // false
 * ```
 */
export function ncIsEmptyArray(value) {
    return ncIsArray(value) && value.length === 0;
}
/**
 * Checks if a value is a string.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is a string, false otherwise.
 *
 * @example
 * ```typescript
 * const value = 'Hello, world!';
 * console.log(ncIsString(value)); // true
 * ```
 */
export function ncIsString(value) {
    return typeof value === 'string';
}
/**
 * Checks if a value is a number.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is a number, false otherwise.
 *
 * @example
 * ```typescript
 * const value = 42;
 * console.log(ncIsNumber(value)); // true
 * ```
 */
export function ncIsNumber(value) {
    return typeof value === 'number' && !Number.isNaN(value);
}
/**
 * Checks if a value is NaN (Not-a-Number).
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is NaN, false otherwise.
 *
 * @example
 * ```typescript
 * console.log(ncIsNaN(NaN)); // true
 * console.log(ncIsNaN("abc")); // true
 * console.log(ncIsNaN(42)); // false
 * console.log(ncIsNaN("42")); // false
 * ```
 */
export function ncIsNaN(value) {
    if (ncIsNumber(value))
        return false;
    if (!value || isNaN(Number(value)))
        return true;
    return false;
}
/**
 * Checks if a value is a boolean.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is a boolean, false otherwise.
 *
 * @example
 * ```typescript
 * const value = true;
 * console.log(ncIsBoolean(value)); // true
 * ```
 */
export function ncIsBoolean(value) {
    return typeof value === 'boolean';
}
/**
 * Checks if a value is undefined.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is undefined, false otherwise.
 *
 * @example
 * ```typescript
 * const value = undefined;
 * console.log(ncIsUndefined(value)); // true
 * ```
 */
export function ncIsUndefined(value) {
    return typeof value === 'undefined';
}
/**
 * Checks if a value is null.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is null, false otherwise.
 *
 * @example
 * ```typescript
 * const value = null;
 * console.log(ncIsNull(value)); // true
 * ```
 */
export function ncIsNull(value) {
    return value === null;
}
/**
 * Checks if a value is a function.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is a function, false otherwise.
 *
 * @example
 * ```typescript
 * const value = () => {};
 * console.log(ncIsFunction(value)); // true
 * ```
 */
export function ncIsFunction(value) {
    return typeof value === 'function';
}
/**
 * Checks if a value is a promise.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is a Promise, false otherwise.
 *
 * @example
 * ```typescript
 * const value = new Promise((resolve) => resolve(true));
 * console.log(ncIsPromise(value)); // true
 * ```
 */
export function ncIsPromise(value) {
    return value instanceof Promise;
}
/**
 * Checks whether an array includes a specific value.
 *
 * If an `objectKey` is provided and the array consists of objects, it will check
 * whether the value of the specified `objectKey` in any object matches the given value.
 *
 * @param {T[]} array - The array to check.
 * @param {any} value - The value to search for in the array.
 * @param {keyof T} [objectKey] - The key to check in objects, if the array contains objects.
 *
 * @returns {boolean} - Returns `true` if the value or object with matching `objectKey` is found, otherwise `false`.
 *
 * @example
 * // For primitive arrays
 * ncIsArrayIncludes([1, 2, 3], 2) // true
 *
 * // For arrays with objects
 * ncIsArrayIncludes([{ id: 1 }, { id: 2 }], 2, 'id') // true
 */
export function ncIsArrayIncludes(array = [], value, objectKey) {
    if (!ncIsArray(array) || !array.length)
        return false;
    if (objectKey && ncIsObject(array[0])) {
        return array.some((item) => item[objectKey] === value);
    }
    return array.includes(value);
}
export function isPrimitiveValue(value) {
    return (ncIsString(value) ||
        ncIsNumber(value) ||
        ncIsBoolean(value) ||
        ncIsNull(value) ||
        ncIsUndefined(value));
}
/**
 * Checks if a value is null or undefined.
 *
 * @param value - The value to check.
 * @returns {boolean} - True if the value is null or undefined, false otherwise.
 *
 * @example
 * ```typescript
 * console.log(ncIsNullOrUndefined(null)); // true
 * console.log(ncIsNullOrUndefined(undefined)); // true
 * console.log(ncIsNullOrUndefined(0)); // false
 * console.log(ncIsNullOrUndefined('')); // false
 * ```
 */
export function ncIsNullOrUndefined(value) {
    return value === null || typeof value === 'undefined';
}
/**
 * Checks if a value is valid.
 *
 * @param val - The value to check.
 * @returns {boolean} - True if the value is valid, false otherwise.
 */
export const isValidValue = (val) => {
    if (ncIsNull(val) || ncIsUndefined(val)) {
        return false;
    }
    if (ncIsString(val) && val === '') {
        return false;
    }
    if (ncIsEmptyArray(val)) {
        return false;
    }
    if (ncIsEmptyObject(val)) {
        return false;
    }
    return true;
};
/**
 * Checks if a value is a valid UUID.
 *
 * @param value - The value to check.
 * @param version - Optional UUID version to check (1, 2, 3, 4, or 5). If not provided, checks for any valid UUID.
 * @returns {boolean} - True if the value is a valid UUID (of the specified version if provided), false otherwise.
 *
 * @example
 * ```typescript
 * console.log(ncIsUUID('550e8400-e29b-41d4-a716-446655440000')); // true (any version)
 * console.log(ncIsUUID('550e8400-e29b-41d4-a716-446655440000', 4)); // true (v4)
 * console.log(ncIsUUID('6ba7b810-9dad-11d1-80b4-00c04fd430c8', 4)); // false (not v4)
 * console.log(ncIsUUID('invalid-uuid')); // false
 * console.log(ncIsUUID(null)); // false
 * ```
 */
export function ncIsUUID(value, version) {
    if (!ncIsString(value))
        return false;
    // UUID regex patterns by version
    const uuidRegexPatterns = {
        1: /^[0-9a-f]{8}-[0-9a-f]{4}-1[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i,
        2: /^[0-9a-f]{8}-[0-9a-f]{4}-2[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i,
        3: /^[0-9a-f]{8}-[0-9a-f]{4}-3[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i,
        4: /^[0-9a-f]{8}-[0-9a-f]{4}-4[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i,
        5: /^[0-9a-f]{8}-[0-9a-f]{4}-5[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i,
    };
    // If version is specified, check for that specific version
    if (version) {
        return uuidRegexPatterns[version].test(value);
    }
    // Otherwise, check for any valid UUID version (v1-v5)
    const uuidRegex = /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;
    return uuidRegex.test(value);
}
//# sourceMappingURL=data:application/json;base64,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