import { NOT_OPERATOR, QUERY_FILTER_TOKENS, QueryFilterLexer, TOKEN_OPERATOR, } from './query-filter-lexer';
import { COMMON_TOKEN } from '../common-token';
import { CommonCstParser } from '../common-cst-parser';
import { parseCst } from './query-filter-cst-parser';
export class QueryFilterParser extends CommonCstParser {
    constructor() {
        super(QUERY_FILTER_TOKENS);
        this.initializeRule();
        // very important to call this after all the rules have been defined.
        // otherwise the parser may not work correctly as it will lack information
        // derived during the self analysis phase.
        this.performSelfAnalysis();
    }
    initializeRule() {
        // not mandatory, using $ (or any other sign) to reduce verbosity (this. this. this. this. .......)
        const $ = this;
        // the parsing methods
        // we define the "rules" of how the syntax will be defined
        $.RULE('multi_clause', () => {
            $.OR([
                { ALT: () => $.SUBRULE($['not_clause'], { LABEL: 'clause' }) },
                { ALT: () => $.SUBRULE($['paren_clause'], { LABEL: 'clause' }) },
            ]);
            $.MANY({
                DEF: () => $.SUBRULE($['and_or_clause'], { LABEL: 'clause' }),
            });
        });
        $.RULE('and_or_clause', () => {
            $.CONSUME(COMMON_TOKEN.IDENTIFIER, { LABEL: 'operator' });
            $.SUBRULE($['paren_clause'], { LABEL: 'clause' });
        });
        $.RULE('not_clause', () => {
            $.CONSUME(NOT_OPERATOR);
            $.SUBRULE($['paren_clause'], { LABEL: 'clause' });
        });
        $.RULE('paren_clause', () => {
            $.CONSUME(COMMON_TOKEN.PAREN_START);
            $.OR([
                { ALT: () => $.SUBRULE($['multi_clause'], { LABEL: 'clause' }) },
                { ALT: () => $.SUBRULE($['call_expression'], { LABEL: 'clause' }) },
            ]);
            $.CONSUME(COMMON_TOKEN.PAREN_END);
        });
        $.RULE('call_expression', () => {
            $.SUBRULE($['VARIABLE']);
            $.CONSUME2(COMMON_TOKEN.COMMA);
            $.CONSUME(TOKEN_OPERATOR);
            $.OPTION(() => {
                $.CONSUME3(COMMON_TOKEN.COMMA);
                // even after comma, the expression arguments can be blank
                $.OPTION2(() => {
                    $.SUBRULE($['expression_arguments']);
                });
            });
        });
        $.RULE('expression_arguments', () => {
            $.MANY_SEP({
                SEP: COMMON_TOKEN.COMMA,
                DEF: () => {
                    $.SUBRULE2($['VARIABLE']);
                },
            });
        });
    }
    parse() {
        return this.multi_clause();
    }
    static parse(text) {
        var _a, _b;
        const parser = new QueryFilterParser();
        const lexResult = QueryFilterLexer.tokenize(text);
        // setting a new input will RESET the parser instance's state.
        parser.input = lexResult.tokens;
        // any top level rule may be used as an entry point
        const cst = parser.parse();
        const parseErrors = [...parser.errors];
        let parsedCst = undefined;
        try {
            parsedCst = parseErrors.length === 0 ? parseCst(cst) : undefined;
        }
        catch (error) {
            parseErrors.push({
                name: (_a = error === null || error === void 0 ? void 0 : error.message) !== null && _a !== void 0 ? _a : 'Invalid filter expression',
                message: (_b = error === null || error === void 0 ? void 0 : error.message) !== null && _b !== void 0 ? _b : 'Invalid filter expression',
                token: null,
                resyncedTokens: [],
                context: null,
            });
        }
        return {
            cst: cst,
            lexErrors: lexResult.errors,
            parseErrors,
            parsedCst,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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