var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { parseParsingError } from './error-message-parser';
import { QueryFilterParser } from './query-filter-parser';
describe('query-filter-parser', () => {
    it('will parse eq expression with double quote', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = `(field1, eq, "hello, 'world")`;
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field1',
                    comparison_op: 'eq',
                    value: "hello, 'world",
                },
            ],
        };
        const result = QueryFilterParser.parse(text);
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse eq expression with single quote', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = `(field1, eq, 'hello, "world')`;
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field1',
                    comparison_op: 'eq',
                    value: 'hello, "world',
                },
            ],
        };
        const result = QueryFilterParser.parse(text);
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse eq expression with mandarin, japanese and cryillic', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = `(Тест, eq, "新年快乐おはよう")`;
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'Тест',
                    comparison_op: 'eq',
                    value: '新年快乐おはよう',
                },
            ],
        };
        const result = QueryFilterParser.parse(text);
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse wrong expression', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = '(field1)';
        const result = QueryFilterParser.parse(text);
        expect(result.parseErrors.length).toBeGreaterThan(0);
    }));
    it('will parse blank expression', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = '(field1, blank)';
        const result = QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field1',
                    comparison_op: 'blank',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse blank value', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = '(Category,gb_eq,)';
        const result = QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'Category',
                    comparison_op: 'gb_eq',
                    value: '',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse value and field with parentheses', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = '("field(1)",eq,"(hello)")';
        const result = QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field(1)',
                    comparison_op: 'eq',
                    value: '(hello)',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse value separated by comma', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = '("field(1)",eq, hello, world,  baby!)';
        const result = QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field(1)',
                    comparison_op: 'eq',
                    value: 'hello,world,baby!',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse complex nested logic', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = '~not(field1, isWithin, nextNumberOfDays, 10)~and((field2, eq, 2)~or(field2, eq, 3))~or(field3, not, 4)';
        const result = QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field1',
                    comparison_op: 'isWithin',
                    logical_op: 'not',
                    comparison_sub_op: undefined,
                    value: ['nextNumberOfDays', '10'],
                },
                {
                    is_group: true,
                    logical_op: 'and',
                    children: [
                        {
                            is_group: false,
                            field: 'field2',
                            comparison_op: 'eq',
                            value: '2',
                        },
                        {
                            is_group: false,
                            field: 'field2',
                            comparison_op: 'eq',
                            logical_op: 'or',
                            value: '3',
                        },
                    ],
                },
                {
                    is_group: false,
                    field: 'field3',
                    comparison_op: 'not',
                    logical_op: 'or',
                    value: '4',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse keyword as value', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = '(Category,is,blank)';
        const result = QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'Category',
                    comparison_op: 'blank',
                    value: undefined,
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse empty quote as value', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = "(Category,eq,'')";
        const result = QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'Category',
                    comparison_op: 'eq',
                    value: '',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse null as null', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = '("field(1)",eq, null)';
        const result = QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field(1)',
                    comparison_op: 'eq',
                    value: null,
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    it('will parse empty string as empty', () => __awaiter(void 0, void 0, void 0, function* () {
        const text = `("field(1)",eq,'')`;
        const result = QueryFilterParser.parse(text);
        const expectedParsedCst = {
            is_group: true,
            logical_op: 'and',
            children: [
                {
                    is_group: false,
                    field: 'field(1)',
                    comparison_op: 'eq',
                    value: '',
                },
            ],
        };
        expect(result.parsedCst).toEqual(expectedParsedCst);
    }));
    describe('error-handling', () => {
        it(`will error when and/or operation is wrong`, () => __awaiter(void 0, void 0, void 0, function* () {
            expect.hasAssertions();
            const text = `(fSingleLineText,eq,"sample,text")or(fSingleLineText,eq,"sample text")`;
            const result = QueryFilterParser.parse(text);
            const message = parseParsingError(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter expression. Expected a valid logical operator like '~or' or '~and', but found 'or'`);
        }));
        it(`will handle parsing error when operation is wrong`, () => __awaiter(void 0, void 0, void 0, function* () {
            const text = `(fSingleLineText,noneInOperation,"sample,text")`;
            const result = QueryFilterParser.parse(text);
            const message = parseParsingError(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter expression: 'noneInOperation' is not a recognized operator. Please use a valid comparison or logical operator`);
        }));
        it(`will handle parsing error when operation is missing`, () => __awaiter(void 0, void 0, void 0, function* () {
            const text = `(fSingleLineText,)`;
            const result = QueryFilterParser.parse(text);
            const message = parseParsingError(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter expression: ')' is not a recognized operator. Please use a valid comparison or logical operator`);
        }));
        it(`will handle parsing error when operation is wrapped in quotes`, () => __awaiter(void 0, void 0, void 0, function* () {
            const text = `(fSingleLineText,"eq")`;
            const result = QueryFilterParser.parse(text);
            const message = parseParsingError(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter expression: '"eq"' is not a recognized operator. Please use a valid comparison or logical operator`);
        }));
        it(`will handle parsing error when no opening parentheses`, () => __awaiter(void 0, void 0, void 0, function* () {
            const text = `fSingleLineText,eq)`;
            const result = QueryFilterParser.parse(text);
            const message = parseParsingError(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter syntax: expected a logical operator like '~not' or opening parenthesis, but found 'fSingleLineText'`);
        }));
        it(`will handle parsing error when no closing parentheses`, () => __awaiter(void 0, void 0, void 0, function* () {
            const text = `(fSingleLineText,eq`;
            const result = QueryFilterParser.parse(text);
            const message = parseParsingError(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter syntax: expected a closing parentheses ')', but found ''`);
        }));
        it(`will handle parsing error when not operator is wrong`, () => __awaiter(void 0, void 0, void 0, function* () {
            const text = `not(fSingleLineText,eq,1)`;
            const result = QueryFilterParser.parse(text);
            const message = parseParsingError(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter syntax: expected a logical operator like '~not' or opening parenthesis, but found 'not'`);
        }));
        it(`will handle parsing error when missing comma`, () => __awaiter(void 0, void 0, void 0, function* () {
            const text = `(fSingleLineText`;
            const result = QueryFilterParser.parse(text);
            const message = parseParsingError(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter syntax: expected comma ',' followed with operator (and value) after field`);
        }));
        it(`will handle parsing error when missing arguments`, () => __awaiter(void 0, void 0, void 0, function* () {
            const text = `(fSingleLineText)`;
            const result = QueryFilterParser.parse(text);
            const message = parseParsingError(result.parseErrors[0]);
            expect(message).toBe(`Invalid filter syntax: expected comma ',' followed with operator (and value) after field`);
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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