import { OrderedProjectRoles, OrderedWorkspaceRoles, ProjectRoles, WorkspaceRolesToProjectRoles, WorkspaceUserRoles, } from './enums';
import { extractRolesObj } from './helperFunctions';
export function extractProjectRolePower(user, 
/**
 * forbiddenCallback is used to keep old function behaviour
 */
forbiddenCallback) {
    const reverseOrderedProjectRoles = [...OrderedProjectRoles].reverse();
    if (!user.base_roles) {
        return -1;
    }
    // get most powerful role of user (TODO moving forward we will confirm that user has only one role)
    let role = null;
    let power = -1;
    for (const r of Object.keys(user.base_roles)) {
        const ind = reverseOrderedProjectRoles.indexOf(r);
        if (ind > power) {
            role = r;
            power = ind;
        }
    }
    const ind = reverseOrderedProjectRoles.indexOf(role);
    if (ind === -1) {
        forbiddenCallback === null || forbiddenCallback === void 0 ? void 0 : forbiddenCallback();
    }
    return ind;
}
/**
 * Get the power of the workspace role of the user.
 * @param user - The user object.
 * @returns The power of the workspace role of the user.
 */
export function extractWorkspaceRolePower(user, 
/**
 * forbiddenCallback is used to keep old function behaviour
 */ forbiddenCallback) {
    const reverseOrderedWorkspaceRoles = [...OrderedWorkspaceRoles].reverse();
    if (!user.workspace_roles) {
        return -1;
    }
    // get most powerful role of user (TODO moving forward we will confirm that user has only one role)
    let role = null;
    let power = -1;
    for (const r of Object.keys(user.workspace_roles)) {
        const ind = reverseOrderedWorkspaceRoles.indexOf(r);
        if (ind > power) {
            role = r;
            power = ind;
        }
    }
    const ind = reverseOrderedWorkspaceRoles.indexOf(role);
    if (ind === -1) {
        forbiddenCallback === null || forbiddenCallback === void 0 ? void 0 : forbiddenCallback();
    }
    return ind;
}
/**
 * Map the workspace roles object to the project roles object.
 * @param wsRoles - The workspace roles object.
 * @returns The project roles object.
 */
export function mapWorkspaceRolesObjToProjectRolesObj(wsRoles) {
    wsRoles = extractRolesObj(wsRoles);
    let baseRoles = null;
    if (wsRoles) {
        for (const r of Object.keys(wsRoles)) {
            if (!baseRoles)
                baseRoles = {};
            baseRoles[WorkspaceRolesToProjectRoles[r]] = wsRoles[r];
        }
    }
    return baseRoles;
}
export function getProjectRole(user, inheritFromWorkspace = false) {
    // get most powerful role of user (TODO moving forward we will confirm that user has only one role)
    let role = null;
    let power = -1;
    if (user.base_roles) {
        for (const r of Object.keys(user.base_roles)) {
            const ind = OrderedProjectRoles.indexOf(r);
            if (ind > power) {
                role = r;
                power = ind;
            }
        }
        return role;
    }
    else if (inheritFromWorkspace && user.workspace_roles) {
        for (const r of Object.keys(user.workspace_roles)) {
            const ind = OrderedWorkspaceRoles.indexOf(r);
            if (ind > power) {
                role = r;
                power = ind;
            }
        }
        return role;
    }
    else {
        return null;
    }
}
export function hasMinimumRoleAccess(user, minimumRole, 
/**
 * forbiddenCallback is used to keep old function behaviour
 */
forbiddenCallback) {
    const power = extractProjectRolePower(user, forbiddenCallback);
    const reverseOrderedProjectRoles = [...OrderedProjectRoles].reverse();
    const minimumRoleIndex = reverseOrderedProjectRoles.indexOf(minimumRole);
    return power >= minimumRoleIndex;
}
// extract corresponding base role from workspace role
export function extractBaseRoleFromWorkspaceRole(workspaceRole) {
    if (!workspaceRole)
        return null;
    let workspaceRoleStr;
    if (typeof workspaceRole === 'object') {
        // If workspaceRole is an object, extract the first key
        workspaceRoleStr = Object.keys(workspaceRole)[0];
    }
    else if (typeof workspaceRole === 'string') {
        // If workspaceRole is a string, use it directly
        workspaceRoleStr = workspaceRole;
    }
    // Extract base role from workspace role
    const baseRole = WorkspaceRolesToProjectRoles[workspaceRoleStr];
    return baseRole || null;
}
export function getEffectiveBaseRole({ baseRole, baseTeamRole, workspaceRole, workspaceTeamRole, defaultBaseRole, }) {
    // Apply role priority hierarchy for base roles:
    // 1. Direct base role (highest priority)
    // 2. Role inherited from base-team
    // 3. Default base role
    // 4. Role inherited from workspace role
    // 5. Role inherited from workspace team role (lowest priority)
    let finalBaseRole = baseRole;
    if (!finalBaseRole || finalBaseRole === ProjectRoles.INHERIT) {
        if (baseTeamRole) {
            finalBaseRole = baseTeamRole;
        }
        else if (defaultBaseRole) {
            finalBaseRole = defaultBaseRole;
        }
        else if (workspaceRole && workspaceRole !== WorkspaceUserRoles.INHERIT) {
            finalBaseRole = extractBaseRoleFromWorkspaceRole(workspaceRole);
        }
        else if (workspaceTeamRole) {
            finalBaseRole = extractBaseRoleFromWorkspaceRole(workspaceTeamRole);
        }
    }
    return finalBaseRole === ProjectRoles.INHERIT || !finalBaseRole
        ? ProjectRoles.NO_ACCESS
        : finalBaseRole;
}
//# sourceMappingURL=data:application/json;base64,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