import UITypes from '../UITypes';
import { numberize } from '../numberUtils';
const dbTypes = [
    'int',
    'smallint',
    'mediumint',
    'bigint',
    'bit',
    'boolean',
    'float',
    'decimal',
    'double',
    'serial',
    'tinyint',
    'date',
    'datetime',
    'timestamp',
    'time',
    'year',
    'char',
    'varchar',
    'nchar',
    'text',
    'tinytext',
    'mediumtext',
    'longtext',
    'binary',
    'varbinary',
    'blob',
    'tinyblob',
    'mediumblob',
    'longblob',
    'enum',
    'set',
    'geometry',
    'point',
    'linestring',
    'polygon',
    'multipoint',
    'multilinestring',
    'multipolygon',
    'json',
];
export class MysqlUi {
    //#region statics
    static getNewTableColumns() {
        return [
            {
                column_name: 'id',
                title: 'Id',
                dt: 'int',
                dtx: 'integer',
                ct: 'int(11)',
                nrqd: false,
                rqd: true,
                ck: false,
                pk: true,
                un: true,
                ai: true,
                cdf: null,
                clen: null,
                np: 11,
                ns: 0,
                dtxp: '11',
                dtxs: '',
                altered: 1,
                uidt: 'ID',
                uip: '',
                uicn: '',
            },
            {
                column_name: 'title',
                title: 'Title',
                dt: 'TEXT',
                dtx: 'specificType',
                ct: null,
                nrqd: true,
                rqd: false,
                ck: false,
                pk: false,
                un: false,
                ai: false,
                cdf: null,
                clen: null,
                np: null,
                ns: null,
                dtxp: '',
                dtxs: '',
                altered: 1,
                uidt: 'SingleLineText',
                uip: '',
                uicn: '',
            },
            {
                column_name: 'created_at',
                title: 'CreatedAt',
                dt: 'timestamp',
                dtx: 'specificType',
                ct: 'timestamp',
                nrqd: true,
                rqd: false,
                ck: false,
                pk: false,
                un: false,
                ai: false,
                cdf: null,
                clen: 45,
                np: null,
                ns: null,
                dtxp: '',
                dtxs: '',
                altered: 1,
                uidt: UITypes.CreatedTime,
                uip: '',
                uicn: '',
                system: true,
            },
            {
                column_name: 'updated_at',
                title: 'UpdatedAt',
                dt: 'timestamp',
                dtx: 'specificType',
                ct: 'timestamp',
                nrqd: true,
                rqd: false,
                ck: false,
                pk: false,
                un: false,
                ai: false,
                clen: 45,
                np: null,
                ns: null,
                dtxp: '',
                dtxs: '',
                altered: 1,
                uidt: UITypes.LastModifiedTime,
                uip: '',
                uicn: '',
                system: true,
            },
            {
                column_name: 'created_by',
                title: 'nc_created_by',
                dt: 'varchar',
                dtx: 'specificType',
                ct: 'varchar(45)',
                nrqd: true,
                rqd: false,
                ck: false,
                pk: false,
                un: false,
                ai: false,
                clen: 45,
                np: null,
                ns: null,
                dtxp: '45',
                dtxs: '',
                altered: 1,
                uidt: UITypes.CreatedBy,
                uip: '',
                uicn: '',
                system: true,
            },
            {
                column_name: 'updated_by',
                title: 'nc_updated_by',
                dt: 'varchar',
                dtx: 'specificType',
                ct: 'varchar(45)',
                nrqd: true,
                rqd: false,
                ck: false,
                pk: false,
                un: false,
                ai: false,
                clen: 45,
                np: null,
                ns: null,
                dtxp: '45',
                dtxs: '',
                altered: 1,
                uidt: UITypes.LastModifiedBy,
                uip: '',
                uicn: '',
                system: true,
            },
            {
                column_name: 'nc_order',
                title: 'nc_order',
                dt: 'decimal',
                dtx: 'decimal',
                ct: 'decimal(40,20)',
                nrqd: true,
                rqd: false,
                ck: false,
                pk: false,
                un: false,
                ai: false,
                cdf: null,
                clen: null,
                np: 40,
                ns: 20,
                dtxp: '40,20',
                dtxs: '',
                altered: 1,
                uidt: UITypes.Order,
                uip: '',
                uicn: '',
                system: true,
            },
        ];
    }
    static getNewColumn(suffix) {
        return {
            column_name: 'title' + suffix,
            dt: 'TEXT',
            dtx: 'specificType',
            ct: null,
            nrqd: true,
            rqd: false,
            ck: false,
            pk: false,
            un: false,
            ai: false,
            cdf: null,
            clen: null,
            np: null,
            ns: null,
            dtxp: '',
            dtxs: '',
            altered: 1,
            uidt: 'SingleLineText',
            uip: '',
            uicn: '',
        };
    }
    static getDefaultLengthForDatatype(type) {
        switch (type) {
            case 'int':
                return 11;
            case 'tinyint':
                return 1;
            case 'smallint':
                return 5;
            case 'mediumint':
                return 9;
            case 'bigint':
                return 20;
            case 'bit':
                return 64;
            case 'boolean':
                return '';
            case 'float':
                return 12;
            case 'decimal':
                return 10;
            case 'double':
                return 22;
            case 'serial':
                return 20;
            case 'date':
                return '';
            case 'datetime':
            case 'timestamp':
                return '';
            case 'time':
                return '';
            case 'year':
                return '';
            case 'char':
                return 255;
            case 'varchar':
                return 255;
            case 'nchar':
                return 255;
            case 'text':
                return '';
            case 'tinytext':
                return '';
            case 'mediumtext':
                return '';
            case 'longtext':
                return '';
            case 'binary':
                return 255;
            case 'varbinary':
                return 65500;
            case 'blob':
                return '';
            case 'tinyblob':
                return '';
            case 'mediumblob':
                return '';
            case 'longblob':
                return '';
            case 'enum':
                return '';
            case 'set':
                return '';
            case 'geometry':
                return '';
            case 'point':
                return '';
            case 'linestring':
                return '';
            case 'polygon':
                return '';
            case 'multipoint':
                return '';
            case 'multilinestring':
                return '';
            case 'multipolygon':
                return '';
            case 'json':
                return '';
        }
    }
    static getDefaultLengthIsDisabled(type) {
        switch (type) {
            case 'int':
            case 'tinyint':
            case 'smallint':
            case 'mediumint':
            case 'bigint':
            case 'float':
            case 'decimal':
            case 'double':
            case 'serial':
            case 'datetime':
            case 'timestamp':
            case 'char':
            case 'varchar':
            case 'nchar':
            case 'binary':
            case 'varbinary':
            case 'enum':
            case 'set':
            case 'geometry':
            case 'point':
            case 'linestring':
            case 'polygon':
            case 'multipoint':
            case 'multilinestring':
            case 'multipolygon':
            case 'json':
            case 'bit':
                return false;
            case 'boolean':
            case 'date':
            case 'time':
            case 'year':
            case 'text':
            case 'tinytext':
            case 'mediumtext':
            case 'longtext':
            case 'blob':
            case 'tinyblob':
            case 'mediumblob':
            case 'longblob':
                return true;
        }
    }
    static getDefaultValueForDatatype(type) {
        switch (type) {
            case 'int':
                return 'eg : ' + 10;
            case 'tinyint':
                return 'eg : ' + 1;
            case 'smallint':
                return 'eg : ' + 10;
            case 'mediumint':
                return 'eg : ' + 10;
            case 'bigint':
                return 'eg : ' + 100;
            case 'bit':
                return 'eg : ' + 1;
            case 'boolean':
                return 'eg : ' + 1;
            case 'float':
                return 'eg : ' + 10.0;
            case 'decimal':
                return 'eg : ' + 10.0;
            case 'double':
                return 'eg : ' + 10.0;
            case 'serial':
                return 'eg : ' + 100;
            case 'date':
                return 'eg : ' + '2020-09-09';
            case 'datetime':
                return ('eg : ' +
                    'now()\n\nCURRENT_TIMESTAMP\n\nCURRENT_TIMESTAMP on update CURRENT_TIMESTAMP\n\n1992-10-12 00:00:00');
            case 'timestamp':
                return ('eg : ' +
                    'now()\n\nCURRENT_TIMESTAMP\n\nCURRENT_TIMESTAMP on update CURRENT_TIMESTAMP\n\n1992-10-12 00:00:00');
            case 'time':
                return 'eg : ' + '00:00:00';
            case 'year':
                return 'eg : ' + '2020';
            case 'char':
                return 'eg : ' + 'a';
            case 'varchar':
                return 'eg : ' + 'hey';
            case 'nchar':
                return 'eg : ' + 'hey';
            case 'text':
                return 'eg : ' + 'hey';
            case 'tinytext':
                return 'eg : ' + 'hey';
            case 'mediumtext':
                return 'eg : ' + 'hey';
            case 'longtext':
                return 'eg : ' + 'hey';
            case 'binary':
                return 'eg : ' + 1;
            case 'varbinary':
                return 'eg : ' + 'hey';
            case 'blob':
                return 'eg : ' + 'hey';
            case 'tinyblob':
                return 'eg : ' + 'hey';
            case 'mediumblob':
                return 'eg : ' + 'hey';
            case 'longblob':
                return 'eg : ' + 'hey';
            case 'enum':
                return 'eg : ' + 'a';
            case 'set':
                return 'eg : ' + 'a';
            case 'geometry':
                return "geometry can't have default value";
            case 'point':
                return "point can't have default value";
            case 'linestring':
                return "linestring can't have default value";
            case 'polygon':
                return "polygon can't have default value";
            case 'multipoint':
                return "multipoint can't have default value";
            case 'multilinestring':
                return "multilinestring can't have default value";
            case 'multipolygon':
                return "multipolygon can't have default value";
            case 'json':
                return "JSON can't have default value";
        }
    }
    static getDefaultScaleForDatatype(type) {
        switch (type) {
            case 'int':
                return ' ';
            case 'tinyint':
                return ' ';
            case 'smallint':
                return ' ';
            case 'mediumint':
                return ' ';
            case 'bigint':
                return ' ';
            case 'bit':
                return ' ';
            case 'boolean':
                return ' ';
            case 'float':
                return '2';
            case 'decimal':
                return '2';
            case 'double':
                return '2';
            case 'serial':
                return ' ';
            case 'date':
            case 'datetime':
            case 'timestamp':
                return ' ';
            case 'time':
                return ' ';
            case 'year':
                return ' ';
            case 'char':
                return ' ';
            case 'varchar':
                return ' ';
            case 'nchar':
                return ' ';
            case 'text':
                return ' ';
            case 'tinytext':
                return ' ';
            case 'mediumtext':
                return ' ';
            case 'longtext':
                return ' ';
            case 'binary':
                return ' ';
            case 'varbinary':
                return ' ';
            case 'blob':
                return ' ';
            case 'tinyblob':
                return ' ';
            case 'mediumblob':
                return ' ';
            case 'longblob':
                return ' ';
            case 'enum':
                return ' ';
            case 'set':
                return ' ';
            case 'geometry':
                return ' ';
            case 'point':
                return ' ';
            case 'linestring':
                return ' ';
            case 'polygon':
                return ' ';
            case 'multipoint':
                return ' ';
            case 'multilinestring':
                return ' ';
            case 'multipolygon':
                return ' ';
            case 'json':
                return ' ';
        }
    }
    static colPropAIDisabled(col, columns) {
        // console.log(col);
        if (col.dt === 'int' ||
            col.dt === 'tinyint' ||
            col.dt === 'bigint' ||
            col.dt === 'smallint') {
            for (let i = 0; i < columns.length; ++i) {
                if (columns[i].cn !== col.cn && columns[i].ai) {
                    return true;
                }
            }
            return false;
        }
        else {
            return true;
        }
    }
    static colPropUNDisabled(col) {
        // console.log(col);
        return !(col.dt === 'int' ||
            col.dt === 'tinyint' ||
            col.dt === 'smallint' ||
            col.dt === 'mediumint' ||
            col.dt === 'bigint');
    }
    static onCheckboxChangeAI(col) {
        if (col.dt === 'int' ||
            col.dt === 'bigint' ||
            col.dt === 'smallint' ||
            col.dt === 'tinyint') {
            col.altered = col.altered || 2;
        }
        // if (!col.ai) {
        //   col.dtx = 'specificType'
        // } else {
        //   col.dtx = ''
        // }
    }
    static onCheckboxChangeAU(col) {
        console.log(col);
        // if (1) {
        col.altered = col.altered || 2;
        // }
        // if (!col.ai) {
        //   col.dtx = 'specificType'
        // } else {
        //   col.dtx = ''
        // }
    }
    static showScale(columnObj) {
        return (columnObj.dt === 'float' ||
            columnObj.dt === 'decimal' ||
            columnObj.dt === 'double' ||
            columnObj.dt === 'real');
    }
    static removeUnsigned(columns) {
        for (let i = 0; i < columns.length; ++i) {
            if (columns[i].altered === 1 &&
                !(columns[i].dt === 'int' ||
                    columns[i].dt === 'bigint' ||
                    columns[i].dt === 'tinyint' ||
                    columns[i].dt === 'smallint' ||
                    columns[i].dt === 'mediumint')) {
                columns[i].un = false;
                console.log('>> resetting unsigned value', columns[i].cn);
            }
            console.log(columns[i].cn);
        }
    }
    static columnEditable(colObj) {
        return colObj.tn !== '_evolutions' || colObj.tn !== 'nc_evolutions';
    }
    /*
  
    static extractFunctionName(query) {
      const reg =
        /^\s*CREATE\s+.*?(?:OR\s+REPLACE\s*)?\s*FUNCTION\s+(?:`?[\w\d_]+`?\.)?`?([\w_\d]+)`?/i;
      const match = query.match(reg);
      return match && match[1];
    }
  
    static extractProcedureName(query) {
      const reg =
        /^\s*CREATE.*?\s+(?:OR\s+REPLACE\s*)?\s*PROCEDURE\s+(?:[\w\d_]+\.)?([\w_\d]+)/i;
      const match = query.match(reg);
      return match && match[1];
    }
  
    static handleRawOutput(result, headers) {
      result = result[0] ? result[0] : [];
      if (Array.isArray(result) && result[0]) {
        const keys = Object.keys(result[0]);
        // set headers before settings result
        for (let i = 0; i < keys.length; i++) {
          const text = keys[i];
          headers.push({ text, value: text, sortable: false });
        }
      } else {
        const keys = Object.keys(result);
        for (let i = 0; i < keys.length; i++) {
          const text = keys[i];
          if (typeof text !== 'function') {
            headers.push({ text, value: text, sortable: false });
          }
        }
        result = [result];
      }
      return result;
    }
  
    static splitQueries(query) {
      /!***
       * we are splitting based on semicolon
       * there are mechanism to escape semicolon within single/double quotes(string)
       *!/
      return query.match(/\b("[^"]*;[^"]*"|'[^']*;[^']*'|[^;])*;/g);
    }
  
    /!**
     * if sql statement is SELECT - it limits to a number
     * @param args
     * @returns {string|*}
     *!/
    static sanitiseQuery(args) {
      let q = args.query.trim().split(';');
  
      if (q[0].startsWith('Select')) {
        q = q[0] + ` LIMIT 0,${args.limit ? args.limit : 100};`;
      } else if (q[0].startsWith('select')) {
        q = q[0] + ` LIMIT 0,${args.limit ? args.limit : 100};`;
      } else if (q[0].startsWith('SELECT')) {
        q = q[0] + ` LIMIT 0,${args.limit ? args.limit : 100};`;
      } else {
        return args.query;
      }
  
      return q;
    }
  
    static getColumnsFromJson(json, tn) {
      const columns = [];
  
      try {
        if (typeof json === 'object' && !Array.isArray(json)) {
          const keys = Object.keys(json);
          for (let i = 0; i < keys.length; ++i) {
            const column = {
              dp: null,
              tn,
              column_name: keys[i],
              cno: keys[i],
              np: 10,
              ns: 0,
              clen: null,
              cop: 1,
              pk: false,
              nrqd: false,
              rqd: false,
              un: false,
              ct: 'int(11) unsigned',
              ai: false,
              unique: false,
              cdf: null,
              cc: '',
              csn: null,
              dtx: 'specificType',
              dtxp: null,
              dtxs: 0,
              altered: 1,
            };
  
            switch (typeof json[keys[i]]) {
              case 'number':
                if (Number.isInteger(json[keys[i]])) {
                  if (MysqlUi.isValidTimestamp(keys[i], json[keys[i]])) {
                    Object.assign(column, {
                      dt: 'timestamp',
                    });
                  } else {
                    Object.assign(column, {
                      dt: 'int',
                      np: 10,
                      ns: 0,
                    });
                  }
                } else {
                  Object.assign(column, {
                    dt: 'float',
                    np: 10,
                    ns: 2,
                    dtxp: '11',
                    dtxs: 2,
                  });
                }
                break;
              case 'string':
                if (MysqlUi.isValidDate(json[keys[i]])) {
                  Object.assign(column, {
                    dt: 'datetime',
                  });
                } else if (json[keys[i]].length <= 255) {
                  Object.assign(column, {
                    dt: 'varchar',
                    np: 255,
                    ns: 0,
                    dtxp: '255',
                  });
                } else {
                  Object.assign(column, {
                    dt: 'text',
                  });
                }
                break;
              case 'boolean':
                Object.assign(column, {
                  dt: 'boolean',
                  np: 3,
                  ns: 0,
                });
                break;
              case 'object':
                Object.assign(column, {
                  dt: 'json',
                  np: 3,
                  ns: 0,
                });
                break;
              default:
                break;
            }
            columns.push(column);
          }
        }
      } catch (e) {
        console.log('Error in getColumnsFromJson', e);
      }
  
      return columns;
    }
  
    static isValidTimestamp(key, value) {
      if (typeof value !== 'number') {
        return false;
      }
      return new Date(value).getTime() > 0 && /(?:_|(?=A))[aA]t$/.test(key);
    }
  
    static isValidDate(value) {
      return new Date(value).getTime() > 0;
    }
  */
    static colPropAuDisabled(_col) {
        return true;
    }
    static getUIType(col) {
        switch (this.getAbstractType(col)) {
            case 'integer':
                return 'Number';
            case 'boolean':
                return 'Checkbox';
            case 'float':
                return 'Decimal';
            case 'date':
                return 'Date';
            case 'datetime':
                return 'DateTime';
            case 'time':
                return 'Time';
            case 'year':
                return 'Year';
            case 'string':
                return 'SingleLineText';
            case 'text':
                return 'LongText';
            case 'blob':
                return 'Attachment';
            case 'enum':
                return 'SingleSelect';
            case 'set':
                return 'MultiSelect';
            case 'json':
                return 'LongText';
        }
    }
    static getAbstractType(col) {
        var _a;
        switch ((_a = col.dt) === null || _a === void 0 ? void 0 : _a.toLowerCase()) {
            case 'int':
            case 'smallint':
            case 'mediumint':
            case 'bigint':
            case 'bit':
                return 'integer';
            case 'boolean':
                return 'boolean';
            case 'float':
            case 'decimal':
            case 'double':
            case 'serial':
                return 'float';
            case 'tinyint':
                if (col.dtxp == '1') {
                    return 'boolean';
                }
                else {
                    return 'integer';
                }
            case 'date':
                return 'date';
            case 'datetime':
            case 'timestamp':
                return 'datetime';
            case 'time':
                return 'time';
            case 'year':
                return 'year';
            case 'char':
            case 'varchar':
            case 'nchar':
                return 'string';
            case 'text':
            case 'tinytext':
            case 'mediumtext':
            case 'longtext':
                return 'text';
            // todo: use proper type
            case 'binary':
                return 'string';
            case 'varbinary':
                return 'text';
            case 'blob':
            case 'tinyblob':
            case 'mediumblob':
            case 'longblob':
                return 'blob';
            case 'enum':
                return 'enum';
            case 'set':
                return 'set';
            case 'geometry':
            case 'point':
            case 'linestring':
            case 'polygon':
            case 'multipoint':
            case 'multilinestring':
            case 'multipolygon':
                return 'string';
            case 'json':
                return 'json';
        }
        return 'string';
    }
    static getDataTypeForUiType(col, idType) {
        var _a, _b, _c, _d;
        const colProp = {};
        switch (col.uidt) {
            case 'ID':
                {
                    const isAutoIncId = idType === 'AI';
                    const isAutoGenId = idType === 'AG';
                    colProp.dt = isAutoGenId ? 'varchar' : 'int';
                    colProp.pk = true;
                    colProp.un = isAutoIncId;
                    colProp.ai = isAutoIncId;
                    colProp.rqd = true;
                    colProp.meta = isAutoGenId ? { ag: 'nc' } : undefined;
                }
                break;
            case 'ForeignKey':
                colProp.dt = 'varchar';
                break;
            case 'SingleLineText':
                colProp.dt = 'text';
                break;
            case 'LongText':
                colProp.dt = 'text';
                break;
            case 'Attachment':
                colProp.dt = 'text';
                break;
            case 'GeoData':
                colProp.dt = 'text';
                break;
            case 'Checkbox':
                colProp.dt = 'tinyint';
                colProp.dtxp = 1;
                colProp.cdf = '0';
                break;
            case 'MultiSelect':
                colProp.dt = 'set';
                if (((_a = col.colOptions) === null || _a === void 0 ? void 0 : _a.options.length) > 64 ||
                    ((_d = (_c = (_b = col.dtxp) === null || _b === void 0 ? void 0 : _b.toString) === null || _c === void 0 ? void 0 : _c.call(_b)) === null || _d === void 0 ? void 0 : _d.split(',').length) > 64) {
                    colProp.dt = 'text';
                }
                break;
            case 'SingleSelect':
                colProp.dt = 'enum';
                break;
            case 'Collaborator':
                colProp.dt = 'varchar';
                break;
            case 'Date':
                colProp.dt = 'date';
                break;
            case 'Year':
                colProp.dt = 'year';
                break;
            case 'Time':
                colProp.dt = 'time';
                break;
            case 'PhoneNumber':
                colProp.dt = 'varchar';
                colProp.validate = {
                    func: ['isMobilePhone'],
                    args: [''],
                    msg: ['Validation failed : isMobilePhone ({cn})'],
                };
                break;
            case 'Email':
                colProp.dt = 'varchar';
                colProp.validate = {
                    func: ['isEmail'],
                    args: [''],
                    msg: ['Validation failed : isEmail ({cn})'],
                };
                break;
            case 'URL':
                colProp.dt = 'text';
                colProp.validate = {
                    func: ['isURL'],
                    args: [''],
                    msg: ['Validation failed : isURL ({cn})'],
                };
                break;
            case 'Number':
                colProp.dt = 'bigint';
                break;
            case 'Decimal':
                colProp.dt = 'decimal';
                break;
            case 'Currency':
                colProp.dt = 'decimal';
                colProp.validate = {
                    func: ['isCurrency'],
                    args: [''],
                    msg: ['Validation failed : isCurrency'],
                };
                break;
            case 'Percent':
                colProp.dt = 'double';
                break;
            case 'Duration':
                colProp.dt = 'decimal';
                break;
            case 'Rating':
                colProp.dt = 'int';
                colProp.cdf = '0';
                break;
            case 'Formula':
                colProp.dt = 'varchar';
                break;
            case 'Rollup':
                colProp.dt = 'varchar';
                break;
            case 'Count':
                colProp.dt = 'int';
                break;
            case 'Lookup':
                colProp.dt = 'varchar';
                break;
            case 'DateTime':
                colProp.dt = 'datetime';
                break;
            case 'CreatedTime':
                colProp.dt = 'timestamp';
                break;
            case 'LastModifiedTime':
                colProp.dt = 'timestamp';
                break;
            case 'AutoNumber':
                colProp.dt = 'int';
                break;
            case 'Barcode':
                colProp.dt = 'varchar';
                break;
            case 'Button':
                colProp.dt = 'varchar';
                break;
            case 'JSON':
                colProp.dt = 'json';
                break;
            case 'Order':
                colProp.dt = 'decimal';
                break;
            default:
                colProp.dt = 'varchar';
                break;
        }
        return colProp;
    }
    static getDataTypeListForUiType(col, idType) {
        switch (col.uidt) {
            case 'ID':
                if (idType === 'AG') {
                    return ['varchar', 'char', 'nchar'];
                }
                else if (idType === 'AI') {
                    return ['int', 'smallint', 'mediumint', 'bigint', 'bit', 'serial'];
                }
                else {
                    return dbTypes;
                }
            case 'ForeignKey':
                return dbTypes;
            case 'SingleLineText':
            case 'LongText':
            case 'Collaborator':
            case 'GeoData':
                return [
                    'text',
                    'mediumtext',
                    'longtext',
                    'varchar',
                    'char',
                    'nchar',
                    'tinytext',
                ];
            case 'Attachment':
                return [
                    'json',
                    'char',
                    'varchar',
                    'nchar',
                    'text',
                    'tinytext',
                    'mediumtext',
                    'longtext',
                ];
            case 'JSON':
                return ['json', 'text', 'tinytext', 'mediumtext', 'longtext'];
            case 'Checkbox':
                return [
                    'int',
                    'smallint',
                    'mediumint',
                    'bigint',
                    'bit',
                    'boolean',
                    'serial',
                    'tinyint',
                ];
            case 'MultiSelect':
                return ['set', 'text', 'tinytext', 'mediumtext', 'longtext'];
            case 'SingleSelect':
                return ['enum', 'text', 'tinytext', 'mediumtext', 'longtext'];
            case 'Year':
                return ['year'];
            case 'Time':
                return ['time'];
            case 'PhoneNumber':
            case 'Email':
                return ['varchar'];
            case 'URL':
                return ['text', 'tinytext', 'mediumtext', 'longtext', 'varchar'];
            case 'Number':
                return [
                    'int',
                    'smallint',
                    'mediumint',
                    'bigint',
                    'bit',
                    'float',
                    'decimal',
                    'double',
                    'serial',
                ];
            case 'Decimal':
                return ['float', 'decimal', 'double', 'serial'];
            case 'Currency':
                return [
                    'decimal',
                    'float',
                    'double',
                    'serial',
                    'int',
                    'smallint',
                    'mediumint',
                    'bigint',
                    'bit',
                ];
            case 'Percent':
                return [
                    'decimal',
                    'float',
                    'double',
                    'serial',
                    'int',
                    'smallint',
                    'mediumint',
                    'bigint',
                    'bit',
                ];
            case 'Duration':
                return [
                    'decimal',
                    'float',
                    'double',
                    'serial',
                    'int',
                    'smallint',
                    'mediumint',
                    'bigint',
                    'bit',
                ];
            case 'Rating':
                return [
                    'decimal',
                    'float',
                    'double',
                    'serial',
                    'int',
                    'smallint',
                    'mediumint',
                    'bigint',
                    'bit',
                ];
            case 'Formula':
            case 'Button':
                return [
                    'char',
                    'varchar',
                    'nchar',
                    'text',
                    'tinytext',
                    'mediumtext',
                    'longtext',
                ];
            case 'Rollup':
                return ['varchar'];
            case 'Count':
                return ['int', 'smallint', 'mediumint', 'bigint', 'serial'];
            case 'Lookup':
                return ['varchar'];
            case 'Date':
                return ['date', 'datetime', 'timestamp', 'varchar'];
            case 'DateTime':
            case 'CreatedTime':
            case 'LastModifiedTime':
                return ['datetime', 'timestamp', 'varchar'];
            case 'AutoNumber':
                return ['int', 'smallint', 'mediumint', 'bigint'];
            case 'Barcode':
                return ['varchar'];
            case 'Geometry':
                return [
                    'geometry',
                    'point',
                    'linestring',
                    'polygon',
                    'multipoint',
                    'multilinestring',
                    'multipolygon',
                ];
            default:
                return dbTypes;
        }
    }
    static getUnsupportedFnList() {
        return [
            'COUNTA',
            'COUNT',
            'DATESTR',
            'ARRAYSORT',
            'ARRAYUNIQUE',
            'ARRAYSLICE',
            'ARRAYCOMPACT',
        ];
    }
    static getCurrentDateDefault(col) {
        // if database datatype timestamp or datetime then return CURRENT_TIMESTAMP
        if (col.dt &&
            (col.dt.toLowerCase() === 'timestamp' ||
                col.dt.toLowerCase() === 'datetime')) {
            return 'CURRENT_TIMESTAMP';
        }
        // database type is not defined(means column create) and ui datatype is datetime then return CURRENT_TIMESTAMP
        // in this scenario it will create column with datatype timestamp/datetime
        if (!col.dt && col.uidt === UITypes.DateTime) {
            return 'CURRENT_TIMESTAMP';
        }
        return null;
    }
    static isEqual(dataType1, dataType2) {
        if (dataType1 === dataType2)
            return true;
        const abstractType1 = this.getAbstractType({ dt: dataType1 });
        const abstractType2 = this.getAbstractType({ dt: dataType2 });
        if (abstractType1 &&
            abstractType1 === abstractType2 &&
            ['integer', 'float'].includes(abstractType1))
            return true;
        return false;
    }
    //#endregion statics
    //#region methods
    getNewTableColumns() {
        return MysqlUi.getNewTableColumns();
    }
    getNewColumn(suffix) {
        return MysqlUi.getNewColumn(suffix);
    }
    getDefaultLengthForDatatype(type) {
        return MysqlUi.getDefaultLengthForDatatype(type);
    }
    getDefaultLengthIsDisabled(type) {
        return MysqlUi.getDefaultLengthIsDisabled(type);
    }
    getDefaultValueForDatatype(type) {
        return MysqlUi.getDefaultValueForDatatype(type);
    }
    getDefaultScaleForDatatype(type) {
        return MysqlUi.getDefaultScaleForDatatype(type);
    }
    colPropAIDisabled(col, columns) {
        return MysqlUi.colPropAIDisabled(col, columns);
    }
    colPropUNDisabled(col) {
        return MysqlUi.colPropUNDisabled(col);
    }
    onCheckboxChangeAI(col) {
        return MysqlUi.onCheckboxChangeAI(col);
    }
    showScale(columnObj) {
        return MysqlUi.showScale(columnObj);
    }
    removeUnsigned(columns) {
        return MysqlUi.removeUnsigned(columns);
    }
    columnEditable(colObj) {
        return MysqlUi.columnEditable(colObj);
    }
    onCheckboxChangeAU(col) {
        return MysqlUi.onCheckboxChangeAU(col);
    }
    colPropAuDisabled(col) {
        return MysqlUi.colPropAuDisabled(col);
    }
    getAbstractType(col) {
        return MysqlUi.getAbstractType(col);
    }
    getUIType(col) {
        return MysqlUi.getUIType(col);
    }
    getDataTypeForUiType(col, idType) {
        return MysqlUi.getDataTypeForUiType(col, idType);
    }
    getDataTypeListForUiType(col, idType) {
        return MysqlUi.getDataTypeListForUiType(col, idType);
    }
    getUnsupportedFnList() {
        return MysqlUi.getUnsupportedFnList();
    }
    getCurrentDateDefault(_col) {
        return MysqlUi.getCurrentDateDefault(_col);
    }
    isEqual(dataType1, dataType2) {
        return MysqlUi.isEqual(dataType1, dataType2);
    }
    adjustLengthAndScale(newColumn, oldColumn) {
        var _a, _b;
        if (newColumn.dt === 'decimal') {
            // get old column length and default length
            const defaultDtxp = numberize(this.getDefaultLengthForDatatype(newColumn.dt));
            let lastDtxp = defaultDtxp;
            if (oldColumn) {
                lastDtxp = (_a = numberize(oldColumn.dtxp)) !== null && _a !== void 0 ? _a : lastDtxp;
            }
            // get default and new column scale
            const defaultDtxs = numberize(this.getDefaultScaleForDatatype(newColumn.dt));
            const newDtxs = (_b = numberize(newColumn.dtxs)) !== null && _b !== void 0 ? _b : defaultDtxs;
            // get new column length based on scale and old length
            // get whichever is the highest, old column length,
            // default column length or default + precision - default if precision > default
            const newDtxp = Math.max(defaultDtxp, lastDtxp, defaultDtxp + Math.max(newDtxs - defaultDtxs, 0));
            newColumn.dtxp = newDtxp;
            newColumn.dtxs = newDtxs;
        }
    }
    isParsedJsonReturnType(col) {
        var _a;
        return ['json'].includes((_a = col.dt) === null || _a === void 0 ? void 0 : _a.toLowerCase());
    }
    get tableNameLengthLimit() {
        return 64;
    }
}
//# sourceMappingURL=data:application/json;base64,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