var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
import { ViewLockType, ViewSettingOverrideOptions } from './enums';
import { ViewTypes } from './globals';
import { ncIsUndefined } from './is';
/**
 * A comprehensive mapping of all available copy view configuration options.
 *
 * This map defines the metadata and constraints for each type of view configuration that can be copied,
 * including display order, labels, i18n keys, and which view types support each configuration option.
 *
 * @remarks
 * Use this map to:
 * - Get the configuration options available for a specific view type
 * - Determine which view types support a particular configuration option
 * - Access i18n labels and display ordering for UI rendering
 *
 * @example
 * ```ts
 * // Get the field visibility option metadata
 * const fieldVisibilityOption = copyViewConfigOptionMap[ViewSettingOverrideOptions.FIELD_VISIBILITY]
 * // Check if it's supported for grid views
 * const isSupported = fieldVisibilityOption.supportedViewTypes.includes(ViewTypes.GRID)
 * ```
 */
export const copyViewConfigOptionMap = {
    [ViewSettingOverrideOptions.FIELD_VISIBILITY]: {
        order: 1,
        label: 'Field visibility',
        value: ViewSettingOverrideOptions.FIELD_VISIBILITY,
        i18nLabel: `objects.copyViewConfig.${ViewSettingOverrideOptions.FIELD_VISIBILITY}`,
        supportedViewTypes: [
            ViewTypes.GRID,
            ViewTypes.GALLERY,
            ViewTypes.KANBAN,
            ViewTypes.FORM,
            ViewTypes.CALENDAR,
            ViewTypes.MAP,
        ],
        icon: 'eyeSlash',
    },
    [ViewSettingOverrideOptions.FIELD_ORDER]: {
        order: 2,
        label: 'Field order',
        value: ViewSettingOverrideOptions.FIELD_ORDER,
        i18nLabel: `objects.copyViewConfig.${ViewSettingOverrideOptions.FIELD_ORDER}`,
        supportedViewTypes: [
            ViewTypes.GRID,
            ViewTypes.GALLERY,
            ViewTypes.KANBAN,
            ViewTypes.FORM,
            ViewTypes.CALENDAR,
            ViewTypes.MAP,
        ],
        icon: 'ncNumberList',
    },
    [ViewSettingOverrideOptions.COLUMN_WIDTH]: {
        order: 3,
        label: 'Column width',
        value: ViewSettingOverrideOptions.COLUMN_WIDTH,
        i18nLabel: `objects.copyViewConfig.${ViewSettingOverrideOptions.COLUMN_WIDTH}`,
        supportedViewTypes: [ViewTypes.GRID],
        icon: 'columnWidth',
    },
    [ViewSettingOverrideOptions.ROW_HEIGHT]: {
        order: 4,
        label: 'Row height',
        value: ViewSettingOverrideOptions.ROW_HEIGHT,
        i18nLabel: `objects.copyViewConfig.${ViewSettingOverrideOptions.ROW_HEIGHT}`,
        supportedViewTypes: [ViewTypes.GRID],
        icon: 'rowHeight',
    },
    [ViewSettingOverrideOptions.FILTER_CONDITION]: {
        order: 5,
        label: 'Filter condition',
        value: ViewSettingOverrideOptions.FILTER_CONDITION,
        i18nLabel: `objects.copyViewConfig.${ViewSettingOverrideOptions.FILTER_CONDITION}`,
        supportedViewTypes: [
            ViewTypes.GRID,
            ViewTypes.GALLERY,
            ViewTypes.KANBAN,
            ViewTypes.CALENDAR,
            ViewTypes.MAP,
        ],
        icon: 'filter',
    },
    [ViewSettingOverrideOptions.GROUP]: {
        order: 6,
        label: 'Group',
        value: ViewSettingOverrideOptions.GROUP,
        i18nLabel: `objects.copyViewConfig.${ViewSettingOverrideOptions.GROUP}`,
        supportedViewTypes: [ViewTypes.GRID],
        icon: 'group',
    },
    [ViewSettingOverrideOptions.SORT]: {
        order: 7,
        label: 'Sort',
        value: ViewSettingOverrideOptions.SORT,
        i18nLabel: `objects.copyViewConfig.${ViewSettingOverrideOptions.SORT}`,
        supportedViewTypes: [
            ViewTypes.GRID,
            ViewTypes.GALLERY,
            ViewTypes.KANBAN,
            ViewTypes.CALENDAR,
        ],
        icon: 'sort',
    },
    [ViewSettingOverrideOptions.ROW_COLORING]: {
        order: 8,
        label: 'Row coloring',
        value: ViewSettingOverrideOptions.ROW_COLORING,
        i18nLabel: `objects.copyViewConfig.${ViewSettingOverrideOptions.ROW_COLORING}`,
        supportedViewTypes: [
            ViewTypes.GRID,
            ViewTypes.GALLERY,
            ViewTypes.KANBAN,
            ViewTypes.CALENDAR,
        ],
        icon: 'ncPaintRoller',
    },
};
/**
 * Retrieves all copy view configuration options with their availability status for a specific view type.
 *
 * This function returns all available configuration options, marking each as enabled or disabled
 * based on whether the source view type supports that particular configuration.
 * The returned options are sorted by their display order.
 *
 * @param sourceViewType - The view type from which configurations will be copied
 * @param destinationViewType - The view type to which configurations will be copied
 * @returns An array of supported destination view type configuration options with a 'disabled' flag indicating support status, sorted by order
 */
export const getCopyViewConfigOptions = (sourceViewType, destinationViewType) => {
    return Object.values(copyViewConfigOptionMap)
        .filter((option) => {
        if (!destinationViewType)
            return true;
        return option.supportedViewTypes.includes(destinationViewType);
    })
        .map((option) => {
        const { supportedViewTypes } = option, rest = __rest(option, ["supportedViewTypes"]);
        return Object.assign(Object.assign({}, rest), { disabled: !ncIsUndefined(sourceViewType) &&
                !supportedViewTypes.includes(sourceViewType) });
    })
        .sort((a, b) => a.order - b.order);
};
/**
 * Filters a list of copy view configuration types to only include those supported by the source view type.
 *
 * This function validates each configuration type against the source view type's capabilities,
 * removing any unsupported or invalid configuration types from the list.
 *
 * @param viewSettingOverrideOptions - Array of configuration types to be validated
 * @param sourceViewType - The view type from which configurations will be copied
 * @param destinationViewType - The view type to which configurations will be copied
 * @returns A filtered array containing only the configuration types supported by the source view type and destination view type
 *
 * @example
 * ```ts
 * // Validate selected config types for a Form view
 * const selectedTypes = [ViewSettingOverrideOptions.FIELD_VISIBILITY, ViewSettingOverrideOptions.FILTER_CONDITION, ViewSettingOverrideOptions.GROUP]
 * const supportedTypes = extractSupportedViewSettingOverrideOptions(selectedTypes, ViewTypes.FORM, ViewTypes.GRID)
 * // Result: Only FieldVisibility will be included since destination Grid view support filter and group but source Forms don't support Filters or Groups
 * ```
 */
export const extractSupportedViewSettingOverrideOptions = (viewSettingOverrideOptions, sourceViewType, destinationViewType) => {
    // extract destination view type options
    const destinationViewTypeOptions = Object.values(ViewSettingOverrideOptions).filter((option) => {
        if (!destinationViewType)
            return true;
        return copyViewConfigOptionMap[option].supportedViewTypes.includes(destinationViewType);
    });
    // return only options which supported in destination as well as source view type
    return (viewSettingOverrideOptions || []).filter((type) => {
        if (!copyViewConfigOptionMap[type] ||
            !destinationViewTypeOptions.includes(type)) {
            return false;
        }
        return copyViewConfigOptionMap[type].supportedViewTypes.includes(sourceViewType);
    });
};
/**
 * Retrieves all view setting override options supported by a specific view type.
 */
export const getViewSettingOverrideOptionsByViewType = (viewType) => {
    return extractSupportedViewSettingOverrideOptions(Object.values(ViewSettingOverrideOptions), viewType);
};
/**
 * Finds the first non-personal view from an array of views based on optional filters.
 *
 * @param views - Array of views to search through
 * @param options - Filter options
 * @param options.excludeViewType - View type(s) to exclude from the search (single type or array)
 * @param options.includeViewType - View type(s) to include in the search (single type or array)
 * @returns The first non-personal view matching the criteria, or undefined if none found
 *
 * @example
 * // Find first non-personal view
 * const view = getFirstNonPersonalView(views, {});
 *
 * @example
 * // Find first non-personal grid view
 * const gridView = getFirstNonPersonalView(views, {
 *   includeViewType: ViewTypes.GRID
 * });
 *
 * @example
 * // Find first non-personal view excluding gallery and kanban
 * const view = getFirstNonPersonalView(views, {
 *   excludeViewType: [ViewTypes.GALLERY, ViewTypes.KANBAN]
 * });
 */
export const getFirstNonPersonalView = (views, { excludeViewType, includeViewType, } = {}) => {
    if (!(views === null || views === void 0 ? void 0 : views.length))
        return undefined;
    return views
        .sort((a, b) => a.order - b.order)
        .find((view) => {
        // Skip personal views
        if (view.lock_type === ViewLockType.Personal) {
            return false;
        }
        // Exclude specified view types
        if (excludeViewType) {
            const excludeTypes = Array.isArray(excludeViewType)
                ? excludeViewType
                : [excludeViewType];
            if (excludeTypes.includes(view.type)) {
                return false;
            }
        }
        // Include only specified view types
        if (includeViewType) {
            const includeTypes = Array.isArray(includeViewType)
                ? includeViewType
                : [includeViewType];
            if (!includeTypes.includes(view.type)) {
                return false;
            }
        }
        return true;
    });
};
//# sourceMappingURL=data:application/json;base64,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