/*
 * Decompiled with CFR 0.152.
 */
package org.nzbhydra.backup;

import com.google.common.base.Stopwatch;
import com.google.common.base.Throwables;
import jakarta.annotation.PostConstruct;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.persistence.Query;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.spi.FileSystemProvider;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.commons.io.FileUtils;
import org.h2.message.DbException;
import org.nzbhydra.GenericResponse;
import org.nzbhydra.NzbHydra;
import org.nzbhydra.backup.BackupEntry;
import org.nzbhydra.backup.ExtractZipFileVisitor;
import org.nzbhydra.backup.FailedBackupData;
import org.nzbhydra.config.ConfigProvider;
import org.nzbhydra.config.ConfigReaderWriter;
import org.nzbhydra.genericstorage.GenericStorage;
import org.nzbhydra.logging.LoggingMarkers;
import org.nzbhydra.misc.TempFileProvider;
import org.nzbhydra.systemcontrol.SystemControl;
import org.nzbhydra.webaccess.HydraOkHttp3ClientHttpRequestFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aot.hint.annotation.Reflective;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

/*
 * Exception performing whole class analysis ignored.
 */
@Component
public class BackupAndRestore {
    private static final Logger logger = LoggerFactory.getLogger(BackupAndRestore.class);
    private static final Pattern FILE_PATTERN = Pattern.compile("nzbhydra-(\\d{4}-\\d{2}-\\d{2} \\d{2}-\\d{2}-\\d{2})\\.zip");
    private static final DateTimeFormatter DATE_PATTERN = DateTimeFormatter.ofPattern("yyyy-MM-dd HH-mm-ss");
    public static final int MIN_AGE_MINUES_LAST_AUTOMATIC_BACKUP = 30;
    @PersistenceContext
    private EntityManager entityManager;
    @Autowired
    private ConfigProvider configProvider;
    @Autowired
    private HydraOkHttp3ClientHttpRequestFactory requestFactory;
    @Autowired
    private SystemControl systemControl;
    @Autowired
    private TempFileProvider tempFileProvider;
    @Autowired
    private GenericStorage genericStorage;
    private final ConfigReaderWriter configReaderWriter = new ConfigReaderWriter();

    @PostConstruct
    public void init() {
        try {
            FileSystemProvider.installedProviders();
        }
        catch (Exception e) {
            logger.error("Unable to load installed FileSystemProviders");
        }
    }

    @Transactional
    public File backup(boolean triggeredByUser) throws Exception {
        boolean created;
        Stopwatch stopwatch = Stopwatch.createStarted();
        File backupFolder = this.getBackupFolder();
        if (!backupFolder.exists() && !(created = backupFolder.mkdirs())) {
            throw new IOException("Unable to create backup target folder " + backupFolder.getAbsolutePath());
        }
        Optional latest = this.getBackupFilesToDate().entrySet().stream().max(Map.Entry.comparingByValue());
        if (!triggeredByUser && latest.isPresent() && ((LocalDateTime)latest.get().getValue()).isAfter(LocalDateTime.now().minusMinutes(30L))) {
            logger.info("Not creating a new automatic backup because the last one ({}) was just created", (Object)((File)latest.get().getKey()).getName());
            return (File)latest.get().getKey();
        }
        this.deleteOldBackupFiles(backupFolder);
        logger.info("Creating backup");
        File backupZip = new File(backupFolder, "nzbhydra-" + LocalDateTime.now().format(DATE_PATTERN) + ".zip");
        this.backupDatabase(backupZip, triggeredByUser);
        if (!backupZip.exists()) {
            throw new RuntimeException("Export to file " + String.valueOf(backupZip) + " was not executed by database");
        }
        HashMap<String, String> env = new HashMap<String, String>();
        env.put("create", "true");
        URI uri = URI.create("jar:" + String.valueOf(backupZip.toPath().toUri()));
        try (FileSystem fs = FileSystems.newFileSystem(uri, env);){
            Path nf = fs.getPath("nzbhydra.yml", new String[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(nf, StandardCharsets.UTF_8, StandardOpenOption.CREATE);){
                writer.write(this.configReaderWriter.getAsYamlString(this.configProvider.getBaseConfig()));
                logger.debug("Successfully wrote config to backup ZIP");
                this.backupCertificates(fs);
            }
        }
        logger.info("Successfully wrote backup to file {}", (Object)backupZip.getAbsolutePath());
        logger.debug(LoggingMarkers.PERFORMANCE, "Creation of backup took {}ms", (Object)stopwatch.elapsed(TimeUnit.MILLISECONDS));
        return backupZip;
    }

    protected void deleteOldBackupFiles(File backupFolder) {
        if (this.configProvider.getBaseConfig().getMain().getDeleteBackupsAfterWeeks().isPresent()) {
            logger.info("Deleting old backups if any exist");
            Integer backupMaxAgeInWeeks = (Integer)this.configProvider.getBaseConfig().getMain().getDeleteBackupsAfterWeeks().get();
            Map fileToBackupTime = this.getBackupFilesToDate();
            for (Map.Entry entry : fileToBackupTime.entrySet()) {
                File zip = (File)entry.getKey();
                LocalDateTime backupDate = (LocalDateTime)entry.getValue();
                if (!backupDate.isBefore(LocalDateTime.now().minusSeconds(604800L * (long)backupMaxAgeInWeeks.intValue()))) continue;
                boolean successfulNewerBackup = fileToBackupTime.entrySet().stream().anyMatch(x -> x.getKey() != zip && ((LocalDateTime)x.getValue()).isAfter(backupDate));
                if (!successfulNewerBackup) {
                    logger.warn("No successful backup was made after the creation of {}. Will not delete it.", (Object)zip.getAbsolutePath());
                    continue;
                }
                logger.info("Deleting backup file {} because it's older than {} weeks and a newer successful backup exists", (Object)zip, (Object)backupMaxAgeInWeeks);
                boolean deleted = zip.delete();
                if (deleted) continue;
                logger.warn("Unable to delete old backup file {}", (Object)zip.getName());
            }
        }
    }

    private Map<File, LocalDateTime> getBackupFilesToDate() {
        HashMap<File, LocalDateTime> fileToBackupTime = new HashMap<File, LocalDateTime>();
        for (BackupEntry existingBackup : this.getExistingBackups()) {
            fileToBackupTime.put(new File(this.getBackupFolder(), existingBackup.getFilename()), LocalDateTime.ofInstant(existingBackup.getCreationDate(), ZoneId.systemDefault()));
        }
        return fileToBackupTime;
    }

    @Reflective
    protected File getBackupFolder() {
        String backupFolder = this.configProvider.getBaseConfig().getMain().getBackupFolder();
        if (backupFolder.contains(File.separator)) {
            return new File(backupFolder);
        }
        return new File(NzbHydra.getDataFolder(), backupFolder);
    }

    public List<BackupEntry> getExistingBackups() {
        ArrayList<BackupEntry> entries = new ArrayList<BackupEntry>();
        File backupFolder = this.getBackupFolder();
        if (backupFolder == null || !backupFolder.exists() || backupFolder.listFiles() == null) {
            return Collections.emptyList();
        }
        for (File file : backupFolder.listFiles((dir, name) -> name != null && name.startsWith("nzbhydra") && name.endsWith(".zip"))) {
            try {
                entries.add(new BackupEntry(file.getName(), Files.readAttributes(file.toPath(), BasicFileAttributes.class, new LinkOption[0]).creationTime().toInstant()));
            }
            catch (IOException e) {
                logger.error("Unable to read creation date of file {}", (Object)file, (Object)e);
            }
        }
        entries.sort((o1, o2) -> o2.getCreationDate().compareTo(o1.getCreationDate()));
        return entries;
    }

    private void backupDatabase(File targetFile, boolean triggeredByUsed) {
        String tempPath;
        File tempFile;
        try {
            tempFile = this.tempFileProvider.getTempFile("databaseTest", ".zip");
            tempPath = tempFile.getAbsolutePath().replace("\\", "/");
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        try {
            String formattedFilepath = targetFile.getAbsolutePath().replace("\\", "/");
            logger.info("Backing up database to {}", (Object)formattedFilepath);
            Query nativeQuery = this.entityManager.createNativeQuery("SCRIPT TO '%s';".formatted(tempPath));
            this.entityManager.createNativeQuery("BACKUP TO '" + formattedFilepath + "';").executeUpdate();
            List resultList = nativeQuery.getResultList();
            logger.debug("Wrote database backup data to {}", (Object)targetFile.getAbsolutePath());
        }
        catch (Exception e) {
            logger.info("Deleting invalid backup file {}", (Object)targetFile);
            targetFile.delete();
            tempFile.delete();
            if (!triggeredByUsed) {
                String dbExceptionMessage = Throwables.getCausalChain((Throwable)e).stream().filter(x -> x instanceof DbException).findFirst().map(Throwable::getMessage).orElse(null);
                this.genericStorage.save("FAILED_BACKUP", (Serializable)new FailedBackupData(dbExceptionMessage));
            }
            throw e;
        }
        finally {
            tempFile.delete();
        }
    }

    private void backupCertificates(FileSystem fileSystem) throws IOException {
        File certificatesFolder = new File(NzbHydra.getDataFolder(), "certificates");
        if (!certificatesFolder.exists()) {
            return;
        }
        logger.info("Backing up certificates folder");
        Files.createDirectory(fileSystem.getPath("certificates", new String[0]), new FileAttribute[0]);
        for (File file : certificatesFolder.listFiles()) {
            try (OutputStream outputStream = Files.newOutputStream(fileSystem.getPath("certificates", file.getName()), StandardOpenOption.CREATE);){
                Files.copy(file.toPath(), outputStream);
                logger.debug("Successfully wrote file {} to backup ZIP", (Object)file.getName());
            }
        }
    }

    public GenericResponse restore(String filename) {
        try {
            File backupFile = new File(this.getBackupFolder(), filename);
            if (!backupFile.exists()) {
                return GenericResponse.notOk((String)"File does not exist");
            }
            this.restoreFromFile(backupFile);
        }
        catch (Exception e) {
            logger.error("Error while restoring", (Throwable)e);
            return GenericResponse.notOk((String)("Error while restoring: " + e.getMessage()));
        }
        return GenericResponse.ok();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public GenericResponse restoreFromFile(InputStream inputStream) {
        File tempFile = null;
        try {
            tempFile = this.tempFileProvider.getTempFile("restore", ".zip");
            FileUtils.copyInputStreamToFile((InputStream)inputStream, (File)tempFile);
            tempFile.deleteOnExit();
            this.restoreFromFile(tempFile);
            GenericResponse genericResponse = GenericResponse.ok();
            return genericResponse;
        }
        catch (Exception e) {
            logger.error("Error while restoring", (Throwable)e);
            GenericResponse genericResponse = GenericResponse.notOk((String)("Error while restoring: " + e.getMessage()));
            return genericResponse;
        }
        finally {
            if (tempFile != null) {
                tempFile.delete();
            }
        }
    }

    protected void restoreFromFile(File backupFile) throws IOException {
        File restoreFolder = new File(NzbHydra.getDataFolder(), "restore");
        if (restoreFolder.exists()) {
            FileUtils.cleanDirectory((File)restoreFolder);
            FileUtils.deleteDirectory((File)restoreFolder);
        } else {
            boolean created = restoreFolder.mkdirs();
            if (!created) {
                throw new IOException("Unable to create folder " + restoreFolder.getCanonicalPath());
            }
        }
        BackupAndRestore.extractZip((File)backupFile, (File)restoreFolder);
        logger.info("Successfully extracted backup file for wrapper. Restarting.");
        this.systemControl.exitWithReturnCode(33);
    }

    private static void extractZip(File zipFile, File targetFolder) throws IOException {
        logger.info("Extracting from file {} to folder {}", (Object)zipFile.getCanonicalPath(), (Object)targetFolder.getCanonicalPath());
        Path dest = targetFolder.toPath().toAbsolutePath().normalize();
        targetFolder.mkdir();
        HashMap env = new HashMap();
        try (FileSystem zipFileSystem = FileSystems.newFileSystem(URI.create("jar:" + String.valueOf(zipFile.toPath().toUri())), env);){
            Path root = zipFileSystem.getPath("/", new String[0]);
            Files.walkFileTree(root, (FileVisitor<? super Path>)new ExtractZipFileVisitor(dest));
        }
        try {
            BackupAndRestore.restoreCertificates((File)targetFolder);
        }
        catch (IOException e) {
            logger.error("Unable to restore certificates", (Throwable)e);
        }
    }

    private static void restoreCertificates(File targetFolder) throws IOException {
        File certificatesBackupFolder = new File(targetFolder, "certificates");
        if (certificatesBackupFolder.exists()) {
            boolean created;
            File certificatesFolder = new File(NzbHydra.getDataFolder(), "certificates");
            if (!certificatesFolder.exists() && !(created = certificatesFolder.mkdir())) {
                logger.error("Unable to create folder {}", (Object)certificatesFolder);
                return;
            }
            for (File file : certificatesBackupFolder.listFiles()) {
                Files.move(file.toPath(), new File(certificatesFolder, file.getName()).toPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
            }
        }
    }
}

