/*
 * Decompiled with CFR 0.152.
 */
package org.nzbhydra.searching.cleanup;

import com.google.common.base.Stopwatch;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import javax.sql.DataSource;
import org.nzbhydra.config.ConfigProvider;
import org.nzbhydra.logging.LoggingMarkers;
import org.nzbhydra.misc.Sleep;
import org.nzbhydra.searching.cleanup.HistoryCleanupTask;
import org.nzbhydra.tasks.HydraTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class HistoryCleanupTask {
    @Autowired
    private ConfigProvider configProvider;
    @Autowired
    private DataSource dataSource;
    private static final Logger logger = LoggerFactory.getLogger(HistoryCleanupTask.class);
    private static final long HOUR = 3600000L;

    @HydraTask(configId="deleteOldHistory", name="Delete old history entries", interval=3600000L)
    public void deleteOldResults() {
        boolean doDelete;
        Stopwatch stopwatch = Stopwatch.createStarted();
        Integer keepSearchResultsForWeeks = this.configProvider.getBaseConfig().getMain().getKeepHistoryForWeeks();
        Integer keepStatsForWeeks = this.configProvider.getBaseConfig().getMain().getKeepStatsForWeeks();
        boolean keepHistory = this.configProvider.getBaseConfig().getMain().isKeepHistory();
        boolean bl = doDelete = keepSearchResultsForWeeks != null || keepStatsForWeeks != null || !keepHistory;
        if (!doDelete) {
            logger.debug(LoggingMarkers.HISTORY_CLEANUP, "No value set to determine how long history entries should be kept");
            return;
        }
        try (Connection connection = this.dataSource.getConnection();){
            Optional optionalHighestId;
            Instant deleteOlderThanHistory;
            if (keepHistory) {
                int keepDays = keepSearchResultsForWeeks * 7;
                logger.info("Starting deletion of old history entries older than {} days", (Object)keepDays);
                deleteOlderThanHistory = Instant.now().minus(keepDays, ChronoUnit.DAYS);
                optionalHighestId = this.getIdBefore(deleteOlderThanHistory, "SEARCH", ASC_DESC.DESC, connection);
            } else {
                optionalHighestId = Optional.of(Integer.MAX_VALUE);
                deleteOlderThanHistory = Instant.now();
            }
            if (optionalHighestId.isPresent()) {
                int highestId = optionalHighestId.get();
                logger.debug(LoggingMarkers.HISTORY_CLEANUP, "Will delete all entries for search IDs lower than {}", (Object)highestId);
                this.deleteOldIdentifiers(Integer.valueOf(highestId), connection);
                this.deleteOldIndexerSearches(Integer.valueOf(highestId), connection);
            }
            Instant deleteOlderThanStats = keepStatsForWeeks != null ? Instant.now().minus(keepStatsForWeeks * 7, ChronoUnit.DAYS) : deleteOlderThanHistory;
            this.deleteOldIndexerApiAccesses(deleteOlderThanStats, connection);
            if (optionalHighestId.isPresent()) {
                this.deleteOldSearches((Integer)optionalHighestId.get(), connection);
            }
            if ((optionalHighestId = this.getIdBefore(deleteOlderThanHistory, "INDEXERNZBDOWNLOAD", ASC_DESC.DESC, connection)).isPresent()) {
                this.deleteOldDownloads((Integer)optionalHighestId.get(), connection);
            }
        }
        catch (SQLException e) {
            logger.error("Error while executing SQL", (Throwable)e);
        }
        logger.info("Deletion of old history entries finished");
        logger.debug(LoggingMarkers.PERFORMANCE, "Cleanup of history took {}ms", (Object)stopwatch.elapsed(TimeUnit.MILLISECONDS));
    }

    public void deleteOldIndexerSearches(Integer searchId, Connection connection) {
        logger.debug(LoggingMarkers.HISTORY_CLEANUP, "Deleting old indexer searches");
        this.deleteOldEntries(searchId.intValue(), "delete from INDEXERSEARCH where SEARCH_ENTITY_ID < ? and rownum() < 10000", "Deleted {} indexer searches from database", connection);
    }

    public void deleteOldIdentifiers(Integer searchId, Connection connection) {
        int identifierId;
        logger.debug(LoggingMarkers.HISTORY_CLEANUP, "Deleting old identifiers");
        this.deleteOldEntries(searchId.intValue(), "delete from SEARCH_IDENTIFIERS where SEARCH_ENTITY_ID < ? and rownum() < 10000", "Deleted {} search identifiers from database", connection);
        try (PreparedStatement statement = connection.prepareStatement("select IDENTIFIERS_ID from SEARCH_IDENTIFIERS order by IDENTIFIERS_ID asc limit 1;");){
            ResultSet resultSet = statement.executeQuery();
            if (!resultSet.next()) {
                logger.debug(LoggingMarkers.HISTORY_CLEANUP, "No older identifiers to delete");
                return;
            }
            identifierId = resultSet.getInt(1);
        }
        catch (SQLException e) {
            logger.error("Error while executing SQL", (Throwable)e);
            return;
        }
        logger.debug(LoggingMarkers.HISTORY_CLEANUP, "Deleting old identifier key value pairs");
        this.deleteOldEntries(identifierId, "delete from IDENTIFIER_KEY_VALUE_PAIR where ID < ? and rownum() < 10000", "Deleted {} identifier key value pairs from database", connection);
    }

    public void deleteOldIndexerApiAccesses(Instant deleteOlderThan, Connection connection) {
        logger.debug(LoggingMarkers.HISTORY_CLEANUP, "Deleting old indexer API accesses");
        Optional optionalId = this.getIdBefore(deleteOlderThan, "INDEXERAPIACCESS", ASC_DESC.DESC, connection);
        if (optionalId.isEmpty()) {
            logger.debug(LoggingMarkers.HISTORY_CLEANUP, "No older indexer API accesses to delete");
            return;
        }
        this.deleteOldEntries(((Integer)optionalId.get()).intValue(), "delete from INDEXERAPIACCESS where ID < ? and rownum() < 10000", "Deleted {} indexer API accesses from database", connection);
    }

    public void deleteOldSearches(Integer searchId, Connection connection) {
        logger.debug(LoggingMarkers.HISTORY_CLEANUP, "Deleting old searches");
        this.deleteOldEntries(searchId.intValue(), "delete from SEARCH where ID < ? and rownum() < 10000", "Deleted {} searches from database", connection);
    }

    public void deleteOldDownloads(Integer downloadId, Connection connection) {
        logger.debug(LoggingMarkers.HISTORY_CLEANUP, "Deleting old downloads");
        this.deleteOldEntries(downloadId.intValue(), "delete from indexernzbdownload where id < ? ", "Deleted {} downloads from database", connection);
        try (PreparedStatement statement = connection.prepareStatement("delete from INDEXERNZBDOWNLOAD where SEARCH_RESULT_ID is null");){
            statement.executeUpdate();
        }
        catch (SQLException e) {
            logger.error("Error while executing SQL", (Throwable)e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Optional<Integer> getIdBefore(Instant deleteOlderThan, String tableName, ASC_DESC ascDesc, Connection connection) {
        try (PreparedStatement statement = connection.prepareStatement("select t.id from " + tableName + " t where t.time < ? order by id " + String.valueOf(ascDesc) + " limit 1");){
            statement.setTimestamp(1, new Timestamp(deleteOlderThan.toEpochMilli()));
            ResultSet resultSet = statement.executeQuery();
            if (!resultSet.next()) {
                logger.debug(LoggingMarkers.HISTORY_CLEANUP, "Did not find any entry in table {} older than {}", (Object)tableName, (Object)deleteOlderThan);
                Optional<Integer> optional2 = Optional.empty();
                return optional2;
            }
            Optional<Integer> optional = Optional.of(resultSet.getInt(1));
            return optional;
        }
        catch (SQLException e) {
            logger.error("Error while executing SQL", (Throwable)e);
            return Optional.empty();
        }
    }

    public void deleteOldEntries(int lowerThan, String sql, String loggerMessage, Connection connection) {
        try {
            int deletedEntities = 0;
            try (PreparedStatement statement = connection.prepareStatement(sql);){
                int lastDeletedEntities;
                do {
                    statement.setInt(1, lowerThan);
                    lastDeletedEntities = statement.executeUpdate();
                    connection.commit();
                    Sleep.sleep((long)20L);
                    logger.debug(LoggingMarkers.HISTORY_CLEANUP, loggerMessage, (Object)(deletedEntities += lastDeletedEntities));
                } while (lastDeletedEntities > 0);
            }
        }
        catch (SQLException e) {
            logger.error("Error while executing SQL", (Throwable)e);
        }
    }
}

