"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const settings_1 = require("../lib/settings");
const logger_1 = __importDefault(require("../logger"));
const plex_api_1 = __importDefault(require("plex-api"));
class PlexAPI {
    constructor({ plexToken, plexSettings, timeout, }) {
        const settings = (0, settings_1.getSettings)();
        let settingsPlex;
        plexSettings
            ? (settingsPlex = plexSettings)
            : (settingsPlex = (0, settings_1.getSettings)().plex);
        this.plexClient = new plex_api_1.default({
            hostname: settingsPlex.ip,
            port: settingsPlex.port,
            https: settingsPlex.useSsl,
            timeout: timeout,
            token: plexToken,
            authenticator: {
                authenticate: (_plexApi, cb) => {
                    if (!plexToken) {
                        return cb('Plex Token not found!');
                    }
                    cb(undefined, plexToken);
                },
            },
            // requestOptions: {
            //   includeChildren: 1,
            // },
            options: {
                identifier: settings.clientId,
                product: 'Overseerr',
                deviceName: 'Overseerr',
                platform: 'Overseerr',
            },
        });
    }
    async getStatus() {
        return await this.plexClient.query('/');
    }
    async getLibraries() {
        const response = await this.plexClient.query('/library/sections');
        return response.MediaContainer.Directory;
    }
    async syncLibraries() {
        const settings = (0, settings_1.getSettings)();
        try {
            const libraries = await this.getLibraries();
            const newLibraries = libraries
                // Remove libraries that are not movie or show
                .filter((library) => library.type === 'movie' || library.type === 'show')
                // Remove libraries that do not have a metadata agent set (usually personal video libraries)
                .filter((library) => library.agent !== 'com.plexapp.agents.none')
                .map((library) => {
                const existing = settings.plex.libraries.find((l) => l.id === library.key && l.name === library.title);
                return {
                    id: library.key,
                    name: library.title,
                    enabled: existing?.enabled ?? false,
                    type: library.type,
                    lastScan: existing?.lastScan,
                };
            });
            settings.plex.libraries = newLibraries;
        }
        catch (e) {
            logger_1.default.error('Failed to fetch Plex libraries', {
                label: 'Plex API',
                message: e.message,
            });
            settings.plex.libraries = [];
        }
        settings.save();
    }
    async getLibraryContents(id, { offset = 0, size = 50 } = {}) {
        const response = await this.plexClient.query({
            uri: `/library/sections/${id}/all?includeGuids=1`,
            extraHeaders: {
                'X-Plex-Container-Start': `${offset}`,
                'X-Plex-Container-Size': `${size}`,
            },
        });
        return {
            totalSize: response.MediaContainer.totalSize,
            items: response.MediaContainer.Metadata ?? [],
        };
    }
    async getMetadata(key, options = {}) {
        const response = await this.plexClient.query(`/library/metadata/${key}${options.includeChildren ? '?includeChildren=1' : ''}`);
        return response.MediaContainer.Metadata[0];
    }
    async getChildrenMetadata(key) {
        const response = await this.plexClient.query(`/library/metadata/${key}/children`);
        return response.MediaContainer.Metadata;
    }
    async getRecentlyAdded(id, options = {
        addedAt: Date.now() - 1000 * 60 * 60,
    }, mediaType) {
        const response = await this.plexClient.query({
            uri: `/library/sections/${id}/all?type=${mediaType === 'show' ? '4' : '1'}&sort=addedAt%3Adesc&addedAt>>=${Math.floor(options.addedAt / 1000)}`,
            extraHeaders: {
                'X-Plex-Container-Start': `0`,
                'X-Plex-Container-Size': `500`,
            },
        });
        return response.MediaContainer.Metadata;
    }
}
exports.default = PlexAPI;
