"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var MediaRequest_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MediaRequest = exports.NoSeasonsAvailableError = exports.DuplicateMediaRequestError = exports.QuotaRestrictedError = exports.RequestPermissionError = void 0;
const radarr_1 = __importDefault(require("../api/servarr/radarr"));
const sonarr_1 = __importDefault(require("../api/servarr/sonarr"));
const themoviedb_1 = __importDefault(require("../api/themoviedb"));
const constants_1 = require("../api/themoviedb/constants");
const media_1 = require("../constants/media");
const datasource_1 = require("../datasource");
const notifications_1 = __importStar(require("../lib/notifications"));
const permissions_1 = require("../lib/permissions");
const settings_1 = require("../lib/settings");
const logger_1 = __importDefault(require("../logger"));
const lodash_1 = require("lodash");
const typeorm_1 = require("typeorm");
const Media_1 = __importDefault(require("./Media"));
const SeasonRequest_1 = __importDefault(require("./SeasonRequest"));
const User_1 = require("./User");
class RequestPermissionError extends Error {
}
exports.RequestPermissionError = RequestPermissionError;
class QuotaRestrictedError extends Error {
}
exports.QuotaRestrictedError = QuotaRestrictedError;
class DuplicateMediaRequestError extends Error {
}
exports.DuplicateMediaRequestError = DuplicateMediaRequestError;
class NoSeasonsAvailableError extends Error {
}
exports.NoSeasonsAvailableError = NoSeasonsAvailableError;
let MediaRequest = MediaRequest_1 = class MediaRequest {
    static async request(requestBody, user, options = {}) {
        const tmdb = new themoviedb_1.default();
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        const requestRepository = (0, datasource_1.getRepository)(MediaRequest_1);
        const userRepository = (0, datasource_1.getRepository)(User_1.User);
        let requestUser = user;
        if (requestBody.userId &&
            !requestUser.hasPermission([
                permissions_1.Permission.MANAGE_USERS,
                permissions_1.Permission.MANAGE_REQUESTS,
            ])) {
            throw new RequestPermissionError('You do not have permission to modify the request user.');
        }
        else if (requestBody.userId) {
            requestUser = await userRepository.findOneOrFail({
                where: { id: requestBody.userId },
            });
        }
        if (!requestUser) {
            throw new Error('User missing from request context.');
        }
        if (requestBody.mediaType === media_1.MediaType.MOVIE &&
            !requestUser.hasPermission(requestBody.is4k
                ? [permissions_1.Permission.REQUEST_4K, permissions_1.Permission.REQUEST_4K_MOVIE]
                : [permissions_1.Permission.REQUEST, permissions_1.Permission.REQUEST_MOVIE], {
                type: 'or',
            })) {
            throw new RequestPermissionError(`You do not have permission to make ${requestBody.is4k ? '4K ' : ''}movie requests.`);
        }
        else if (requestBody.mediaType === media_1.MediaType.TV &&
            !requestUser.hasPermission(requestBody.is4k
                ? [permissions_1.Permission.REQUEST_4K, permissions_1.Permission.REQUEST_4K_TV]
                : [permissions_1.Permission.REQUEST, permissions_1.Permission.REQUEST_TV], {
                type: 'or',
            })) {
            throw new RequestPermissionError(`You do not have permission to make ${requestBody.is4k ? '4K ' : ''}series requests.`);
        }
        const quotas = await requestUser.getQuota();
        if (requestBody.mediaType === media_1.MediaType.MOVIE && quotas.movie.restricted) {
            throw new QuotaRestrictedError('Movie Quota exceeded.');
        }
        else if (requestBody.mediaType === media_1.MediaType.TV && quotas.tv.restricted) {
            throw new QuotaRestrictedError('Series Quota exceeded.');
        }
        const tmdbMedia = requestBody.mediaType === media_1.MediaType.MOVIE
            ? await tmdb.getMovie({ movieId: requestBody.mediaId })
            : await tmdb.getTvShow({ tvId: requestBody.mediaId });
        let media = await mediaRepository.findOne({
            where: {
                tmdbId: requestBody.mediaId,
                mediaType: requestBody.mediaType,
            },
            relations: ['requests'],
        });
        if (!media) {
            media = new Media_1.default({
                tmdbId: tmdbMedia.id,
                tvdbId: requestBody.tvdbId ?? tmdbMedia.external_ids.tvdb_id,
                status: !requestBody.is4k ? media_1.MediaStatus.PENDING : media_1.MediaStatus.UNKNOWN,
                status4k: requestBody.is4k ? media_1.MediaStatus.PENDING : media_1.MediaStatus.UNKNOWN,
                mediaType: requestBody.mediaType,
            });
        }
        else {
            if (media.status === media_1.MediaStatus.UNKNOWN && !requestBody.is4k) {
                media.status = media_1.MediaStatus.PENDING;
            }
            if (media.status4k === media_1.MediaStatus.UNKNOWN && requestBody.is4k) {
                media.status4k = media_1.MediaStatus.PENDING;
            }
        }
        const existing = await requestRepository
            .createQueryBuilder('request')
            .leftJoin('request.media', 'media')
            .leftJoinAndSelect('request.requestedBy', 'user')
            .where('request.is4k = :is4k', { is4k: requestBody.is4k })
            .andWhere('media.tmdbId = :tmdbId', { tmdbId: tmdbMedia.id })
            .andWhere('media.mediaType = :mediaType', {
            mediaType: requestBody.mediaType,
        })
            .getMany();
        if (existing && existing.length > 0) {
            // If there is an existing movie request that isn't declined, don't allow a new one.
            if (requestBody.mediaType === media_1.MediaType.MOVIE &&
                existing[0].status !== media_1.MediaRequestStatus.DECLINED) {
                logger_1.default.warn('Duplicate request for media blocked', {
                    tmdbId: tmdbMedia.id,
                    mediaType: requestBody.mediaType,
                    is4k: requestBody.is4k,
                    label: 'Media Request',
                });
                throw new DuplicateMediaRequestError('Request for this media already exists.');
            }
            // If an existing auto-request for this media exists from the same user,
            // don't allow a new one.
            if (existing.find((r) => r.requestedBy.id === requestUser.id && r.isAutoRequest)) {
                throw new DuplicateMediaRequestError('Auto-request for this media and user already exists.');
            }
        }
        if (requestBody.mediaType === media_1.MediaType.MOVIE) {
            await mediaRepository.save(media);
            const request = new MediaRequest_1({
                type: media_1.MediaType.MOVIE,
                media,
                requestedBy: requestUser,
                // If the user is an admin or has the "auto approve" permission, automatically approve the request
                status: user.hasPermission([
                    requestBody.is4k
                        ? permissions_1.Permission.AUTO_APPROVE_4K
                        : permissions_1.Permission.AUTO_APPROVE,
                    requestBody.is4k
                        ? permissions_1.Permission.AUTO_APPROVE_4K_MOVIE
                        : permissions_1.Permission.AUTO_APPROVE_MOVIE,
                    permissions_1.Permission.MANAGE_REQUESTS,
                ], { type: 'or' })
                    ? media_1.MediaRequestStatus.APPROVED
                    : media_1.MediaRequestStatus.PENDING,
                modifiedBy: user.hasPermission([
                    requestBody.is4k
                        ? permissions_1.Permission.AUTO_APPROVE_4K
                        : permissions_1.Permission.AUTO_APPROVE,
                    requestBody.is4k
                        ? permissions_1.Permission.AUTO_APPROVE_4K_MOVIE
                        : permissions_1.Permission.AUTO_APPROVE_MOVIE,
                    permissions_1.Permission.MANAGE_REQUESTS,
                ], { type: 'or' })
                    ? user
                    : undefined,
                is4k: requestBody.is4k,
                serverId: requestBody.serverId,
                profileId: requestBody.profileId,
                rootFolder: requestBody.rootFolder,
                tags: requestBody.tags,
                isAutoRequest: options.isAutoRequest ?? false,
            });
            await requestRepository.save(request);
            return request;
        }
        else {
            const tmdbMediaShow = tmdbMedia;
            const requestedSeasons = requestBody.seasons === 'all'
                ? tmdbMediaShow.seasons.map((season) => season.season_number)
                : requestBody.seasons;
            let existingSeasons = [];
            // We need to check existing requests on this title to make sure we don't double up on seasons that were
            // already requested. In the case they were, we just throw out any duplicates but still approve the request.
            // (Unless there are no seasons, in which case we abort)
            if (media.requests) {
                existingSeasons = media.requests
                    .filter((request) => request.is4k === requestBody.is4k &&
                    request.status !== media_1.MediaRequestStatus.DECLINED)
                    .reduce((seasons, request) => {
                    const combinedSeasons = request.seasons.map((season) => season.seasonNumber);
                    return [...seasons, ...combinedSeasons];
                }, []);
            }
            // We should also check seasons that are available/partially available but don't have existing requests
            if (media.seasons) {
                existingSeasons = [
                    ...existingSeasons,
                    ...media.seasons
                        .filter((season) => season[requestBody.is4k ? 'status4k' : 'status'] !==
                        media_1.MediaStatus.UNKNOWN)
                        .map((season) => season.seasonNumber),
                ];
            }
            const finalSeasons = requestedSeasons.filter((rs) => !existingSeasons.includes(rs));
            if (finalSeasons.length === 0) {
                throw new NoSeasonsAvailableError('No seasons available to request');
            }
            else if (quotas.tv.limit &&
                finalSeasons.length > (quotas.tv.remaining ?? 0)) {
                throw new QuotaRestrictedError('Series Quota exceeded.');
            }
            await mediaRepository.save(media);
            const request = new MediaRequest_1({
                type: media_1.MediaType.TV,
                media,
                requestedBy: requestUser,
                // If the user is an admin or has the "auto approve" permission, automatically approve the request
                status: user.hasPermission([
                    requestBody.is4k
                        ? permissions_1.Permission.AUTO_APPROVE_4K
                        : permissions_1.Permission.AUTO_APPROVE,
                    requestBody.is4k
                        ? permissions_1.Permission.AUTO_APPROVE_4K_TV
                        : permissions_1.Permission.AUTO_APPROVE_TV,
                    permissions_1.Permission.MANAGE_REQUESTS,
                ], { type: 'or' })
                    ? media_1.MediaRequestStatus.APPROVED
                    : media_1.MediaRequestStatus.PENDING,
                modifiedBy: user.hasPermission([
                    requestBody.is4k
                        ? permissions_1.Permission.AUTO_APPROVE_4K
                        : permissions_1.Permission.AUTO_APPROVE,
                    requestBody.is4k
                        ? permissions_1.Permission.AUTO_APPROVE_4K_TV
                        : permissions_1.Permission.AUTO_APPROVE_TV,
                    permissions_1.Permission.MANAGE_REQUESTS,
                ], { type: 'or' })
                    ? user
                    : undefined,
                is4k: requestBody.is4k,
                serverId: requestBody.serverId,
                profileId: requestBody.profileId,
                rootFolder: requestBody.rootFolder,
                languageProfileId: requestBody.languageProfileId,
                tags: requestBody.tags,
                seasons: finalSeasons.map((sn) => new SeasonRequest_1.default({
                    seasonNumber: sn,
                    status: user.hasPermission([
                        requestBody.is4k
                            ? permissions_1.Permission.AUTO_APPROVE_4K
                            : permissions_1.Permission.AUTO_APPROVE,
                        requestBody.is4k
                            ? permissions_1.Permission.AUTO_APPROVE_4K_TV
                            : permissions_1.Permission.AUTO_APPROVE_TV,
                        permissions_1.Permission.MANAGE_REQUESTS,
                    ], { type: 'or' })
                        ? media_1.MediaRequestStatus.APPROVED
                        : media_1.MediaRequestStatus.PENDING,
                })),
                isAutoRequest: options.isAutoRequest ?? false,
            });
            await requestRepository.save(request);
            return request;
        }
    }
    constructor(init) {
        Object.assign(this, init);
    }
    async sendMedia() {
        await Promise.all([this.sendToRadarr(), this.sendToSonarr()]);
    }
    async notifyNewRequest() {
        if (this.status === media_1.MediaRequestStatus.PENDING) {
            const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
            const media = await mediaRepository.findOne({
                where: { id: this.media.id },
            });
            if (!media) {
                logger_1.default.error('Media data not found', {
                    label: 'Media Request',
                    requestId: this.id,
                    mediaId: this.media.id,
                });
                return;
            }
            this.sendNotification(media, notifications_1.Notification.MEDIA_PENDING);
            if (this.isAutoRequest) {
                this.sendNotification(media, notifications_1.Notification.MEDIA_AUTO_REQUESTED);
            }
        }
    }
    /**
     * Notification for approval
     *
     * We only check on AfterUpdate as to not trigger this for
     * auto approved content
     */
    async notifyApprovedOrDeclined(autoApproved = false) {
        if (this.status === media_1.MediaRequestStatus.APPROVED ||
            this.status === media_1.MediaRequestStatus.DECLINED) {
            const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
            const media = await mediaRepository.findOne({
                where: { id: this.media.id },
            });
            if (!media) {
                logger_1.default.error('Media data not found', {
                    label: 'Media Request',
                    requestId: this.id,
                    mediaId: this.media.id,
                });
                return;
            }
            if (media[this.is4k ? 'status4k' : 'status'] === media_1.MediaStatus.AVAILABLE) {
                logger_1.default.warn('Media became available before request was approved. Skipping approval notification', { label: 'Media Request', requestId: this.id, mediaId: this.media.id });
                return;
            }
            this.sendNotification(media, this.status === media_1.MediaRequestStatus.APPROVED
                ? autoApproved
                    ? notifications_1.Notification.MEDIA_AUTO_APPROVED
                    : notifications_1.Notification.MEDIA_APPROVED
                : notifications_1.Notification.MEDIA_DECLINED);
            if (this.status === media_1.MediaRequestStatus.APPROVED &&
                autoApproved &&
                this.isAutoRequest) {
                this.sendNotification(media, notifications_1.Notification.MEDIA_AUTO_REQUESTED);
            }
        }
    }
    async autoapprovalNotification() {
        if (this.status === media_1.MediaRequestStatus.APPROVED) {
            this.notifyApprovedOrDeclined(true);
        }
    }
    async updateParentStatus() {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        const media = await mediaRepository.findOne({
            where: { id: this.media.id },
            relations: { requests: true },
        });
        if (!media) {
            logger_1.default.error('Media data not found', {
                label: 'Media Request',
                requestId: this.id,
                mediaId: this.media.id,
            });
            return;
        }
        const seasonRequestRepository = (0, datasource_1.getRepository)(SeasonRequest_1.default);
        if (this.status === media_1.MediaRequestStatus.APPROVED &&
            // Do not update the status if the item is already partially available or available
            media[this.is4k ? 'status4k' : 'status'] !== media_1.MediaStatus.AVAILABLE &&
            media[this.is4k ? 'status4k' : 'status'] !==
                media_1.MediaStatus.PARTIALLY_AVAILABLE) {
            media[this.is4k ? 'status4k' : 'status'] = media_1.MediaStatus.PROCESSING;
            mediaRepository.save(media);
        }
        if (media.mediaType === media_1.MediaType.MOVIE &&
            this.status === media_1.MediaRequestStatus.DECLINED) {
            media[this.is4k ? 'status4k' : 'status'] = media_1.MediaStatus.UNKNOWN;
            mediaRepository.save(media);
        }
        /**
         * If the media type is TV, and we are declining a request,
         * we must check if its the only pending request and that
         * there the current media status is just pending (meaning no
         * other requests have yet to be approved)
         */
        if (media.mediaType === media_1.MediaType.TV &&
            this.status === media_1.MediaRequestStatus.DECLINED &&
            media.requests.filter((request) => request.status === media_1.MediaRequestStatus.PENDING).length === 0 &&
            media[this.is4k ? 'status4k' : 'status'] === media_1.MediaStatus.PENDING) {
            media[this.is4k ? 'status4k' : 'status'] = media_1.MediaStatus.UNKNOWN;
            mediaRepository.save(media);
        }
        // Approve child seasons if parent is approved
        if (media.mediaType === media_1.MediaType.TV &&
            this.status === media_1.MediaRequestStatus.APPROVED) {
            this.seasons.forEach((season) => {
                season.status = media_1.MediaRequestStatus.APPROVED;
                seasonRequestRepository.save(season);
            });
        }
    }
    async handleRemoveParentUpdate() {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        const fullMedia = await mediaRepository.findOneOrFail({
            where: { id: this.media.id },
            relations: { requests: true },
        });
        if (!fullMedia.requests.some((request) => !request.is4k) &&
            fullMedia.status !== media_1.MediaStatus.AVAILABLE) {
            fullMedia.status = media_1.MediaStatus.UNKNOWN;
        }
        if (!fullMedia.requests.some((request) => request.is4k) &&
            fullMedia.status4k !== media_1.MediaStatus.AVAILABLE) {
            fullMedia.status4k = media_1.MediaStatus.UNKNOWN;
        }
        mediaRepository.save(fullMedia);
    }
    async sendToRadarr() {
        if (this.status === media_1.MediaRequestStatus.APPROVED &&
            this.type === media_1.MediaType.MOVIE) {
            try {
                const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
                const settings = (0, settings_1.getSettings)();
                if (settings.radarr.length === 0 && !settings.radarr[0]) {
                    logger_1.default.info('No Radarr server configured, skipping request processing', {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                    return;
                }
                let radarrSettings = settings.radarr.find((radarr) => radarr.isDefault && radarr.is4k === this.is4k);
                if (this.serverId !== null &&
                    this.serverId >= 0 &&
                    radarrSettings?.id !== this.serverId) {
                    radarrSettings = settings.radarr.find((radarr) => radarr.id === this.serverId);
                    logger_1.default.info(`Request has an override server: ${radarrSettings?.name}`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                }
                if (!radarrSettings) {
                    logger_1.default.warn(`There is no default ${this.is4k ? '4K ' : ''}Radarr server configured. Did you set any of your ${this.is4k ? '4K ' : ''}Radarr servers as default?`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                    return;
                }
                let rootFolder = radarrSettings.activeDirectory;
                let qualityProfile = radarrSettings.activeProfileId;
                let tags = radarrSettings.tags ? [...radarrSettings.tags] : [];
                if (this.rootFolder &&
                    this.rootFolder !== '' &&
                    this.rootFolder !== radarrSettings.activeDirectory) {
                    rootFolder = this.rootFolder;
                    logger_1.default.info(`Request has an override root folder: ${rootFolder}`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                }
                if (this.profileId &&
                    this.profileId !== radarrSettings.activeProfileId) {
                    qualityProfile = this.profileId;
                    logger_1.default.info(`Request has an override quality profile ID: ${qualityProfile}`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                }
                if (this.tags && !(0, lodash_1.isEqual)(this.tags, radarrSettings.tags)) {
                    tags = this.tags;
                    logger_1.default.info(`Request has override tags`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                        tagIds: tags,
                    });
                }
                const tmdb = new themoviedb_1.default();
                const radarr = new radarr_1.default({
                    apiKey: radarrSettings.apiKey,
                    url: radarr_1.default.buildUrl(radarrSettings, '/api/v3'),
                });
                const movie = await tmdb.getMovie({ movieId: this.media.tmdbId });
                const media = await mediaRepository.findOne({
                    where: { id: this.media.id },
                });
                if (!media) {
                    logger_1.default.error('Media data not found', {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                    return;
                }
                if (radarrSettings.tagRequests) {
                    let userTag = (await radarr.getTags()).find((v) => v.label.startsWith(this.requestedBy.id + ' - '));
                    if (!userTag) {
                        logger_1.default.info(`Requester has no active tag. Creating new`, {
                            label: 'Media Request',
                            requestId: this.id,
                            mediaId: this.media.id,
                            userId: this.requestedBy.id,
                            newTag: this.requestedBy.id + ' - ' + this.requestedBy.displayName,
                        });
                        userTag = await radarr.createTag({
                            label: this.requestedBy.id + ' - ' + this.requestedBy.displayName,
                        });
                    }
                    if (userTag.id) {
                        if (!tags?.find((v) => v === userTag?.id)) {
                            tags?.push(userTag.id);
                        }
                    }
                    else {
                        logger_1.default.warn(`Requester has no tag and failed to add one`, {
                            label: 'Media Request',
                            requestId: this.id,
                            mediaId: this.media.id,
                            userId: this.requestedBy.id,
                            radarrServer: radarrSettings.hostname + ':' + radarrSettings.port,
                        });
                    }
                }
                if (media[this.is4k ? 'status4k' : 'status'] === media_1.MediaStatus.AVAILABLE) {
                    logger_1.default.warn('Media already exists, marking request as APPROVED', {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                    const requestRepository = (0, datasource_1.getRepository)(MediaRequest_1);
                    this.status = media_1.MediaRequestStatus.APPROVED;
                    await requestRepository.save(this);
                    return;
                }
                const radarrMovieOptions = {
                    profileId: qualityProfile,
                    qualityProfileId: qualityProfile,
                    rootFolderPath: rootFolder,
                    minimumAvailability: radarrSettings.minimumAvailability,
                    title: movie.title,
                    tmdbId: movie.id,
                    year: Number(movie.release_date.slice(0, 4)),
                    monitored: true,
                    tags,
                    searchNow: !radarrSettings.preventSearch,
                };
                // Run this asynchronously so we don't wait for it on the UI side
                radarr
                    .addMovie(radarrMovieOptions)
                    .then(async (radarrMovie) => {
                    // We grab media again here to make sure we have the latest version of it
                    const media = await mediaRepository.findOne({
                        where: { id: this.media.id },
                    });
                    if (!media) {
                        throw new Error('Media data not found');
                    }
                    media[this.is4k ? 'externalServiceId4k' : 'externalServiceId'] =
                        radarrMovie.id;
                    media[this.is4k ? 'externalServiceSlug4k' : 'externalServiceSlug'] =
                        radarrMovie.titleSlug;
                    media[this.is4k ? 'serviceId4k' : 'serviceId'] = radarrSettings?.id;
                    await mediaRepository.save(media);
                })
                    .catch(async () => {
                    const requestRepository = (0, datasource_1.getRepository)(MediaRequest_1);
                    this.status = media_1.MediaRequestStatus.FAILED;
                    requestRepository.save(this);
                    logger_1.default.warn('Something went wrong sending movie request to Radarr, marking status as FAILED', {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                        radarrMovieOptions,
                    });
                    this.sendNotification(media, notifications_1.Notification.MEDIA_FAILED);
                });
                logger_1.default.info('Sent request to Radarr', {
                    label: 'Media Request',
                    requestId: this.id,
                    mediaId: this.media.id,
                });
            }
            catch (e) {
                logger_1.default.error('Something went wrong sending request to Radarr', {
                    label: 'Media Request',
                    errorMessage: e.message,
                    requestId: this.id,
                    mediaId: this.media.id,
                });
                throw new Error(e.message);
            }
        }
    }
    async sendToSonarr() {
        if (this.status === media_1.MediaRequestStatus.APPROVED &&
            this.type === media_1.MediaType.TV) {
            try {
                const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
                const settings = (0, settings_1.getSettings)();
                if (settings.sonarr.length === 0 && !settings.sonarr[0]) {
                    logger_1.default.warn('No Sonarr server configured, skipping request processing', {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                    return;
                }
                let sonarrSettings = settings.sonarr.find((sonarr) => sonarr.isDefault && sonarr.is4k === this.is4k);
                if (this.serverId !== null &&
                    this.serverId >= 0 &&
                    sonarrSettings?.id !== this.serverId) {
                    sonarrSettings = settings.sonarr.find((sonarr) => sonarr.id === this.serverId);
                    logger_1.default.info(`Request has an override server: ${sonarrSettings?.name}`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                }
                if (!sonarrSettings) {
                    logger_1.default.warn(`There is no default ${this.is4k ? '4K ' : ''}Sonarr server configured. Did you set any of your ${this.is4k ? '4K ' : ''}Sonarr servers as default?`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                    return;
                }
                const media = await mediaRepository.findOne({
                    where: { id: this.media.id },
                    relations: { requests: true },
                });
                if (!media) {
                    throw new Error('Media data not found');
                }
                if (media[this.is4k ? 'status4k' : 'status'] === media_1.MediaStatus.AVAILABLE) {
                    logger_1.default.warn('Media already exists, marking request as APPROVED', {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                    const requestRepository = (0, datasource_1.getRepository)(MediaRequest_1);
                    this.status = media_1.MediaRequestStatus.APPROVED;
                    await requestRepository.save(this);
                    return;
                }
                const tmdb = new themoviedb_1.default();
                const sonarr = new sonarr_1.default({
                    apiKey: sonarrSettings.apiKey,
                    url: sonarr_1.default.buildUrl(sonarrSettings, '/api/v3'),
                });
                const series = await tmdb.getTvShow({ tvId: media.tmdbId });
                const tvdbId = series.external_ids.tvdb_id ?? media.tvdbId;
                if (!tvdbId) {
                    const requestRepository = (0, datasource_1.getRepository)(MediaRequest_1);
                    await mediaRepository.remove(media);
                    await requestRepository.remove(this);
                    throw new Error('TVDB ID not found');
                }
                let seriesType = 'standard';
                // Change series type to anime if the anime keyword is present on tmdb
                if (series.keywords.results.some((keyword) => keyword.id === constants_1.ANIME_KEYWORD_ID)) {
                    seriesType = sonarrSettings.animeSeriesType ?? 'anime';
                }
                let rootFolder = seriesType === 'anime' && sonarrSettings.activeAnimeDirectory
                    ? sonarrSettings.activeAnimeDirectory
                    : sonarrSettings.activeDirectory;
                let qualityProfile = seriesType === 'anime' && sonarrSettings.activeAnimeProfileId
                    ? sonarrSettings.activeAnimeProfileId
                    : sonarrSettings.activeProfileId;
                let languageProfile = seriesType === 'anime' && sonarrSettings.activeAnimeLanguageProfileId
                    ? sonarrSettings.activeAnimeLanguageProfileId
                    : sonarrSettings.activeLanguageProfileId;
                let tags = seriesType === 'anime'
                    ? sonarrSettings.animeTags
                        ? [...sonarrSettings.animeTags]
                        : []
                    : sonarrSettings.tags
                        ? [...sonarrSettings.tags]
                        : [];
                if (this.rootFolder &&
                    this.rootFolder !== '' &&
                    this.rootFolder !== rootFolder) {
                    rootFolder = this.rootFolder;
                    logger_1.default.info(`Request has an override root folder: ${rootFolder}`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                }
                if (this.profileId && this.profileId !== qualityProfile) {
                    qualityProfile = this.profileId;
                    logger_1.default.info(`Request has an override quality profile ID: ${qualityProfile}`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                }
                if (this.languageProfileId &&
                    this.languageProfileId !== languageProfile) {
                    languageProfile = this.languageProfileId;
                    logger_1.default.info(`Request has an override language profile ID: ${languageProfile}`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                    });
                }
                if (this.tags && !(0, lodash_1.isEqual)(this.tags, tags)) {
                    tags = this.tags;
                    logger_1.default.info(`Request has override tags`, {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                        tagIds: tags,
                    });
                }
                if (sonarrSettings.tagRequests) {
                    let userTag = (await sonarr.getTags()).find((v) => v.label.startsWith(this.requestedBy.id + ' - '));
                    if (!userTag) {
                        logger_1.default.info(`Requester has no active tag. Creating new`, {
                            label: 'Media Request',
                            requestId: this.id,
                            mediaId: this.media.id,
                            userId: this.requestedBy.id,
                            newTag: this.requestedBy.id + ' - ' + this.requestedBy.displayName,
                        });
                        userTag = await sonarr.createTag({
                            label: this.requestedBy.id + ' - ' + this.requestedBy.displayName,
                        });
                    }
                    if (userTag.id) {
                        if (!tags?.find((v) => v === userTag?.id)) {
                            tags?.push(userTag.id);
                        }
                    }
                    else {
                        logger_1.default.warn(`Requester has no tag and failed to add one`, {
                            label: 'Media Request',
                            requestId: this.id,
                            mediaId: this.media.id,
                            userId: this.requestedBy.id,
                            sonarrServer: sonarrSettings.hostname + ':' + sonarrSettings.port,
                        });
                    }
                }
                const sonarrSeriesOptions = {
                    profileId: qualityProfile,
                    languageProfileId: languageProfile,
                    rootFolderPath: rootFolder,
                    title: series.name,
                    tvdbid: tvdbId,
                    seasons: this.seasons.map((season) => season.seasonNumber),
                    seasonFolder: sonarrSettings.enableSeasonFolders,
                    seriesType,
                    tags,
                    monitored: true,
                    searchNow: !sonarrSettings.preventSearch,
                };
                // Run this asynchronously so we don't wait for it on the UI side
                sonarr
                    .addSeries(sonarrSeriesOptions)
                    .then(async (sonarrSeries) => {
                    // We grab media again here to make sure we have the latest version of it
                    const media = await mediaRepository.findOne({
                        where: { id: this.media.id },
                        relations: { requests: true },
                    });
                    if (!media) {
                        throw new Error('Media data not found');
                    }
                    media[this.is4k ? 'externalServiceId4k' : 'externalServiceId'] =
                        sonarrSeries.id;
                    media[this.is4k ? 'externalServiceSlug4k' : 'externalServiceSlug'] =
                        sonarrSeries.titleSlug;
                    media[this.is4k ? 'serviceId4k' : 'serviceId'] = sonarrSettings?.id;
                    await mediaRepository.save(media);
                })
                    .catch(async () => {
                    const requestRepository = (0, datasource_1.getRepository)(MediaRequest_1);
                    this.status = media_1.MediaRequestStatus.FAILED;
                    requestRepository.save(this);
                    logger_1.default.warn('Something went wrong sending series request to Sonarr, marking status as FAILED', {
                        label: 'Media Request',
                        requestId: this.id,
                        mediaId: this.media.id,
                        sonarrSeriesOptions,
                    });
                    this.sendNotification(media, notifications_1.Notification.MEDIA_FAILED);
                });
                logger_1.default.info('Sent request to Sonarr', {
                    label: 'Media Request',
                    requestId: this.id,
                    mediaId: this.media.id,
                });
            }
            catch (e) {
                logger_1.default.error('Something went wrong sending request to Sonarr', {
                    label: 'Media Request',
                    errorMessage: e.message,
                    requestId: this.id,
                    mediaId: this.media.id,
                });
                throw new Error(e.message);
            }
        }
    }
    async sendNotification(media, type) {
        const tmdb = new themoviedb_1.default();
        try {
            const mediaType = this.type === media_1.MediaType.MOVIE ? 'Movie' : 'Series';
            let event;
            let notifyAdmin = true;
            let notifySystem = true;
            switch (type) {
                case notifications_1.Notification.MEDIA_APPROVED:
                    event = `${this.is4k ? '4K ' : ''}${mediaType} Request Approved`;
                    notifyAdmin = false;
                    break;
                case notifications_1.Notification.MEDIA_DECLINED:
                    event = `${this.is4k ? '4K ' : ''}${mediaType} Request Declined`;
                    notifyAdmin = false;
                    break;
                case notifications_1.Notification.MEDIA_PENDING:
                    event = `New ${this.is4k ? '4K ' : ''}${mediaType} Request`;
                    break;
                case notifications_1.Notification.MEDIA_AUTO_REQUESTED:
                    event = `${this.is4k ? '4K ' : ''}${mediaType} Request Automatically Submitted`;
                    notifyAdmin = false;
                    notifySystem = false;
                    break;
                case notifications_1.Notification.MEDIA_AUTO_APPROVED:
                    event = `${this.is4k ? '4K ' : ''}${mediaType} Request Automatically Approved`;
                    break;
                case notifications_1.Notification.MEDIA_FAILED:
                    event = `${this.is4k ? '4K ' : ''}${mediaType} Request Failed`;
                    break;
            }
            if (this.type === media_1.MediaType.MOVIE) {
                const movie = await tmdb.getMovie({ movieId: media.tmdbId });
                notifications_1.default.sendNotification(type, {
                    media,
                    request: this,
                    notifyAdmin,
                    notifySystem,
                    notifyUser: notifyAdmin ? undefined : this.requestedBy,
                    event,
                    subject: `${movie.title}${movie.release_date ? ` (${movie.release_date.slice(0, 4)})` : ''}`,
                    message: (0, lodash_1.truncate)(movie.overview, {
                        length: 500,
                        separator: /\s/,
                        omission: '…',
                    }),
                    image: `https://image.tmdb.org/t/p/w600_and_h900_bestv2${movie.poster_path}`,
                });
            }
            else if (this.type === media_1.MediaType.TV) {
                const tv = await tmdb.getTvShow({ tvId: media.tmdbId });
                notifications_1.default.sendNotification(type, {
                    media,
                    request: this,
                    notifyAdmin,
                    notifySystem,
                    notifyUser: notifyAdmin ? undefined : this.requestedBy,
                    event,
                    subject: `${tv.name}${tv.first_air_date ? ` (${tv.first_air_date.slice(0, 4)})` : ''}`,
                    message: (0, lodash_1.truncate)(tv.overview, {
                        length: 500,
                        separator: /\s/,
                        omission: '…',
                    }),
                    image: `https://image.tmdb.org/t/p/w600_and_h900_bestv2${tv.poster_path}`,
                    extra: [
                        {
                            name: 'Requested Seasons',
                            value: this.seasons
                                .map((season) => season.seasonNumber)
                                .join(', '),
                        },
                    ],
                });
            }
        }
        catch (e) {
            logger_1.default.error('Something went wrong sending media notification(s)', {
                label: 'Notifications',
                errorMessage: e.message,
                requestId: this.id,
                mediaId: this.media.id,
            });
        }
    }
};
__decorate([
    (0, typeorm_1.PrimaryGeneratedColumn)(),
    __metadata("design:type", Number)
], MediaRequest.prototype, "id", void 0);
__decorate([
    (0, typeorm_1.Column)({ type: 'integer' }),
    __metadata("design:type", Number)
], MediaRequest.prototype, "status", void 0);
__decorate([
    (0, typeorm_1.ManyToOne)(() => Media_1.default, (media) => media.requests, {
        eager: true,
        onDelete: 'CASCADE',
    }),
    __metadata("design:type", Media_1.default)
], MediaRequest.prototype, "media", void 0);
__decorate([
    (0, typeorm_1.ManyToOne)(() => User_1.User, (user) => user.requests, {
        eager: true,
        onDelete: 'CASCADE',
    }),
    __metadata("design:type", User_1.User)
], MediaRequest.prototype, "requestedBy", void 0);
__decorate([
    (0, typeorm_1.ManyToOne)(() => User_1.User, {
        nullable: true,
        cascade: true,
        eager: true,
        onDelete: 'SET NULL',
    }),
    __metadata("design:type", User_1.User)
], MediaRequest.prototype, "modifiedBy", void 0);
__decorate([
    (0, typeorm_1.CreateDateColumn)(),
    __metadata("design:type", Date)
], MediaRequest.prototype, "createdAt", void 0);
__decorate([
    (0, typeorm_1.UpdateDateColumn)(),
    __metadata("design:type", Date)
], MediaRequest.prototype, "updatedAt", void 0);
__decorate([
    (0, typeorm_1.Column)({ type: 'varchar' }),
    __metadata("design:type", String)
], MediaRequest.prototype, "type", void 0);
__decorate([
    (0, typeorm_1.RelationCount)((request) => request.seasons),
    __metadata("design:type", Number)
], MediaRequest.prototype, "seasonCount", void 0);
__decorate([
    (0, typeorm_1.OneToMany)(() => SeasonRequest_1.default, (season) => season.request, {
        eager: true,
        cascade: true,
    }),
    __metadata("design:type", Array)
], MediaRequest.prototype, "seasons", void 0);
__decorate([
    (0, typeorm_1.Column)({ default: false }),
    __metadata("design:type", Boolean)
], MediaRequest.prototype, "is4k", void 0);
__decorate([
    (0, typeorm_1.Column)({ nullable: true }),
    __metadata("design:type", Number)
], MediaRequest.prototype, "serverId", void 0);
__decorate([
    (0, typeorm_1.Column)({ nullable: true }),
    __metadata("design:type", Number)
], MediaRequest.prototype, "profileId", void 0);
__decorate([
    (0, typeorm_1.Column)({ nullable: true }),
    __metadata("design:type", String)
], MediaRequest.prototype, "rootFolder", void 0);
__decorate([
    (0, typeorm_1.Column)({ nullable: true }),
    __metadata("design:type", Number)
], MediaRequest.prototype, "languageProfileId", void 0);
__decorate([
    (0, typeorm_1.Column)({
        type: 'text',
        nullable: true,
        transformer: {
            from: (value) => {
                if (value) {
                    if (value === 'none') {
                        return [];
                    }
                    return value.split(',').map((v) => Number(v));
                }
                return null;
            },
            to: (value) => {
                if (value) {
                    const finalValue = value.join(',');
                    // We want to keep the actual state of an "empty array" so we use
                    // the keyword "none" to track this.
                    if (!finalValue) {
                        return 'none';
                    }
                    return finalValue;
                }
                return null;
            },
        },
    }),
    __metadata("design:type", Array)
], MediaRequest.prototype, "tags", void 0);
__decorate([
    (0, typeorm_1.Column)({ default: false }),
    __metadata("design:type", Boolean)
], MediaRequest.prototype, "isAutoRequest", void 0);
__decorate([
    (0, typeorm_1.AfterUpdate)(),
    (0, typeorm_1.AfterInsert)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MediaRequest.prototype, "sendMedia", null);
__decorate([
    (0, typeorm_1.AfterInsert)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MediaRequest.prototype, "notifyNewRequest", null);
__decorate([
    (0, typeorm_1.AfterUpdate)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], MediaRequest.prototype, "notifyApprovedOrDeclined", null);
__decorate([
    (0, typeorm_1.AfterInsert)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MediaRequest.prototype, "autoapprovalNotification", null);
__decorate([
    (0, typeorm_1.AfterUpdate)(),
    (0, typeorm_1.AfterInsert)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MediaRequest.prototype, "updateParentStatus", null);
__decorate([
    (0, typeorm_1.AfterRemove)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], MediaRequest.prototype, "handleRemoveParentUpdate", null);
MediaRequest = MediaRequest_1 = __decorate([
    (0, typeorm_1.Entity)(),
    __metadata("design:paramtypes", [Object])
], MediaRequest);
exports.MediaRequest = MediaRequest;
exports.default = MediaRequest;
