"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const themoviedb_1 = __importDefault(require("../../api/themoviedb"));
const media_1 = require("../../constants/media");
const datasource_1 = require("../../datasource");
const Media_1 = __importDefault(require("../../entity/Media"));
const Season_1 = __importDefault(require("../../entity/Season"));
const settings_1 = require("../../lib/settings");
const logger_1 = __importDefault(require("../../logger"));
const asyncLock_1 = __importDefault(require("../../utils/asyncLock"));
const crypto_1 = require("crypto");
// Default scan rates (can be overidden)
const BUNDLE_SIZE = 20;
const UPDATE_RATE = 4 * 1000;
class BaseScanner {
    constructor(scannerName, { updateRate, bundleSize, } = {}) {
        this.progress = 0;
        this.items = [];
        this.totalSize = 0;
        this.enable4kMovie = false;
        this.enable4kShow = false;
        this.running = false;
        this.asyncLock = new asyncLock_1.default();
        this.tmdb = new themoviedb_1.default();
        this.scannerName = scannerName;
        this.bundleSize = bundleSize ?? BUNDLE_SIZE;
        this.updateRate = updateRate ?? UPDATE_RATE;
    }
    async getExisting(tmdbId, mediaType) {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        const existing = await mediaRepository.findOne({
            where: { tmdbId: tmdbId, mediaType },
        });
        return existing;
    }
    async processMovie(tmdbId, { is4k = false, mediaAddedAt, ratingKey, serviceId, externalServiceId, externalServiceSlug, processing = false, title = 'Unknown Title', } = {}) {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        await this.asyncLock.dispatch(tmdbId, async () => {
            const existing = await this.getExisting(tmdbId, media_1.MediaType.MOVIE);
            if (existing) {
                let changedExisting = false;
                if (existing[is4k ? 'status4k' : 'status'] !== media_1.MediaStatus.AVAILABLE) {
                    existing[is4k ? 'status4k' : 'status'] = processing
                        ? media_1.MediaStatus.PROCESSING
                        : media_1.MediaStatus.AVAILABLE;
                    if (mediaAddedAt) {
                        existing.mediaAddedAt = mediaAddedAt;
                    }
                    changedExisting = true;
                }
                if (!changedExisting && !existing.mediaAddedAt && mediaAddedAt) {
                    existing.mediaAddedAt = mediaAddedAt;
                    changedExisting = true;
                }
                if (ratingKey &&
                    existing[is4k ? 'ratingKey4k' : 'ratingKey'] !== ratingKey) {
                    existing[is4k ? 'ratingKey4k' : 'ratingKey'] = ratingKey;
                    changedExisting = true;
                }
                if (serviceId !== undefined &&
                    existing[is4k ? 'serviceId4k' : 'serviceId'] !== serviceId) {
                    existing[is4k ? 'serviceId4k' : 'serviceId'] = serviceId;
                    changedExisting = true;
                }
                if (externalServiceId !== undefined &&
                    existing[is4k ? 'externalServiceId4k' : 'externalServiceId'] !==
                        externalServiceId) {
                    existing[is4k ? 'externalServiceId4k' : 'externalServiceId'] =
                        externalServiceId;
                    changedExisting = true;
                }
                if (externalServiceSlug !== undefined &&
                    existing[is4k ? 'externalServiceSlug4k' : 'externalServiceSlug'] !==
                        externalServiceSlug) {
                    existing[is4k ? 'externalServiceSlug4k' : 'externalServiceSlug'] =
                        externalServiceSlug;
                    changedExisting = true;
                }
                if (changedExisting) {
                    await mediaRepository.save(existing);
                    this.log(`Media for ${title} exists. Changes were detected and the title will be updated.`, 'info');
                }
                else {
                    this.log(`Title already exists and no changes detected for ${title}`);
                }
            }
            else {
                const newMedia = new Media_1.default();
                newMedia.tmdbId = tmdbId;
                newMedia.status =
                    !is4k && !processing
                        ? media_1.MediaStatus.AVAILABLE
                        : !is4k && processing
                            ? media_1.MediaStatus.PROCESSING
                            : media_1.MediaStatus.UNKNOWN;
                newMedia.status4k =
                    is4k && this.enable4kMovie && !processing
                        ? media_1.MediaStatus.AVAILABLE
                        : is4k && this.enable4kMovie && processing
                            ? media_1.MediaStatus.PROCESSING
                            : media_1.MediaStatus.UNKNOWN;
                newMedia.mediaType = media_1.MediaType.MOVIE;
                newMedia.serviceId = !is4k ? serviceId : undefined;
                newMedia.serviceId4k = is4k ? serviceId : undefined;
                newMedia.externalServiceId = !is4k ? externalServiceId : undefined;
                newMedia.externalServiceId4k = is4k ? externalServiceId : undefined;
                newMedia.externalServiceSlug = !is4k ? externalServiceSlug : undefined;
                newMedia.externalServiceSlug4k = is4k ? externalServiceSlug : undefined;
                if (mediaAddedAt) {
                    newMedia.mediaAddedAt = mediaAddedAt;
                }
                if (ratingKey) {
                    newMedia.ratingKey = !is4k ? ratingKey : undefined;
                    newMedia.ratingKey4k =
                        is4k && this.enable4kMovie ? ratingKey : undefined;
                }
                await mediaRepository.save(newMedia);
                this.log(`Saved new media: ${title}`);
            }
        });
    }
    /**
     * processShow takes a TMDB ID and an array of ProcessableSeasons, which
     * should include the total episodes a sesaon has + the total available
     * episodes that each season currently has. Unlike processMovie, this method
     * does not take an `is4k` option. We handle both the 4k _and_ non 4k status
     * in one method.
     *
     * Note: If 4k is not enable, ProcessableSeasons should combine their episode counts
     * into the normal episodes properties and avoid using the 4k properties.
     */
    async processShow(tmdbId, tvdbId, seasons, { mediaAddedAt, ratingKey, serviceId, externalServiceId, externalServiceSlug, is4k = false, title = 'Unknown Title', } = {}) {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        await this.asyncLock.dispatch(tmdbId, async () => {
            const media = await this.getExisting(tmdbId, media_1.MediaType.TV);
            const newSeasons = [];
            const currentStandardSeasonsAvailable = (media?.seasons.filter((season) => season.status === media_1.MediaStatus.AVAILABLE) ?? []).length;
            const current4kSeasonsAvailable = (media?.seasons.filter((season) => season.status4k === media_1.MediaStatus.AVAILABLE) ?? []).length;
            for (const season of seasons) {
                const existingSeason = media?.seasons.find((es) => es.seasonNumber === season.seasonNumber);
                // We update the rating keys in the seasons loop because we need episode counts
                if (media && season.episodes > 0 && media.ratingKey !== ratingKey) {
                    media.ratingKey = ratingKey;
                }
                if (media &&
                    season.episodes4k > 0 &&
                    this.enable4kShow &&
                    media.ratingKey4k !== ratingKey) {
                    media.ratingKey4k = ratingKey;
                }
                if (existingSeason) {
                    // Here we update seasons if they already exist.
                    // If the season is already marked as available, we
                    // force it to stay available (to avoid competing scanners)
                    existingSeason.status =
                        (season.totalEpisodes === season.episodes && season.episodes > 0) ||
                            existingSeason.status === media_1.MediaStatus.AVAILABLE
                            ? media_1.MediaStatus.AVAILABLE
                            : season.episodes > 0
                                ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                : !season.is4kOverride && season.processing
                                    ? media_1.MediaStatus.PROCESSING
                                    : existingSeason.status;
                    // Same thing here, except we only do updates if 4k is enabled
                    existingSeason.status4k =
                        (this.enable4kShow &&
                            season.episodes4k === season.totalEpisodes &&
                            season.episodes4k > 0) ||
                            existingSeason.status4k === media_1.MediaStatus.AVAILABLE
                            ? media_1.MediaStatus.AVAILABLE
                            : this.enable4kShow && season.episodes4k > 0
                                ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                : season.is4kOverride && season.processing
                                    ? media_1.MediaStatus.PROCESSING
                                    : existingSeason.status4k;
                }
                else {
                    newSeasons.push(new Season_1.default({
                        seasonNumber: season.seasonNumber,
                        status: season.totalEpisodes === season.episodes && season.episodes > 0
                            ? media_1.MediaStatus.AVAILABLE
                            : season.episodes > 0
                                ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                : !season.is4kOverride && season.processing
                                    ? media_1.MediaStatus.PROCESSING
                                    : media_1.MediaStatus.UNKNOWN,
                        status4k: this.enable4kShow &&
                            season.totalEpisodes === season.episodes4k &&
                            season.episodes4k > 0
                            ? media_1.MediaStatus.AVAILABLE
                            : this.enable4kShow && season.episodes4k > 0
                                ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                                : season.is4kOverride && season.processing
                                    ? media_1.MediaStatus.PROCESSING
                                    : media_1.MediaStatus.UNKNOWN,
                    }));
                }
            }
            const isAllStandardSeasons = seasons.length &&
                seasons.every((season) => season.episodes === season.totalEpisodes && season.episodes > 0);
            const isAll4kSeasons = seasons.length &&
                seasons.every((season) => season.episodes4k === season.totalEpisodes && season.episodes4k > 0);
            if (media) {
                media.seasons = [...media.seasons, ...newSeasons];
                const newStandardSeasonsAvailable = (media.seasons.filter((season) => season.status === media_1.MediaStatus.AVAILABLE) ?? []).length;
                const new4kSeasonsAvailable = (media.seasons.filter((season) => season.status4k === media_1.MediaStatus.AVAILABLE) ?? []).length;
                // If at least one new season has become available, update
                // the lastSeasonChange field so we can trigger notifications
                if (newStandardSeasonsAvailable > currentStandardSeasonsAvailable) {
                    this.log(`Detected ${newStandardSeasonsAvailable - currentStandardSeasonsAvailable} new standard season(s) for ${title}`, 'debug');
                    media.lastSeasonChange = new Date();
                    if (mediaAddedAt) {
                        media.mediaAddedAt = mediaAddedAt;
                    }
                }
                if (new4kSeasonsAvailable > current4kSeasonsAvailable) {
                    this.log(`Detected ${new4kSeasonsAvailable - current4kSeasonsAvailable} new 4K season(s) for ${title}`, 'debug');
                    media.lastSeasonChange = new Date();
                }
                if (!media.mediaAddedAt && mediaAddedAt) {
                    media.mediaAddedAt = mediaAddedAt;
                }
                if (serviceId !== undefined) {
                    media[is4k ? 'serviceId4k' : 'serviceId'] = serviceId;
                }
                if (externalServiceId !== undefined) {
                    media[is4k ? 'externalServiceId4k' : 'externalServiceId'] =
                        externalServiceId;
                }
                if (externalServiceSlug !== undefined) {
                    media[is4k ? 'externalServiceSlug4k' : 'externalServiceSlug'] =
                        externalServiceSlug;
                }
                // If the show is already available, and there are no new seasons, dont adjust
                // the status
                const shouldStayAvailable = media.status === media_1.MediaStatus.AVAILABLE &&
                    newSeasons.filter((season) => season.status !== media_1.MediaStatus.UNKNOWN)
                        .length === 0;
                const shouldStayAvailable4k = media.status4k === media_1.MediaStatus.AVAILABLE &&
                    newSeasons.filter((season) => season.status4k !== media_1.MediaStatus.UNKNOWN)
                        .length === 0;
                media.status =
                    isAllStandardSeasons || shouldStayAvailable
                        ? media_1.MediaStatus.AVAILABLE
                        : media.seasons.some((season) => season.status === media_1.MediaStatus.PARTIALLY_AVAILABLE ||
                            season.status === media_1.MediaStatus.AVAILABLE)
                            ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                            : !seasons.length ||
                                media.seasons.some((season) => season.status === media_1.MediaStatus.PROCESSING)
                                ? media_1.MediaStatus.PROCESSING
                                : media_1.MediaStatus.UNKNOWN;
                media.status4k =
                    (isAll4kSeasons || shouldStayAvailable4k) && this.enable4kShow
                        ? media_1.MediaStatus.AVAILABLE
                        : this.enable4kShow &&
                            media.seasons.some((season) => season.status4k === media_1.MediaStatus.PARTIALLY_AVAILABLE ||
                                season.status4k === media_1.MediaStatus.AVAILABLE)
                            ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                            : !seasons.length ||
                                media.seasons.some((season) => season.status4k === media_1.MediaStatus.PROCESSING)
                                ? media_1.MediaStatus.PROCESSING
                                : media_1.MediaStatus.UNKNOWN;
                await mediaRepository.save(media);
                this.log(`Updating existing title: ${title}`);
            }
            else {
                const newMedia = new Media_1.default({
                    mediaType: media_1.MediaType.TV,
                    seasons: newSeasons,
                    tmdbId,
                    tvdbId,
                    mediaAddedAt,
                    serviceId: !is4k ? serviceId : undefined,
                    serviceId4k: is4k ? serviceId : undefined,
                    externalServiceId: !is4k ? externalServiceId : undefined,
                    externalServiceId4k: is4k ? externalServiceId : undefined,
                    externalServiceSlug: !is4k ? externalServiceSlug : undefined,
                    externalServiceSlug4k: is4k ? externalServiceSlug : undefined,
                    ratingKey: newSeasons.some((sn) => sn.status === media_1.MediaStatus.PARTIALLY_AVAILABLE ||
                        sn.status === media_1.MediaStatus.AVAILABLE)
                        ? ratingKey
                        : undefined,
                    ratingKey4k: this.enable4kShow &&
                        newSeasons.some((sn) => sn.status4k === media_1.MediaStatus.PARTIALLY_AVAILABLE ||
                            sn.status4k === media_1.MediaStatus.AVAILABLE)
                        ? ratingKey
                        : undefined,
                    status: isAllStandardSeasons
                        ? media_1.MediaStatus.AVAILABLE
                        : newSeasons.some((season) => season.status === media_1.MediaStatus.PARTIALLY_AVAILABLE ||
                            season.status === media_1.MediaStatus.AVAILABLE)
                            ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                            : newSeasons.some((season) => season.status === media_1.MediaStatus.PROCESSING)
                                ? media_1.MediaStatus.PROCESSING
                                : media_1.MediaStatus.UNKNOWN,
                    status4k: isAll4kSeasons && this.enable4kShow
                        ? media_1.MediaStatus.AVAILABLE
                        : this.enable4kShow &&
                            newSeasons.some((season) => season.status4k === media_1.MediaStatus.PARTIALLY_AVAILABLE ||
                                season.status4k === media_1.MediaStatus.AVAILABLE)
                            ? media_1.MediaStatus.PARTIALLY_AVAILABLE
                            : newSeasons.some((season) => season.status4k === media_1.MediaStatus.PROCESSING)
                                ? media_1.MediaStatus.PROCESSING
                                : media_1.MediaStatus.UNKNOWN,
                });
                await mediaRepository.save(newMedia);
                this.log(`Saved ${title}`);
            }
        });
    }
    /**
     * Call startRun from child class whenever a run is starting to
     * ensure required values are set
     *
     * Returns the session ID which is requried for the cleanup method
     */
    startRun() {
        const settings = (0, settings_1.getSettings)();
        const sessionId = (0, crypto_1.randomUUID)();
        this.sessionId = sessionId;
        this.log('Scan starting', 'info', { sessionId });
        this.enable4kMovie = settings.radarr.some((radarr) => radarr.is4k);
        if (this.enable4kMovie) {
            this.log('At least one 4K Radarr server was detected. 4K movie detection is now enabled', 'info');
        }
        this.enable4kShow = settings.sonarr.some((sonarr) => sonarr.is4k);
        if (this.enable4kShow) {
            this.log('At least one 4K Sonarr server was detected. 4K series detection is now enabled', 'info');
        }
        this.running = true;
        return sessionId;
    }
    /**
     * Call at end of run loop to perform cleanup
     */
    endRun(sessionId) {
        if (this.sessionId === sessionId) {
            this.running = false;
        }
    }
    cancel() {
        this.running = false;
    }
    async loop(processFn, { start = 0, end = this.bundleSize, sessionId, } = {}) {
        const slicedItems = this.items.slice(start, end);
        if (!this.running) {
            throw new Error('Sync was aborted.');
        }
        if (this.sessionId !== sessionId) {
            throw new Error('New session was started. Old session aborted.');
        }
        if (start < this.items.length) {
            this.progress = start;
            await this.processItems(processFn, slicedItems);
            await new Promise((resolve, reject) => setTimeout(() => {
                this.loop(processFn, {
                    start: start + this.bundleSize,
                    end: end + this.bundleSize,
                    sessionId,
                })
                    .then(() => resolve())
                    .catch((e) => reject(new Error(e.message)));
            }, this.updateRate));
        }
    }
    async processItems(processFn, items) {
        await Promise.all(items.map(async (item) => {
            await processFn(item);
        }));
    }
    log(message, level = 'debug', optional) {
        logger_1.default[level](message, { label: this.scannerName, ...optional });
    }
    get protectedUpdateRate() {
        return this.updateRate;
    }
    get protectedBundleSize() {
        return this.bundleSize;
    }
}
exports.default = BaseScanner;
