import { Collection, DropTarget, DropTargetDelegate, KeyboardDelegate, Node } from "@react-types/shared";
import { InvalidationContext, Layout, LayoutInfo, Rect, Size, Point } from "@react-stately/virtualizer";
import { Key } from "react";
import { ColumnSize, TableCollection } from "@react-types/table";
import { GridNode } from "@react-types/grid";
import { TableColumnLayout } from "@react-stately/table";
export type ListLayoutOptions<T> = {
    /** The height of a row in px. */
    rowHeight?: number;
    estimatedRowHeight?: number;
    headingHeight?: number;
    estimatedHeadingHeight?: number;
    padding?: number;
    indentationForItem?: (collection: Collection<Node<T>>, key: Key) => number;
    collator?: Intl.Collator;
    loaderHeight?: number;
    placeholderHeight?: number;
    allowDisabledKeyFocus?: boolean;
};
export interface LayoutNode {
    node?: Node<unknown>;
    layoutInfo: LayoutInfo;
    header?: LayoutInfo;
    children?: LayoutNode[];
    validRect: Rect;
}
/**
 * The ListLayout class is an implementation of a collection view {@link Layout}
 * it is used for creating lists and lists with indented sub-lists.
 *
 * To configure a ListLayout, you can use the properties to define the
 * layouts and/or use the method for defining indentation.
 * The {@link ListKeyboardDelegate} extends the existing collection view
 * delegate with an additional method to do this (it uses the same delegate object as
 * the collection view itself).
 */
export class ListLayout<T> extends Layout<Node<T>> implements KeyboardDelegate, DropTargetDelegate {
    protected rowHeight: number;
    protected estimatedRowHeight: number;
    protected headingHeight: number;
    protected estimatedHeadingHeight: number;
    protected padding: number;
    protected indentationForItem?: (collection: Collection<Node<T>>, key: Key) => number;
    protected layoutInfos: Map<Key, LayoutInfo>;
    protected layoutNodes: Map<Key, LayoutNode>;
    protected contentSize: Size;
    collection: Collection<Node<T>>;
    disabledKeys: Set<Key>;
    allowDisabledKeyFocus: boolean;
    isLoading: boolean;
    protected lastWidth: number;
    protected lastCollection: Collection<Node<T>>;
    protected rootNodes: LayoutNode[];
    protected collator: Intl.Collator;
    protected invalidateEverything: boolean;
    protected loaderHeight: number;
    protected placeholderHeight: number;
    protected lastValidRect: Rect;
    protected validRect: Rect;
    /**
     * Creates a new ListLayout with options. See the list of properties below for a description
     * of the options that can be provided.
     */
    constructor(options?: ListLayoutOptions<T>);
    getLayoutInfo(key: Key): LayoutInfo;
    getVisibleLayoutInfos(rect: Rect): LayoutInfo[];
    isVisible(node: LayoutNode, rect: Rect): boolean;
    protected shouldInvalidateEverything(invalidationContext: InvalidationContext<Node<T>, unknown>): boolean;
    validate(invalidationContext: InvalidationContext<Node<T>, unknown>): void;
    buildCollection(): LayoutNode[];
    isValid(node: Node<T>, y: number): boolean;
    buildChild(node: Node<T>, x: number, y: number): LayoutNode;
    buildNode(node: Node<T>, x: number, y: number): LayoutNode;
    buildSection(node: Node<T>, x: number, y: number): LayoutNode;
    buildItem(node: Node<T>, x: number, y: number): LayoutNode;
    updateItemSize(key: Key, size: Size): boolean;
    updateLayoutNode(key: Key, oldLayoutInfo: LayoutInfo, newLayoutInfo: LayoutInfo): void;
    getContentSize(): Size;
    getKeyAbove(key: Key): Key | null;
    getKeyBelow(key: Key): Key | null;
    getKeyPageAbove(key: Key): Key | null;
    getKeyPageBelow(key: Key): Key | null;
    getFirstKey(): Key | null;
    getLastKey(): Key | null;
    getKeyForSearch(search: string, fromKey?: Key): Key | null;
    getInitialLayoutInfo(layoutInfo: LayoutInfo): LayoutInfo;
    getFinalLayoutInfo(layoutInfo: LayoutInfo): LayoutInfo;
    getDropTargetFromPoint(x: number, y: number, isValidDropTarget: (target: DropTarget) => boolean): DropTarget;
}
type TableLayoutOptions<T> = ListLayoutOptions<T> & {
    columnLayout: TableColumnLayout<T>;
    initialCollection: TableCollection<T>;
};
export class TableLayout<T> extends ListLayout<T> {
    collection: TableCollection<T>;
    lastCollection: TableCollection<T>;
    columnWidths: Map<Key, number>;
    stickyColumnIndices: number[];
    wasLoading: boolean;
    isLoading: boolean;
    lastPersistedKeys: Set<Key>;
    persistedIndices: Map<Key, number[]>;
    columnLayout: TableColumnLayout<T>;
    controlledColumns: Map<Key, GridNode<unknown>>;
    uncontrolledColumns: Map<Key, GridNode<unknown>>;
    uncontrolledWidths: Map<Key, ColumnSize>;
    resizingColumn: Key | null;
    constructor(options: TableLayoutOptions<T>);
    protected shouldInvalidateEverything(invalidationContext: InvalidationContext<Node<T>, unknown>): boolean;
    getResizerPosition(): Key;
    getColumnWidth(key: Key): number;
    getColumnMinWidth(key: Key): number;
    getColumnMaxWidth(key: Key): number;
    startResize(key: Key): void;
    updateResizedColumns(key: Key, width: number): Map<Key, ColumnSize>;
    endResize(): void;
    buildCollection(): LayoutNode[];
    buildHeader(): LayoutNode;
    buildHeaderRow(headerRow: GridNode<T>, x: number, y: number): LayoutNode;
    setChildHeights(children: LayoutNode[], height: number): void;
    getRenderedColumnWidth(node: GridNode<T>): number;
    getEstimatedHeight(node: GridNode<T>, width: number, height: number, estimatedHeight: number): {
        height: number;
        isEstimated: boolean;
    };
    buildColumn(node: GridNode<T>, x: number, y: number): LayoutNode;
    buildBody(y: number): LayoutNode;
    buildNode(node: GridNode<T>, x: number, y: number): LayoutNode;
    buildRow(node: GridNode<T>, x: number, y: number): LayoutNode;
    buildCell(node: GridNode<T>, x: number, y: number): LayoutNode;
    getVisibleLayoutInfos(rect: Rect): LayoutInfo[];
    addVisibleLayoutInfos(res: LayoutInfo[], node: LayoutNode, rect: Rect): void;
    binarySearch(items: LayoutNode[], point: Point, axis: 'x' | 'y'): number;
    buildPersistedIndices(): void;
    getInitialLayoutInfo(layoutInfo: LayoutInfo): LayoutInfo;
}

//# sourceMappingURL=types.d.ts.map
