"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const plexapi_1 = __importDefault(require("../api/plexapi"));
const radarr_1 = __importDefault(require("../api/servarr/radarr"));
const sonarr_1 = __importDefault(require("../api/servarr/sonarr"));
const media_1 = require("../constants/media");
const datasource_1 = require("../datasource");
const Media_1 = __importDefault(require("../entity/Media"));
const MediaRequest_1 = __importDefault(require("../entity/MediaRequest"));
const User_1 = require("../entity/User");
const settings_1 = require("../lib/settings");
const logger_1 = __importDefault(require("../logger"));
class AvailabilitySync {
    constructor() {
        this.running = false;
    }
    async run() {
        const settings = (0, settings_1.getSettings)();
        this.running = true;
        this.plexSeasonsCache = {};
        this.sonarrSeasonsCache = {};
        this.radarrServers = settings.radarr.filter((server) => server.syncEnabled);
        this.sonarrServers = settings.sonarr.filter((server) => server.syncEnabled);
        try {
            logger_1.default.info(`Starting availability sync...`, {
                label: 'Availability Sync',
            });
            const pageSize = 50;
            const userRepository = (0, datasource_1.getRepository)(User_1.User);
            const admin = await userRepository.findOne({
                select: { id: true, plexToken: true },
                where: { id: 1 },
            });
            if (admin) {
                this.plexClient = new plexapi_1.default({ plexToken: admin.plexToken });
            }
            else {
                logger_1.default.error('An admin is not configured.');
            }
            for await (const media of this.loadAvailableMediaPaginated(pageSize)) {
                if (!this.running) {
                    throw new Error('Job aborted');
                }
                // Check plex, radarr, and sonarr for that specific media and
                // if unavailable, then we change the status accordingly.
                // If a non-4k or 4k version exists in at least one of the instances, we will only update that specific version
                if (media.mediaType === 'movie') {
                    let movieExists = false;
                    let movieExists4k = false;
                    const { existsInPlex } = await this.mediaExistsInPlex(media, false);
                    const { existsInPlex: existsInPlex4k } = await this.mediaExistsInPlex(media, true);
                    const existsInRadarr = await this.mediaExistsInRadarr(media, false);
                    const existsInRadarr4k = await this.mediaExistsInRadarr(media, true);
                    if (existsInPlex || existsInRadarr) {
                        movieExists = true;
                        logger_1.default.info(`The non-4K movie [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                            label: 'Availability Sync',
                        });
                    }
                    if (existsInPlex4k || existsInRadarr4k) {
                        movieExists4k = true;
                        logger_1.default.info(`The 4K movie [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                            label: 'Availability Sync',
                        });
                    }
                    if (!movieExists && media.status === media_1.MediaStatus.AVAILABLE) {
                        await this.mediaUpdater(media, false);
                    }
                    if (!movieExists4k && media.status4k === media_1.MediaStatus.AVAILABLE) {
                        await this.mediaUpdater(media, true);
                    }
                }
                // If both versions still exist in plex, we still need
                // to check through sonarr to verify season availability
                if (media.mediaType === 'tv') {
                    let showExists = false;
                    let showExists4k = false;
                    const { existsInPlex, seasonsMap: plexSeasonsMap = new Map() } = await this.mediaExistsInPlex(media, false);
                    const { existsInPlex: existsInPlex4k, seasonsMap: plexSeasonsMap4k = new Map(), } = await this.mediaExistsInPlex(media, true);
                    const { existsInSonarr, seasonsMap: sonarrSeasonsMap } = await this.mediaExistsInSonarr(media, false);
                    const { existsInSonarr: existsInSonarr4k, seasonsMap: sonarrSeasonsMap4k, } = await this.mediaExistsInSonarr(media, true);
                    if (existsInPlex || existsInSonarr) {
                        showExists = true;
                        logger_1.default.info(`The non-4K show [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                            label: 'Availability Sync',
                        });
                    }
                    if (existsInPlex4k || existsInSonarr4k) {
                        showExists4k = true;
                        logger_1.default.info(`The 4K show [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                            label: 'Availability Sync',
                        });
                    }
                    // Here we will create a final map that will cross compare
                    // with plex and sonarr. Filtered seasons will go through
                    // each season and assume the season does not exist. If Plex or
                    // Sonarr finds that season, we will change the final seasons value
                    // to true.
                    const filteredSeasonsMap = new Map();
                    media.seasons
                        .filter((season) => season.status === media_1.MediaStatus.AVAILABLE ||
                        season.status === media_1.MediaStatus.PARTIALLY_AVAILABLE)
                        .forEach((season) => filteredSeasonsMap.set(season.seasonNumber, false));
                    const finalSeasons = new Map([
                        ...filteredSeasonsMap,
                        ...plexSeasonsMap,
                        ...sonarrSeasonsMap,
                    ]);
                    const filteredSeasonsMap4k = new Map();
                    media.seasons
                        .filter((season) => season.status4k === media_1.MediaStatus.AVAILABLE ||
                        season.status4k === media_1.MediaStatus.PARTIALLY_AVAILABLE)
                        .forEach((season) => filteredSeasonsMap4k.set(season.seasonNumber, false));
                    const finalSeasons4k = new Map([
                        ...filteredSeasonsMap4k,
                        ...plexSeasonsMap4k,
                        ...sonarrSeasonsMap4k,
                    ]);
                    if ([...finalSeasons.values()].includes(false)) {
                        await this.seasonUpdater(media, finalSeasons, false);
                    }
                    if ([...finalSeasons4k.values()].includes(false)) {
                        await this.seasonUpdater(media, finalSeasons4k, true);
                    }
                    if (!showExists &&
                        (media.status === media_1.MediaStatus.AVAILABLE ||
                            media.status === media_1.MediaStatus.PARTIALLY_AVAILABLE)) {
                        await this.mediaUpdater(media, false);
                    }
                    if (!showExists4k &&
                        (media.status4k === media_1.MediaStatus.AVAILABLE ||
                            media.status4k === media_1.MediaStatus.PARTIALLY_AVAILABLE)) {
                        await this.mediaUpdater(media, true);
                    }
                }
            }
        }
        catch (ex) {
            logger_1.default.error('Failed to complete availability sync.', {
                errorMessage: ex.message,
                label: 'Availability Sync',
            });
        }
        finally {
            logger_1.default.info(`Availability sync complete.`, {
                label: 'Availability Sync',
            });
            this.running = false;
        }
    }
    cancel() {
        this.running = false;
    }
    async *loadAvailableMediaPaginated(pageSize) {
        let offset = 0;
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        const whereOptions = [
            { status: media_1.MediaStatus.AVAILABLE },
            { status: media_1.MediaStatus.PARTIALLY_AVAILABLE },
            { status4k: media_1.MediaStatus.AVAILABLE },
            { status4k: media_1.MediaStatus.PARTIALLY_AVAILABLE },
        ];
        let mediaPage;
        do {
            yield* (mediaPage = await mediaRepository.find({
                where: whereOptions,
                skip: offset,
                take: pageSize,
            }));
            offset += pageSize;
        } while (mediaPage.length > 0);
    }
    async mediaUpdater(media, is4k) {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        try {
            // If media type is tv, check if a season is processing
            // to see if we need to keep the external metadata
            let isMediaProcessing = false;
            if (media.mediaType === 'tv') {
                const requestRepository = (0, datasource_1.getRepository)(MediaRequest_1.default);
                const request = await requestRepository
                    .createQueryBuilder('request')
                    .leftJoinAndSelect('request.media', 'media')
                    .where('(media.id = :id)', {
                    id: media.id,
                })
                    .andWhere('(request.is4k = :is4k AND request.status = :requestStatus)', {
                    requestStatus: media_1.MediaRequestStatus.APPROVED,
                    is4k: is4k,
                })
                    .getOne();
                if (request) {
                    isMediaProcessing = true;
                }
            }
            // Set the non-4K or 4K media to deleted
            // and change related columns to null if media
            // is not processing
            media[is4k ? 'status4k' : 'status'] = media_1.MediaStatus.DELETED;
            media[is4k ? 'serviceId4k' : 'serviceId'] = isMediaProcessing
                ? media[is4k ? 'serviceId4k' : 'serviceId']
                : null;
            media[is4k ? 'externalServiceId4k' : 'externalServiceId'] =
                isMediaProcessing
                    ? media[is4k ? 'externalServiceId4k' : 'externalServiceId']
                    : null;
            media[is4k ? 'externalServiceSlug4k' : 'externalServiceSlug'] =
                isMediaProcessing
                    ? media[is4k ? 'externalServiceSlug4k' : 'externalServiceSlug']
                    : null;
            media[is4k ? 'ratingKey4k' : 'ratingKey'] = isMediaProcessing
                ? media[is4k ? 'ratingKey4k' : 'ratingKey']
                : null;
            logger_1.default.info(`The ${is4k ? '4K' : 'non-4K'} ${media.mediaType === 'movie' ? 'movie' : 'show'} [TMDB ID ${media.tmdbId}] was not found in any ${media.mediaType === 'movie' ? 'Radarr' : 'Sonarr'} and Plex instance. Status will be changed to deleted.`, { label: 'Availability Sync' });
            await mediaRepository.save(media);
        }
        catch (ex) {
            logger_1.default.debug(`Failure updating the ${is4k ? '4K' : 'non-4K'} ${media.mediaType === 'tv' ? 'show' : 'movie'} [TMDB ID ${media.tmdbId}].`, {
                errorMessage: ex.message,
                label: 'Availability Sync',
            });
        }
    }
    async seasonUpdater(media, seasons, is4k) {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        // Filter out only the values that are false
        // (media that should be deleted)
        const seasonsPendingRemoval = new Map(
        // Disabled linter as only the value is needed from the filter
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        [...seasons].filter(([_, exists]) => !exists));
        // Retrieve the season keys to pass into our log
        const seasonKeys = [...seasonsPendingRemoval.keys()];
        try {
            for (const mediaSeason of media.seasons) {
                if (seasonsPendingRemoval.has(mediaSeason.seasonNumber)) {
                    mediaSeason[is4k ? 'status4k' : 'status'] = media_1.MediaStatus.DELETED;
                }
            }
            if (media.status === media_1.MediaStatus.AVAILABLE && !is4k) {
                media.status = media_1.MediaStatus.PARTIALLY_AVAILABLE;
                logger_1.default.info(`Marking the non-4K show [TMDB ID ${media.tmdbId}] as PARTIALLY_AVAILABLE because season removal has occurred.`, { label: 'Availability Sync' });
            }
            if (media.status4k === media_1.MediaStatus.AVAILABLE && is4k) {
                media.status4k = media_1.MediaStatus.PARTIALLY_AVAILABLE;
                logger_1.default.info(`Marking the 4K show [TMDB ID ${media.tmdbId}] as PARTIALLY_AVAILABLE because season removal has occurred.`, { label: 'Availability Sync' });
            }
            media.lastSeasonChange = new Date();
            await mediaRepository.save(media);
            logger_1.default.info(`The ${is4k ? '4K' : 'non-4K'} season(s) [${seasonKeys}] [TMDB ID ${media.tmdbId}] was not found in any ${media.mediaType === 'tv' ? 'Sonarr' : 'Radarr'} and Plex instance. Status will be changed to deleted.`, { label: 'Availability Sync' });
        }
        catch (ex) {
            logger_1.default.debug(`Failure updating the ${is4k ? '4K' : 'non-4K'} season(s) [${seasonKeys}], TMDB ID ${media.tmdbId}.`, {
                errorMessage: ex.message,
                label: 'Availability Sync',
            });
        }
    }
    async mediaExistsInRadarr(media, is4k) {
        let existsInRadarr = false;
        // Check for availability in all of the available radarr servers
        // If any find the media, we will assume the media exists
        for (const server of this.radarrServers.filter((server) => server.is4k === is4k)) {
            const radarrAPI = new radarr_1.default({
                apiKey: server.apiKey,
                url: radarr_1.default.buildUrl(server, '/api/v3'),
            });
            try {
                let radarr;
                if (media.externalServiceId && !is4k) {
                    radarr = await radarrAPI.getMovie({
                        id: media.externalServiceId,
                    });
                }
                if (media.externalServiceId4k && is4k) {
                    radarr = await radarrAPI.getMovie({
                        id: media.externalServiceId4k,
                    });
                }
                if (radarr && radarr.hasFile) {
                    existsInRadarr = true;
                }
            }
            catch (ex) {
                if (!ex.message.includes('404')) {
                    existsInRadarr = true;
                    logger_1.default.debug(`Failure retrieving the ${is4k ? '4K' : 'non-4K'} movie [TMDB ID ${media.tmdbId}] from Radarr.`, {
                        errorMessage: ex.message,
                        label: 'Availability Sync',
                    });
                }
            }
        }
        return existsInRadarr;
    }
    async mediaExistsInSonarr(media, is4k) {
        let existsInSonarr = false;
        let preventSeasonSearch = false;
        // Check for availability in all of the available sonarr servers
        // If any find the media, we will assume the media exists
        for (const server of this.sonarrServers.filter((server) => {
            return server.is4k === is4k;
        })) {
            const sonarrAPI = new sonarr_1.default({
                apiKey: server.apiKey,
                url: sonarr_1.default.buildUrl(server, '/api/v3'),
            });
            try {
                let sonarr;
                if (media.externalServiceId && !is4k) {
                    sonarr = await sonarrAPI.getSeriesById(media.externalServiceId);
                    this.sonarrSeasonsCache[`${server.id}-${media.externalServiceId}`] =
                        sonarr.seasons;
                }
                if (media.externalServiceId4k && is4k) {
                    sonarr = await sonarrAPI.getSeriesById(media.externalServiceId4k);
                    this.sonarrSeasonsCache[`${server.id}-${media.externalServiceId4k}`] =
                        sonarr.seasons;
                }
                if (sonarr && sonarr.statistics.episodeFileCount > 0) {
                    existsInSonarr = true;
                }
            }
            catch (ex) {
                if (!ex.message.includes('404')) {
                    existsInSonarr = true;
                    preventSeasonSearch = true;
                    logger_1.default.debug(`Failure retrieving the ${is4k ? '4K' : 'non-4K'} show [TMDB ID ${media.tmdbId}] from Sonarr.`, {
                        errorMessage: ex.message,
                        label: 'Availability Sync',
                    });
                }
            }
        }
        // Here we check each season for availability
        // If the API returns an error other than a 404,
        // we will have to prevent the season check from happening
        const seasonsMap = new Map();
        if (!preventSeasonSearch) {
            const filteredSeasons = media.seasons.filter((season) => season[is4k ? 'status4k' : 'status'] === media_1.MediaStatus.AVAILABLE ||
                season[is4k ? 'status4k' : 'status'] ===
                    media_1.MediaStatus.PARTIALLY_AVAILABLE);
            for (const season of filteredSeasons) {
                const seasonExists = await this.seasonExistsInSonarr(media, season, is4k);
                if (seasonExists) {
                    seasonsMap.set(season.seasonNumber, true);
                }
            }
        }
        return { existsInSonarr, seasonsMap };
    }
    async seasonExistsInSonarr(media, season, is4k) {
        let seasonExists = false;
        // Check each sonarr instance to see if the media still exists
        // If found, we will assume the media exists and prevent removal
        // We can use the cache we built when we fetched the series with mediaExistsInSonarr
        for (const server of this.sonarrServers.filter((server) => server.is4k === is4k)) {
            let sonarrSeasons;
            if (media.externalServiceId && !is4k) {
                sonarrSeasons =
                    this.sonarrSeasonsCache[`${server.id}-${media.externalServiceId}`];
            }
            if (media.externalServiceId4k && is4k) {
                sonarrSeasons =
                    this.sonarrSeasonsCache[`${server.id}-${media.externalServiceId4k}`];
            }
            const seasonIsAvailable = sonarrSeasons?.find(({ seasonNumber, statistics }) => season.seasonNumber === seasonNumber &&
                statistics?.episodeFileCount &&
                statistics?.episodeFileCount > 0);
            if (seasonIsAvailable && sonarrSeasons) {
                seasonExists = true;
            }
        }
        return seasonExists;
    }
    async mediaExistsInPlex(media, is4k) {
        const ratingKey = media.ratingKey;
        const ratingKey4k = media.ratingKey4k;
        let existsInPlex = false;
        let preventSeasonSearch = false;
        // Check each plex instance to see if the media still exists
        // If found, we will assume the media exists and prevent removal
        // We can use the cache we built when we fetched the series with mediaExistsInPlex
        try {
            let plexMedia;
            if (ratingKey && !is4k) {
                plexMedia = await this.plexClient?.getMetadata(ratingKey);
                if (media.mediaType === 'tv') {
                    this.plexSeasonsCache[ratingKey] =
                        await this.plexClient?.getChildrenMetadata(ratingKey);
                }
            }
            if (ratingKey4k && is4k) {
                plexMedia = await this.plexClient?.getMetadata(ratingKey4k);
                if (media.mediaType === 'tv') {
                    this.plexSeasonsCache[ratingKey4k] =
                        await this.plexClient?.getChildrenMetadata(ratingKey4k);
                }
            }
            if (plexMedia) {
                existsInPlex = true;
            }
        }
        catch (ex) {
            if (!ex.message.includes('404')) {
                existsInPlex = true;
                preventSeasonSearch = true;
                logger_1.default.debug(`Failure retrieving the ${is4k ? '4K' : 'non-4K'} ${media.mediaType === 'tv' ? 'show' : 'movie'} [TMDB ID ${media.tmdbId}] from Plex.`, {
                    errorMessage: ex.message,
                    label: 'Availability Sync',
                });
            }
        }
        // Here we check each season in plex for availability
        // If the API returns an error other than a 404,
        // we will have to prevent the season check from happening
        if (media.mediaType === 'tv') {
            const seasonsMap = new Map();
            if (!preventSeasonSearch) {
                const filteredSeasons = media.seasons.filter((season) => season[is4k ? 'status4k' : 'status'] === media_1.MediaStatus.AVAILABLE ||
                    season[is4k ? 'status4k' : 'status'] ===
                        media_1.MediaStatus.PARTIALLY_AVAILABLE);
                for (const season of filteredSeasons) {
                    const seasonExists = await this.seasonExistsInPlex(media, season, is4k);
                    if (seasonExists) {
                        seasonsMap.set(season.seasonNumber, true);
                    }
                }
            }
            return { existsInPlex, seasonsMap };
        }
        return { existsInPlex };
    }
    async seasonExistsInPlex(media, season, is4k) {
        const ratingKey = media.ratingKey;
        const ratingKey4k = media.ratingKey4k;
        let seasonExistsInPlex = false;
        // Check each plex instance to see if the season exists
        let plexSeasons;
        if (ratingKey && !is4k) {
            plexSeasons = this.plexSeasonsCache[ratingKey];
        }
        if (ratingKey4k && is4k) {
            plexSeasons = this.plexSeasonsCache[ratingKey4k];
        }
        const seasonIsAvailable = plexSeasons?.find((plexSeason) => plexSeason.index === season.seasonNumber);
        if (seasonIsAvailable) {
            seasonExistsInPlex = true;
        }
        return seasonExistsInPlex;
    }
}
const availabilitySync = new AvailabilitySync();
exports.default = availabilitySync;
