"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const issue_1 = require("../../../constants/issue");
const media_1 = require("../../../constants/media");
const settings_1 = require("../../../lib/settings");
const logger_1 = __importDefault(require("../../../logger"));
const axios_1 = __importDefault(require("axios"));
const __1 = require("..");
const agent_1 = require("./agent");
class LunaSeaAgent extends agent_1.BaseAgent {
    getSettings() {
        if (this.settings) {
            return this.settings;
        }
        const settings = (0, settings_1.getSettings)();
        return settings.notifications.agents.lunasea;
    }
    buildPayload(type, payload) {
        return {
            notification_type: __1.Notification[type],
            event: payload.event,
            subject: payload.subject,
            message: payload.message,
            image: payload.image ?? null,
            email: payload.notifyUser?.email,
            username: payload.notifyUser?.displayName,
            avatar: payload.notifyUser?.avatar,
            media: payload.media
                ? {
                    media_type: payload.media.mediaType,
                    tmdbId: payload.media.tmdbId,
                    tvdbId: payload.media.tvdbId,
                    status: media_1.MediaStatus[payload.media.status],
                    status4k: media_1.MediaStatus[payload.media.status4k],
                }
                : null,
            extra: payload.extra ?? [],
            request: payload.request
                ? {
                    request_id: payload.request.id,
                    requestedBy_email: payload.request.requestedBy.email,
                    requestedBy_username: payload.request.requestedBy.displayName,
                    requestedBy_avatar: payload.request.requestedBy.avatar,
                }
                : null,
            issue: payload.issue
                ? {
                    issue_id: payload.issue.id,
                    issue_type: issue_1.IssueType[payload.issue.issueType],
                    issue_status: issue_1.IssueStatus[payload.issue.status],
                    createdBy_email: payload.issue.createdBy.email,
                    createdBy_username: payload.issue.createdBy.displayName,
                    createdBy_avatar: payload.issue.createdBy.avatar,
                }
                : null,
            comment: payload.comment
                ? {
                    comment_message: payload.comment.message,
                    commentedBy_email: payload.comment.user.email,
                    commentedBy_username: payload.comment.user.displayName,
                    commentedBy_avatar: payload.comment.user.avatar,
                }
                : null,
        };
    }
    shouldSend() {
        const settings = this.getSettings();
        if (settings.enabled && settings.options.webhookUrl) {
            return true;
        }
        return false;
    }
    async send(type, payload) {
        const settings = this.getSettings();
        if (!payload.notifySystem ||
            !(0, __1.hasNotificationType)(type, settings.types ?? 0)) {
            return true;
        }
        logger_1.default.debug('Sending LunaSea notification', {
            label: 'Notifications',
            type: __1.Notification[type],
            subject: payload.subject,
        });
        try {
            await axios_1.default.post(settings.options.webhookUrl, this.buildPayload(type, payload), settings.options.profileName
                ? {
                    headers: {
                        Authorization: `Basic ${Buffer.from(`${settings.options.profileName}:`).toString('base64')}`,
                    },
                }
                : undefined);
            return true;
        }
        catch (e) {
            logger_1.default.error('Error sending LunaSea notification', {
                label: 'Notifications',
                type: __1.Notification[type],
                subject: payload.subject,
                errorMessage: e.message,
                response: e.response?.data,
            });
            return false;
        }
    }
}
exports.default = LunaSeaAgent;
