"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sonarrScanner = void 0;
const sonarr_1 = __importDefault(require("../../../api/servarr/sonarr"));
const datasource_1 = require("../../../datasource");
const Media_1 = __importDefault(require("../../../entity/Media"));
const baseScanner_1 = __importDefault(require("../../../lib/scanners/baseScanner"));
const settings_1 = require("../../../lib/settings");
const lodash_1 = require("lodash");
class SonarrScanner extends baseScanner_1.default {
    constructor() {
        super('Sonarr Scan', { bundleSize: 50 });
    }
    status() {
        return {
            running: this.running,
            progress: this.progress,
            total: this.items.length,
            currentServer: this.currentServer,
            servers: this.servers,
        };
    }
    async run() {
        const settings = (0, settings_1.getSettings)();
        const sessionId = this.startRun();
        try {
            this.servers = (0, lodash_1.uniqWith)(settings.sonarr, (sonarrA, sonarrB) => {
                return (sonarrA.hostname === sonarrB.hostname &&
                    sonarrA.port === sonarrB.port &&
                    sonarrA.baseUrl === sonarrB.baseUrl);
            });
            for (const server of this.servers) {
                this.currentServer = server;
                if (server.syncEnabled) {
                    this.log(`Beginning to process Sonarr server: ${server.name}`, 'info');
                    this.sonarrApi = new sonarr_1.default({
                        apiKey: server.apiKey,
                        url: sonarr_1.default.buildUrl(server, '/api/v3'),
                    });
                    this.items = await this.sonarrApi.getSeries();
                    await this.loop(this.processSonarrSeries.bind(this), { sessionId });
                }
                else {
                    this.log(`Sync not enabled. Skipping Sonarr server: ${server.name}`);
                }
            }
            this.log('Sonarr scan complete', 'info');
        }
        catch (e) {
            this.log('Scan interrupted', 'error', { errorMessage: e.message });
        }
        finally {
            this.endRun(sessionId);
        }
    }
    async processSonarrSeries(sonarrSeries) {
        try {
            const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
            const server4k = this.enable4kShow && this.currentServer.is4k;
            const processableSeasons = [];
            let tvShow;
            const media = await mediaRepository.findOne({
                where: { tvdbId: sonarrSeries.tvdbId },
            });
            if (!media || !media.tmdbId) {
                tvShow = await this.tmdb.getShowByTvdbId({
                    tvdbId: sonarrSeries.tvdbId,
                });
            }
            else {
                tvShow = await this.tmdb.getTvShow({ tvId: media.tmdbId });
            }
            const tmdbId = tvShow.id;
            const filteredSeasons = sonarrSeries.seasons.filter((sn) => tvShow.seasons.find((s) => s.season_number === sn.seasonNumber));
            for (const season of filteredSeasons) {
                const totalAvailableEpisodes = season.statistics?.episodeFileCount ?? 0;
                processableSeasons.push({
                    seasonNumber: season.seasonNumber,
                    episodes: !server4k ? totalAvailableEpisodes : 0,
                    episodes4k: server4k ? totalAvailableEpisodes : 0,
                    totalEpisodes: season.statistics?.totalEpisodeCount ?? 0,
                    processing: season.monitored && totalAvailableEpisodes === 0,
                    is4kOverride: server4k,
                });
            }
            await this.processShow(tmdbId, sonarrSeries.tvdbId, processableSeasons, {
                serviceId: this.currentServer.id,
                externalServiceId: sonarrSeries.id,
                externalServiceSlug: sonarrSeries.titleSlug,
                title: sonarrSeries.title,
                is4k: server4k,
            });
        }
        catch (e) {
            this.log('Failed to process Sonarr media', 'error', {
                errorMessage: e.message,
                title: sonarrSeries.title,
            });
        }
    }
}
exports.sonarrScanner = new SonarrScanner();
