import { Readable, Writable } from 'svelte/store';

type ExtendedStore<T, Get = () => T, Store = Readable<T>> = Store & {
    get: Get;
};
type LoadingStore = Readable<boolean> & {
    toPromise: (locale?: Config.Locale, route?: string) => Promise<void[] | void>;
    get: () => boolean;
};
declare namespace DotNotation {
    type Input = any;
    type Output<V = any, K extends keyof V = keyof V> = {
        [P in K]?: V[K];
    } | null | V;
    type T = <I = Input>(input: I, preserveArrays?: boolean, parentKey?: string) => Output<I>;
}
declare namespace Logger {
    type Level = 'error' | 'warn' | 'debug';
    type Prefix = string;
    type T = {
        [key in Logger.Level]: (value: any) => void;
    };
    type FactoryProps = {
        /**
         * You can setup your custom logger using this property.
         *
         * @default console
         */
        logger?: Logger.T;
        /**
         * You can manage log level using this property.
         *
         * @default 'warn'
         */
        level?: Logger.Level;
        /**
         * You can prefix output logs using this property.
         *
         * @default '[i18n]: '
         */
        prefix?: Logger.Prefix;
    };
}
declare namespace Config {
    type Loader = Loader.LoaderModule;
    type Translations = Translations.T;
    type Locale = Translations.Locales[number];
    type InitLocale = Locale | undefined;
    type FallbackLocale = Locale | undefined;
    type FallbackValue = any;
    type T<P extends Parser.Params = Parser.Params> = {
        /**
         * You can use loaders to define your asyncronous translation load. All loaded data are stored so loader is triggered only once – in case there is no previous version of the translation. It can get refreshed according to `config.cache`.
         */
        loaders?: Loader[];
        /**
         * Locale-indexed translations, which should be in place before loaders will trigger. It's useful for static pages and synchronous translations – for example locally defined language names which are the same for all of the language mutations.
         *
         * @example {
         *  "en": {"lang": {"en": "English", "cs": "Česky"}}
         *  "cs": {"lang": {"en": "English", "cs": "Česky"}}
         * }
         */
        translations?: Translations.T;
        /**
         * If you set this property, translations will be initialized immediately using this locale.
         */
        initLocale?: InitLocale;
        /**
         * If you set this property, translations are automatically loaded not for current `$locale` only, but for this locale as well. In case there is no translation for current `$locale`, fallback locale translation is used instead of translation key placeholder. This is also used as a fallback when unknown locale is set.
         */
        fallbackLocale?: FallbackLocale;
        /**
         * By default, translation key is returned in case no translation is found for given translation key. For example, `$t('unknown.key')` will result in `'unknown.key'` output. You can set this output value using this config prop.
         */
        fallbackValue?: FallbackValue;
        /**
         * Preprocessor strategy or a custom function. Defines, how to transform the translation data immediately after the load.
         * @default 'full'
         *
         * @example 'full'
         * {a: {b: [{c: {d: 1}}, {c: {d: 2}}]}} => {"a.b.0.c.d": 1, "a.b.1.c.d": 2}
         *
         * @example 'preserveArrays'
         * {a: {b: [{c: {d: 1}}, {c: {d: 2}}]}} => {"a.b": [{"c.d": 1}, {"c.d": 2}]}
         *
         * @example 'none'
         * {a: {b: [{c: {d: 1}}, {c: {d: 2}}]}} => {a: {b: [{c: {d: 1}}, {c: {d: 2}}]}}
         */
        preprocess?: 'full' | 'preserveArrays' | 'none' | ((input: Translations.Input) => any);
        /**
         * This property defines translation syntax you want to use.
         */
        parser: Parser.T<P>;
        /**
         * When you are running your app on Node.js server, translations are loaded only once during the SSR. This property allows you to setup a refresh period in milliseconds when your translations are refetched on the server.
         *
         * @default 86400000 // 24 hours
         *
         * @tip You can set to `Number.POSITIVE_INFINITY` to disable server-side refreshing.
         */
        cache?: number;
        /**
         * Custom logger configuration.
         */
        log?: Logger.FactoryProps;
    };
}
declare namespace Loader {
    type Key = string;
    type Locale = Config.Locale;
    type Route = string | RegExp;
    type IndexedKeys = Translations.LocaleIndexed<Key[]>;
    type LoaderModule = {
        /**
         * Represents the translation namespace. This key is used as a translation prefix so it should be module-unique. You can access your translation later using `$t('key.yourTranslation')`. It shouldn't include `.` (dot) character.
         */
        key: Key;
        /**
         * Locale (e.g. `en`, `de`) which is this loader for.
         */
        locale: Locale;
        /**
         * Function returning a `Promise` with translation data. You can use it to load files locally, fetch it from your API etc...
        */
        loader: T;
        /**
        * Define routes this loader should be triggered for. You can use Regular expressions too. For example `[/\/.ome/]` will be triggered for `/home` and `/rome` route as well (but still only once). Leave this `undefined` in case you want to load this module with any route (useful for common translations).
        */
        routes?: Route[];
    };
    type T = () => Promise<Translations.Input>;
}
declare namespace Parser {
    type Value = any;
    type Params = Array<unknown>;
    type Locale = Config.Locale;
    type Key = Loader.Key;
    type Output = any;
    type Parse<P extends Parser.Params = Parser.Params> = (
    /**
     * Translation value from the definitions.
    */
    value: Value, 
    /**
     * Array of rest parameters given by user (e.g. payload variables etc...)
     */
    params: P, 
    /**
     * Locale of translated message.
     */
    locale: Locale, 
    /**
     * This key is serialized path to translation (e.g., `home.content.title`)
     */
    key: Key) => Output;
    type T<P extends Parser.Params = Parser.Params> = {
        /**
         * Parse function deals with interpolation of user payload and returns interpolated message.
        */
        parse: Parse<P>;
    };
}
declare namespace Translations {
    type Locales<T = string> = T[];
    type SerializedTranslations = LocaleIndexed<DotNotation.Input>;
    type TranslationData<T = any> = Loader.LoaderModule & {
        data: T;
    };
    type FetchTranslations = (loaders: Loader.LoaderModule[]) => Promise<SerializedTranslations>;
    type TranslationFunction<P extends Parser.Params = Parser.Params> = (key: string, ...restParams: P) => any;
    type LocalTranslationFunction<P extends Parser.Params = Parser.Params> = (locale: Config.Locale, key: string, ...restParams: P) => any;
    type Translate = <P extends Parser.Params = Parser.Params>(props: {
        parser: Parser.T<P>;
        key: string;
        params: P;
        translations: SerializedTranslations;
        locale: Locales[number];
        fallbackLocale?: Config.FallbackLocale;
        fallbackValue?: Config.FallbackValue;
    }) => string;
    type Input<V = any> = {
        [K in any]: Input<V> | V;
    };
    type LocaleIndexed<V, L extends string = string> = {
        [locale in Locales<L>[number]]: V;
    };
    type T<V = any, L extends string = string> = LocaleIndexed<Input<V>, L>;
}

declare class I18n<ParserParams extends Parser.Params = any> {
    constructor(config?: Config.T<ParserParams>);
    private cachedAt;
    private loadedKeys;
    private currentRoute;
    private config;
    private isLoading;
    private promises;
    loading: LoadingStore;
    private privateRawTranslations;
    rawTranslations: ExtendedStore<Translations.SerializedTranslations>;
    private privateTranslations;
    translations: ExtendedStore<Translations.SerializedTranslations>;
    locales: ExtendedStore<Config.Locale[]>;
    private internalLocale;
    private loaderTrigger;
    private localeHelper;
    locale: ExtendedStore<Config.Locale, () => Config.Locale, Writable<string>> & {
        forceSet: any;
    };
    initialized: Readable<boolean>;
    private translation;
    t: ExtendedStore<Translations.TranslationFunction<ParserParams>, Translations.TranslationFunction<ParserParams>>;
    l: ExtendedStore<Translations.LocalTranslationFunction<ParserParams>, Translations.LocalTranslationFunction<ParserParams>>;
    private getLocale;
    setLocale: (locale?: string) => Promise<void | void[]> | undefined;
    setRoute: (route: string) => Promise<void | void[]> | undefined;
    configLoader(config: Config.T<ParserParams>): Promise<void>;
    loadConfig: (config: Config.T<ParserParams>) => Promise<void>;
    getTranslationProps: ($locale?: string, $route?: string) => Promise<[Translations.SerializedTranslations, Loader.IndexedKeys] | [
    ]>;
    addTranslations: (translations?: Translations.SerializedTranslations, keys?: Loader.IndexedKeys) => void;
    private loader;
    loadTranslations: (locale: Config.Locale, route?: string) => Promise<void | void[]> | undefined;
}

export { Config, Loader, Logger, Parser, Translations, I18n as default };
