"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.aes_decrypt = exports.aes_encrypt = exports.decrypt = exports.encrypt = exports.randomKey = exports.secretbox = exports.utf8ToBytes = void 0;
const chacha_js_1 = require("./chacha.js");
const salsa_js_1 = require("./salsa.js");
const utils_js_1 = require("./utils.js");
Object.defineProperty(exports, "utf8ToBytes", { enumerable: true, get: function () { return utils_js_1.utf8ToBytes; } });
const aes_js_1 = require("./webcrypto/aes.js");
const utils_js_2 = require("./webcrypto/utils.js");
/**
 * Alias to xsalsa20poly1305, for compatibility with libsodium / nacl
 */
function secretbox(key, nonce) {
    (0, utils_js_1.ensureBytes)(key);
    (0, utils_js_1.ensureBytes)(nonce);
    const xs = (0, salsa_js_1.xsalsa20poly1305)(key, nonce);
    return { seal: xs.encrypt, open: xs.decrypt };
}
exports.secretbox = secretbox;
function randomKey() {
    return (0, utils_js_2.randomBytes)(32);
}
exports.randomKey = randomKey;
/**
 * Encrypt plaintext under key with random nonce, using xchacha20poly1305.
 * User never touches nonce: it is prepended to ciphertext.
 */
function encrypt(key, plaintext) {
    (0, utils_js_1.ensureBytes)(key);
    const nonce = (0, utils_js_2.randomBytes)(24);
    const ciphertext = (0, chacha_js_1.xchacha20poly1305)(key, nonce).encrypt(plaintext);
    return (0, utils_js_1.concatBytes)(nonce, ciphertext);
}
exports.encrypt = encrypt;
/**
 * Decrypt plaintext under key with random nonce, using xchacha20poly1305.
 * User never touches nonce: it is prepended to ciphertext.
 */
function decrypt(key, ciphertext) {
    const nonceLength = 24;
    (0, utils_js_1.ensureBytes)(ciphertext);
    if (ciphertext.length <= nonceLength)
        throw new Error('invalid ciphertext length');
    const nonce = ciphertext.subarray(0, nonceLength);
    const ciphertextWithoutNonce = ciphertext.subarray(nonceLength);
    return (0, chacha_js_1.xchacha20poly1305)(key, nonce).decrypt(ciphertextWithoutNonce);
}
exports.decrypt = decrypt;
async function aes_encrypt(key, plaintext) {
    const nonceLength = 12;
    (0, utils_js_1.ensureBytes)(key);
    const nonce = (0, utils_js_2.randomBytes)(nonceLength);
    const ciphertext = await (0, aes_js_1.aes_256_gcm)(key, nonce).encrypt(plaintext);
    return (0, utils_js_1.concatBytes)(nonce, ciphertext);
}
exports.aes_encrypt = aes_encrypt;
async function aes_decrypt(key, ciphertext) {
    const nonceLength = 12;
    (0, utils_js_1.ensureBytes)(ciphertext);
    if (ciphertext.length <= nonceLength)
        throw new Error('invalid ciphertext length');
    const nonce = ciphertext.subarray(0, nonceLength);
    const ciphertextWithoutNonce = ciphertext.subarray(nonceLength);
    return (0, aes_js_1.aes_256_gcm)(key, nonce).decrypt(ciphertextWithoutNonce);
}
exports.aes_decrypt = aes_decrypt;
//# sourceMappingURL=simple.js.map