const dgram = require('dgram');
const Joi = require('joi');
const radius = require('radius');
const {
  dictionaries: {
    rfc2865,
    freeradius_internal,
  },
  attributesToArray,
} = require('node-radius-utils');

const {
  getEphemeralPort,
} = require('./Utils');

const optionsSchema = Joi.object().keys({
  host: Joi.string().trim().required(),
  hostPort: Joi.number().min(0).optional(),
  localPort: Joi.number().min(0).optional(),
  timeout: Joi.number().min(0).optional(),
  retries: Joi.number().min(0).optional(),
  dictionaries: Joi.array().items(Joi.string().trim()).optional(),
});

class Client {
  constructor(options) {
    if (!options) throw new Error('Options are required.');
    const result = Joi.validate(options, optionsSchema);
    if (result.error) {
      throw new Error(result.error.details.map(d => d.message).join('. '));
    }
    this.options = options;
    this.options.retries = options.retries || 3;
    this.options.timeout = options.timeout || 2500;
    this.options.hostPort = options.hostPort || 1812;

    if (this.options.dictionaries) {
      this.options.dictionaries.forEach((dictionary) => {
        radius.add_dictionary(dictionary);
      });
    }
  }

  async accessRequest(info) {
    // eslint-disable-next-line
    info.code = freeradius_internal.values.PACKET_TYPE.ACCESS_REQUEST;
    const response = await this.send(info);
    if (response.code === freeradius_internal.values.PACKET_TYPE.ACCESS_REJECT) {
      const error = new Error(response.attributes[rfc2865.attributes.REPLY_MESSAGE]);
      error.response = response;
      throw error;
    }
    return response;
  }

  async send(info) {
    if (!Array.isArray(info.attributes)) {
      // eslint-disable-next-line
      info.attributes = attributesToArray(info.attributes);
    }
    const encodedPacket = radius.encode(info);

    let numTries = 0;

    while (numTries <= this.options.retries) {
      try {
        /* eslint-disable no-await-in-loop */
        const response = await this.trySend(info, encodedPacket);
        return response;
      } catch (error) {
        if (error.retryable) {
          numTries += 1;
          if (numTries > this.options.retries) {
            error.message += ` (${numTries - 1} retries)`;
            throw error;
          }
        } else {
          throw error;
        }
      }
    }

    throw new Error('Error sending request.');
  }

  trySend(info, encodedPacket) {
    return new Promise((resolve, reject) => {
      const {
        host,
        hostPort,
      } = this.options;
      const localPort = this.options.localPort || getEphemeralPort();
      const socket = dgram.createSocket('udp4');
      let timedOut = false;
      let recievedResponse = false;

      const timeoutId = setTimeout(() => {
        if (!timedOut && !recievedResponse) {
          timedOut = true;
          socket.close();
          const error = new Error(`Timed out after ${this.options.timeout}ms`);
          error.retryable = true;
          reject(error);
        }
      }, this.options.timeout);

      socket.on('error', err => reject(err));
      socket.on('listening', () => {
        socket.send(encodedPacket, 0, encodedPacket.length, hostPort, host, (err) => {
          if (err) reject(err);
        });
      });
      socket.on('message', (packet) => {
        clearTimeout(timeoutId);
        recievedResponse = true;
        socket.close();
        try {
          const response = radius.decode({
            packet,
            secret: this.options.secret,
          });
          const isValid = radius.verify_response({
            response: packet,
            request: encodedPacket,
            secret: info.secret,
          });
          if (!isValid) {
            const error = new Error('Invalid RADIUS response');
            error.response = response;
            reject(error);
          } else {
            const {
              code,
              identifier,
              attributes,
            } = response;
            if (attributes.Class) {
              try {
                attributes.Class = attributes.Class.map(b => b.toString('utf-8'));
              } catch (error) {
                attributes['x-apollo-error'] = 'Unable to parse Class attributes';
              }
            }
            resolve({
              code,
              identifier,
              attributes,
            }, response);
          }
        } catch (e) {
          reject(e);
        }
      });
      socket.bind(localPort);
    });
  }
}

module.exports = Client;
