// event.ts
import { schnorr as schnorr2 } from "@noble/curves/secp256k1";
import { sha256 } from "@noble/hashes/sha256";
import { bytesToHex as bytesToHex2 } from "@noble/hashes/utils";

// keys.ts
import { schnorr } from "@noble/curves/secp256k1";
import { bytesToHex } from "@noble/hashes/utils";
function getPublicKey(privateKey) {
  return bytesToHex(schnorr.getPublicKey(privateKey));
}

// utils.ts
var utf8Decoder = new TextDecoder("utf-8");
var utf8Encoder = new TextEncoder();

// event.ts
var verifiedSymbol = Symbol("verified");
var Kind = /* @__PURE__ */ ((Kind2) => {
  Kind2[Kind2["Metadata"] = 0] = "Metadata";
  Kind2[Kind2["Text"] = 1] = "Text";
  Kind2[Kind2["RecommendRelay"] = 2] = "RecommendRelay";
  Kind2[Kind2["Contacts"] = 3] = "Contacts";
  Kind2[Kind2["EncryptedDirectMessage"] = 4] = "EncryptedDirectMessage";
  Kind2[Kind2["EventDeletion"] = 5] = "EventDeletion";
  Kind2[Kind2["Repost"] = 6] = "Repost";
  Kind2[Kind2["Reaction"] = 7] = "Reaction";
  Kind2[Kind2["BadgeAward"] = 8] = "BadgeAward";
  Kind2[Kind2["ChannelCreation"] = 40] = "ChannelCreation";
  Kind2[Kind2["ChannelMetadata"] = 41] = "ChannelMetadata";
  Kind2[Kind2["ChannelMessage"] = 42] = "ChannelMessage";
  Kind2[Kind2["ChannelHideMessage"] = 43] = "ChannelHideMessage";
  Kind2[Kind2["ChannelMuteUser"] = 44] = "ChannelMuteUser";
  Kind2[Kind2["Blank"] = 255] = "Blank";
  Kind2[Kind2["Report"] = 1984] = "Report";
  Kind2[Kind2["ZapRequest"] = 9734] = "ZapRequest";
  Kind2[Kind2["Zap"] = 9735] = "Zap";
  Kind2[Kind2["RelayList"] = 10002] = "RelayList";
  Kind2[Kind2["ClientAuth"] = 22242] = "ClientAuth";
  Kind2[Kind2["NwcRequest"] = 23194] = "NwcRequest";
  Kind2[Kind2["HttpAuth"] = 27235] = "HttpAuth";
  Kind2[Kind2["ProfileBadge"] = 30008] = "ProfileBadge";
  Kind2[Kind2["BadgeDefinition"] = 30009] = "BadgeDefinition";
  Kind2[Kind2["Article"] = 30023] = "Article";
  Kind2[Kind2["FileMetadata"] = 1063] = "FileMetadata";
  return Kind2;
})(Kind || {});
function getBlankEvent(kind = 255 /* Blank */) {
  return {
    kind,
    content: "",
    tags: [],
    created_at: 0
  };
}
function finishEvent(t, privateKey) {
  const event = t;
  event.pubkey = getPublicKey(privateKey);
  event.id = getEventHash(event);
  event.sig = getSignature(event, privateKey);
  event[verifiedSymbol] = true;
  return event;
}
function serializeEvent(evt) {
  if (!validateEvent(evt))
    throw new Error("can't serialize event with wrong or missing properties");
  return JSON.stringify([0, evt.pubkey, evt.created_at, evt.kind, evt.tags, evt.content]);
}
function getEventHash(event) {
  let eventHash = sha256(utf8Encoder.encode(serializeEvent(event)));
  return bytesToHex2(eventHash);
}
var isRecord = (obj) => obj instanceof Object;
function validateEvent(event) {
  if (!isRecord(event))
    return false;
  if (typeof event.kind !== "number")
    return false;
  if (typeof event.content !== "string")
    return false;
  if (typeof event.created_at !== "number")
    return false;
  if (typeof event.pubkey !== "string")
    return false;
  if (!event.pubkey.match(/^[a-f0-9]{64}$/))
    return false;
  if (!Array.isArray(event.tags))
    return false;
  for (let i = 0; i < event.tags.length; i++) {
    let tag = event.tags[i];
    if (!Array.isArray(tag))
      return false;
    for (let j = 0; j < tag.length; j++) {
      if (typeof tag[j] === "object")
        return false;
    }
  }
  return true;
}
function verifySignature(event) {
  if (typeof event[verifiedSymbol] === "boolean")
    return event[verifiedSymbol];
  const hash = getEventHash(event);
  if (hash !== event.id) {
    return event[verifiedSymbol] = false;
  }
  try {
    return event[verifiedSymbol] = schnorr2.verify(event.sig, hash, event.pubkey);
  } catch (err) {
    return event[verifiedSymbol] = false;
  }
}
function signEvent(event, key) {
  console.warn(
    "nostr-tools: `signEvent` is deprecated and will be removed or changed in the future. Please use `getSignature` instead."
  );
  return getSignature(event, key);
}
function getSignature(event, key) {
  return bytesToHex2(schnorr2.sign(getEventHash(event), key));
}
export {
  Kind,
  finishEvent,
  getBlankEvent,
  getEventHash,
  getSignature,
  serializeEvent,
  signEvent,
  validateEvent,
  verifiedSymbol,
  verifySignature
};
