"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _crypto = _interopRequireDefault(require("crypto"));

var _os = _interopRequireDefault(require("os"));

var _constants = _interopRequireDefault(require("constants"));

var _tls = require("tls");

var _stream = require("stream");

var _identity = require("@azure/identity");

var _bulkLoad = _interopRequireDefault(require("./bulk-load"));

var _debug = _interopRequireDefault(require("./debug"));

var _events = require("events");

var _instanceLookup = require("./instance-lookup");

var _transientErrorLookup = require("./transient-error-lookup");

var _packet = require("./packet");

var _preloginPayload = _interopRequireDefault(require("./prelogin-payload"));

var _login7Payload = _interopRequireDefault(require("./login7-payload"));

var _ntlmPayload = _interopRequireDefault(require("./ntlm-payload"));

var _request = _interopRequireDefault(require("./request"));

var _rpcrequestPayload = _interopRequireDefault(require("./rpcrequest-payload"));

var _sqlbatchPayload = _interopRequireDefault(require("./sqlbatch-payload"));

var _messageIo = _interopRequireDefault(require("./message-io"));

var _tokenStreamParser = require("./token/token-stream-parser");

var _transaction = require("./transaction");

var _errors = require("./errors");

var _connector = require("./connector");

var _library = require("./library");

var _tdsVersions = require("./tds-versions");

var _message = _interopRequireDefault(require("./message"));

var _ntlm = require("./ntlm");

var _nodeAbortController = require("node-abort-controller");

var _dataType = require("./data-type");

var _bulkLoadPayload = require("./bulk-load-payload");

var _esAggregateError = _interopRequireDefault(require("es-aggregate-error"));

var _package = require("../package.json");

var _url = require("url");

var _handler = require("./token/handler");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let trustServerWarningEmitted = false;

const emitTrustServerCertificateWarning = () => {
  if (!trustServerWarningEmitted) {
    trustServerWarningEmitted = true;
    process.emitWarning('`config.options.trustServerCertificate` will default to false in the future. To silence this message, specify a value explicitly in the config options');
  }
};

let domainRenameToTenantIdWarningEmitted = false;

const emitDomainRenameToTenantIdWarning = () => {
  if (!domainRenameToTenantIdWarningEmitted) {
    domainRenameToTenantIdWarningEmitted = true;
    process.emitWarning('`When using authentication type `azure-active-directory-password`,' + ' config.authentication.options.domain` will be renamed to config.authentications.options.tenantId`' + ' in the future. Rename `domain` to `tenantId` to silence this message.');
  }
};

/**
 * @private
 */
const KEEP_ALIVE_INITIAL_DELAY = 30 * 1000;
/**
 * @private
 */

const DEFAULT_CONNECT_TIMEOUT = 15 * 1000;
/**
 * @private
 */

const DEFAULT_CLIENT_REQUEST_TIMEOUT = 15 * 1000;
/**
 * @private
 */

const DEFAULT_CANCEL_TIMEOUT = 5 * 1000;
/**
 * @private
 */

const DEFAULT_CONNECT_RETRY_INTERVAL = 500;
/**
 * @private
 */

const DEFAULT_PACKET_SIZE = 4 * 1024;
/**
 * @private
 */

const DEFAULT_TEXTSIZE = 2147483647;
/**
 * @private
 */

const DEFAULT_DATEFIRST = 7;
/**
 * @private
 */

const DEFAULT_PORT = 1433;
/**
 * @private
 */

const DEFAULT_TDS_VERSION = '7_4';
/**
 * @private
 */

const DEFAULT_LANGUAGE = 'us_english';
/**
 * @private
 */

const DEFAULT_DATEFORMAT = 'mdy';

/**
 * @private
 */
const CLEANUP_TYPE = {
  NORMAL: 0,
  REDIRECT: 1,
  RETRY: 2
};

/**
 * A [[Connection]] instance represents a single connection to a database server.
 *
 * ```js
 * var Connection = require('tedious').Connection;
 * var config = {
 *  "authentication": {
 *    ...,
 *    "options": {...}
 *  },
 *  "options": {...}
 * };
 * var connection = new Connection(config);
 * ```
 *
 * Only one request at a time may be executed on a connection. Once a [[Request]]
 * has been initiated (with [[Connection.callProcedure]], [[Connection.execSql]],
 * or [[Connection.execSqlBatch]]), another should not be initiated until the
 * [[Request]]'s completion callback is called.
 */
class Connection extends _events.EventEmitter {
  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * Note: be aware of the different options field:
   * 1. config.authentication.options
   * 2. config.options
   *
   * ```js
   * const { Connection } = require('tedious');
   *
   * const config = {
   *  "authentication": {
   *    ...,
   *    "options": {...}
   *  },
   *  "options": {...}
   * };
   *
   * const connection = new Connection(config);
   * ```
   *
   * @param config
   */
  constructor(config) {
    super();
    this.fedAuthRequired = void 0;
    this.config = void 0;
    this.secureContext = void 0;
    this.inTransaction = void 0;
    this.transactionDescriptors = void 0;
    this.transactionDepth = void 0;
    this.isSqlBatch = void 0;
    this.curTransientRetryCount = void 0;
    this.transientErrorLookup = void 0;
    this.closed = void 0;
    this.loginError = void 0;
    this.debug = void 0;
    this.ntlmpacket = void 0;
    this.ntlmpacketBuffer = void 0;
    this.routingData = void 0;
    this.messageIo = void 0;
    this.state = void 0;
    this.resetConnectionOnNextRequest = void 0;
    this.request = void 0;
    this.procReturnStatusValue = void 0;
    this.socket = void 0;
    this.messageBuffer = void 0;
    this.connectTimer = void 0;
    this.cancelTimer = void 0;
    this.requestTimer = void 0;
    this.retryTimer = void 0;
    this._cancelAfterRequestSent = void 0;
    this.databaseCollation = void 0;

    if (typeof config !== 'object' || config === null) {
      throw new TypeError('The "config" argument is required and must be of type Object.');
    }

    if (typeof config.server !== 'string') {
      throw new TypeError('The "config.server" property is required and must be of type string.');
    }

    this.fedAuthRequired = false;
    let authentication;

    if (config.authentication !== undefined) {
      if (typeof config.authentication !== 'object' || config.authentication === null) {
        throw new TypeError('The "config.authentication" property must be of type Object.');
      }

      const type = config.authentication.type;
      const options = config.authentication.options === undefined ? {} : config.authentication.options;

      if (typeof type !== 'string') {
        throw new TypeError('The "config.authentication.type" property must be of type string.');
      }

      if (type !== 'default' && type !== 'ntlm' && type !== 'azure-active-directory-password' && type !== 'azure-active-directory-access-token' && type !== 'azure-active-directory-msi-vm' && type !== 'azure-active-directory-msi-app-service' && type !== 'azure-active-directory-service-principal-secret' && type !== 'azure-active-directory-default') {
        throw new TypeError('The "type" property must one of "default", "ntlm", "azure-active-directory-password", "azure-active-directory-access-token", "azure-active-directory-default", "azure-active-directory-msi-vm" or "azure-active-directory-msi-app-service" or "azure-active-directory-service-principal-secret".');
      }

      if (typeof options !== 'object' || options === null) {
        throw new TypeError('The "config.authentication.options" property must be of type object.');
      }

      if (type === 'ntlm') {
        if (typeof options.domain !== 'string') {
          throw new TypeError('The "config.authentication.options.domain" property must be of type string.');
        }

        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        authentication = {
          type: 'ntlm',
          options: {
            userName: options.userName,
            password: options.password,
            domain: options.domain && options.domain.toUpperCase()
          }
        };
      } else if (type === 'azure-active-directory-password') {
        var _options$tenantId, _options$clientId;

        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        } else if (options.clientId === undefined) {
          emitAzureADPasswordClientIdDeprecationWarning();
        }

        if (options.domain !== undefined && typeof options.domain !== 'string') {
          throw new TypeError('The "config.authentication.options.domain" property must be of type string.');
        } else if (options.domain !== undefined) {
          emitDomainRenameToTenantIdWarning();
        }

        if (options.tenantId !== undefined && typeof options.tenantId !== 'string') {
          throw new TypeError('The "config.authentication.options.tenantId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-password',
          options: {
            userName: options.userName,
            password: options.password,
            tenantId: (_options$tenantId = options.tenantId) !== null && _options$tenantId !== void 0 ? _options$tenantId : options.domain,
            clientId: (_options$clientId = options.clientId) !== null && _options$clientId !== void 0 ? _options$clientId : '7f98cb04-cd1e-40df-9140-3bf7e2cea4db'
          }
        };
      } else if (type === 'azure-active-directory-access-token') {
        if (typeof options.token !== 'string') {
          throw new TypeError('The "config.authentication.options.token" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-access-token',
          options: {
            token: options.token
          }
        };
      } else if (type === 'azure-active-directory-msi-vm') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-msi-vm',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-default') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-default',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-msi-app-service') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-msi-app-service',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-service-principal-secret') {
        if (typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        if (typeof options.clientSecret !== 'string') {
          throw new TypeError('The "config.authentication.options.clientSecret" property must be of type string.');
        }

        if (typeof options.tenantId !== 'string') {
          throw new TypeError('The "config.authentication.options.tenantId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-service-principal-secret',
          options: {
            clientId: options.clientId,
            clientSecret: options.clientSecret,
            tenantId: options.tenantId
          }
        };
      } else {
        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        authentication = {
          type: 'default',
          options: {
            userName: options.userName,
            password: options.password
          }
        };
      }
    } else {
      authentication = {
        type: 'default',
        options: {
          userName: undefined,
          password: undefined
        }
      };
    }

    this.config = {
      server: config.server,
      authentication: authentication,
      options: {
        abortTransactionOnError: false,
        appName: undefined,
        camelCaseColumns: false,
        cancelTimeout: DEFAULT_CANCEL_TIMEOUT,
        columnEncryptionKeyCacheTTL: 2 * 60 * 60 * 1000,
        // Units: miliseconds
        columnEncryptionSetting: false,
        columnNameReplacer: undefined,
        connectionRetryInterval: DEFAULT_CONNECT_RETRY_INTERVAL,
        connectTimeout: DEFAULT_CONNECT_TIMEOUT,
        connectionIsolationLevel: _transaction.ISOLATION_LEVEL.READ_COMMITTED,
        cryptoCredentialsDetails: {},
        database: undefined,
        datefirst: DEFAULT_DATEFIRST,
        dateFormat: DEFAULT_DATEFORMAT,
        debug: {
          data: false,
          packet: false,
          payload: false,
          token: false
        },
        enableAnsiNull: true,
        enableAnsiNullDefault: true,
        enableAnsiPadding: true,
        enableAnsiWarnings: true,
        enableArithAbort: true,
        enableConcatNullYieldsNull: true,
        enableCursorCloseOnCommit: null,
        enableImplicitTransactions: false,
        enableNumericRoundabort: false,
        enableQuotedIdentifier: true,
        encrypt: true,
        fallbackToDefaultDb: false,
        encryptionKeyStoreProviders: undefined,
        instanceName: undefined,
        isolationLevel: _transaction.ISOLATION_LEVEL.READ_COMMITTED,
        language: DEFAULT_LANGUAGE,
        localAddress: undefined,
        maxRetriesOnTransientErrors: 3,
        multiSubnetFailover: false,
        packetSize: DEFAULT_PACKET_SIZE,
        port: DEFAULT_PORT,
        readOnlyIntent: false,
        requestTimeout: DEFAULT_CLIENT_REQUEST_TIMEOUT,
        rowCollectionOnDone: false,
        rowCollectionOnRequestCompletion: false,
        serverName: undefined,
        serverSupportsColumnEncryption: false,
        tdsVersion: DEFAULT_TDS_VERSION,
        textsize: DEFAULT_TEXTSIZE,
        trustedServerNameAE: undefined,
        trustServerCertificate: true,
        useColumnNames: false,
        useUTC: true,
        workstationId: undefined,
        lowerCaseGuids: false
      }
    };

    if (config.options) {
      if (config.options.port && config.options.instanceName) {
        throw new Error('Port and instanceName are mutually exclusive, but ' + config.options.port + ' and ' + config.options.instanceName + ' provided');
      }

      if (config.options.abortTransactionOnError !== undefined) {
        if (typeof config.options.abortTransactionOnError !== 'boolean' && config.options.abortTransactionOnError !== null) {
          throw new TypeError('The "config.options.abortTransactionOnError" property must be of type string or null.');
        }

        this.config.options.abortTransactionOnError = config.options.abortTransactionOnError;
      }

      if (config.options.appName !== undefined) {
        if (typeof config.options.appName !== 'string') {
          throw new TypeError('The "config.options.appName" property must be of type string.');
        }

        this.config.options.appName = config.options.appName;
      }

      if (config.options.camelCaseColumns !== undefined) {
        if (typeof config.options.camelCaseColumns !== 'boolean') {
          throw new TypeError('The "config.options.camelCaseColumns" property must be of type boolean.');
        }

        this.config.options.camelCaseColumns = config.options.camelCaseColumns;
      }

      if (config.options.cancelTimeout !== undefined) {
        if (typeof config.options.cancelTimeout !== 'number') {
          throw new TypeError('The "config.options.cancelTimeout" property must be of type number.');
        }

        this.config.options.cancelTimeout = config.options.cancelTimeout;
      }

      if (config.options.columnNameReplacer) {
        if (typeof config.options.columnNameReplacer !== 'function') {
          throw new TypeError('The "config.options.cancelTimeout" property must be of type function.');
        }

        this.config.options.columnNameReplacer = config.options.columnNameReplacer;
      }

      if (config.options.connectTimeout !== undefined) {
        if (typeof config.options.connectTimeout !== 'number') {
          throw new TypeError('The "config.options.connectTimeout" property must be of type number.');
        }

        this.config.options.connectTimeout = config.options.connectTimeout;
      }

      if (config.options.connectionIsolationLevel !== undefined) {
        (0, _transaction.assertValidIsolationLevel)(config.options.connectionIsolationLevel, 'config.options.connectionIsolationLevel');
        this.config.options.connectionIsolationLevel = config.options.connectionIsolationLevel;
      }

      if (config.options.connectTimeout !== undefined) {
        if (typeof config.options.connectTimeout !== 'number') {
          throw new TypeError('The "config.options.connectTimeout" property must be of type number.');
        }

        this.config.options.connectTimeout = config.options.connectTimeout;
      }

      if (config.options.cryptoCredentialsDetails !== undefined) {
        if (typeof config.options.cryptoCredentialsDetails !== 'object' || config.options.cryptoCredentialsDetails === null) {
          throw new TypeError('The "config.options.cryptoCredentialsDetails" property must be of type Object.');
        }

        this.config.options.cryptoCredentialsDetails = config.options.cryptoCredentialsDetails;
      }

      if (config.options.database !== undefined) {
        if (typeof config.options.database !== 'string') {
          throw new TypeError('The "config.options.database" property must be of type string.');
        }

        this.config.options.database = config.options.database;
      }

      if (config.options.datefirst !== undefined) {
        if (typeof config.options.datefirst !== 'number' && config.options.datefirst !== null) {
          throw new TypeError('The "config.options.datefirst" property must be of type number.');
        }

        if (config.options.datefirst !== null && (config.options.datefirst < 1 || config.options.datefirst > 7)) {
          throw new RangeError('The "config.options.datefirst" property must be >= 1 and <= 7');
        }

        this.config.options.datefirst = config.options.datefirst;
      }

      if (config.options.dateFormat !== undefined) {
        if (typeof config.options.dateFormat !== 'string' && config.options.dateFormat !== null) {
          throw new TypeError('The "config.options.dateFormat" property must be of type string or null.');
        }

        this.config.options.dateFormat = config.options.dateFormat;
      }

      if (config.options.debug) {
        if (config.options.debug.data !== undefined) {
          if (typeof config.options.debug.data !== 'boolean') {
            throw new TypeError('The "config.options.debug.data" property must be of type boolean.');
          }

          this.config.options.debug.data = config.options.debug.data;
        }

        if (config.options.debug.packet !== undefined) {
          if (typeof config.options.debug.packet !== 'boolean') {
            throw new TypeError('The "config.options.debug.packet" property must be of type boolean.');
          }

          this.config.options.debug.packet = config.options.debug.packet;
        }

        if (config.options.debug.payload !== undefined) {
          if (typeof config.options.debug.payload !== 'boolean') {
            throw new TypeError('The "config.options.debug.payload" property must be of type boolean.');
          }

          this.config.options.debug.payload = config.options.debug.payload;
        }

        if (config.options.debug.token !== undefined) {
          if (typeof config.options.debug.token !== 'boolean') {
            throw new TypeError('The "config.options.debug.token" property must be of type boolean.');
          }

          this.config.options.debug.token = config.options.debug.token;
        }
      }

      if (config.options.enableAnsiNull !== undefined) {
        if (typeof config.options.enableAnsiNull !== 'boolean' && config.options.enableAnsiNull !== null) {
          throw new TypeError('The "config.options.enableAnsiNull" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiNull = config.options.enableAnsiNull;
      }

      if (config.options.enableAnsiNullDefault !== undefined) {
        if (typeof config.options.enableAnsiNullDefault !== 'boolean' && config.options.enableAnsiNullDefault !== null) {
          throw new TypeError('The "config.options.enableAnsiNullDefault" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiNullDefault = config.options.enableAnsiNullDefault;
      }

      if (config.options.enableAnsiPadding !== undefined) {
        if (typeof config.options.enableAnsiPadding !== 'boolean' && config.options.enableAnsiPadding !== null) {
          throw new TypeError('The "config.options.enableAnsiPadding" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiPadding = config.options.enableAnsiPadding;
      }

      if (config.options.enableAnsiWarnings !== undefined) {
        if (typeof config.options.enableAnsiWarnings !== 'boolean' && config.options.enableAnsiWarnings !== null) {
          throw new TypeError('The "config.options.enableAnsiWarnings" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiWarnings = config.options.enableAnsiWarnings;
      }

      if (config.options.enableArithAbort !== undefined) {
        if (typeof config.options.enableArithAbort !== 'boolean' && config.options.enableArithAbort !== null) {
          throw new TypeError('The "config.options.enableArithAbort" property must be of type boolean or null.');
        }

        this.config.options.enableArithAbort = config.options.enableArithAbort;
      }

      if (config.options.enableConcatNullYieldsNull !== undefined) {
        if (typeof config.options.enableConcatNullYieldsNull !== 'boolean' && config.options.enableConcatNullYieldsNull !== null) {
          throw new TypeError('The "config.options.enableConcatNullYieldsNull" property must be of type boolean or null.');
        }

        this.config.options.enableConcatNullYieldsNull = config.options.enableConcatNullYieldsNull;
      }

      if (config.options.enableCursorCloseOnCommit !== undefined) {
        if (typeof config.options.enableCursorCloseOnCommit !== 'boolean' && config.options.enableCursorCloseOnCommit !== null) {
          throw new TypeError('The "config.options.enableCursorCloseOnCommit" property must be of type boolean or null.');
        }

        this.config.options.enableCursorCloseOnCommit = config.options.enableCursorCloseOnCommit;
      }

      if (config.options.enableImplicitTransactions !== undefined) {
        if (typeof config.options.enableImplicitTransactions !== 'boolean' && config.options.enableImplicitTransactions !== null) {
          throw new TypeError('The "config.options.enableImplicitTransactions" property must be of type boolean or null.');
        }

        this.config.options.enableImplicitTransactions = config.options.enableImplicitTransactions;
      }

      if (config.options.enableNumericRoundabort !== undefined) {
        if (typeof config.options.enableNumericRoundabort !== 'boolean' && config.options.enableNumericRoundabort !== null) {
          throw new TypeError('The "config.options.enableNumericRoundabort" property must be of type boolean or null.');
        }

        this.config.options.enableNumericRoundabort = config.options.enableNumericRoundabort;
      }

      if (config.options.enableQuotedIdentifier !== undefined) {
        if (typeof config.options.enableQuotedIdentifier !== 'boolean' && config.options.enableQuotedIdentifier !== null) {
          throw new TypeError('The "config.options.enableQuotedIdentifier" property must be of type boolean or null.');
        }

        this.config.options.enableQuotedIdentifier = config.options.enableQuotedIdentifier;
      }

      if (config.options.encrypt !== undefined) {
        if (typeof config.options.encrypt !== 'boolean') {
          throw new TypeError('The "config.options.encrypt" property must be of type boolean.');
        }

        this.config.options.encrypt = config.options.encrypt;
      }

      if (config.options.fallbackToDefaultDb !== undefined) {
        if (typeof config.options.fallbackToDefaultDb !== 'boolean') {
          throw new TypeError('The "config.options.fallbackToDefaultDb" property must be of type boolean.');
        }

        this.config.options.fallbackToDefaultDb = config.options.fallbackToDefaultDb;
      }

      if (config.options.instanceName !== undefined) {
        if (typeof config.options.instanceName !== 'string') {
          throw new TypeError('The "config.options.instanceName" property must be of type string.');
        }

        this.config.options.instanceName = config.options.instanceName;
        this.config.options.port = undefined;
      }

      if (config.options.isolationLevel !== undefined) {
        (0, _transaction.assertValidIsolationLevel)(config.options.isolationLevel, 'config.options.isolationLevel');
        this.config.options.isolationLevel = config.options.isolationLevel;
      }

      if (config.options.language !== undefined) {
        if (typeof config.options.language !== 'string' && config.options.language !== null) {
          throw new TypeError('The "config.options.language" property must be of type string or null.');
        }

        this.config.options.language = config.options.language;
      }

      if (config.options.localAddress !== undefined) {
        if (typeof config.options.localAddress !== 'string') {
          throw new TypeError('The "config.options.localAddress" property must be of type string.');
        }

        this.config.options.localAddress = config.options.localAddress;
      }

      if (config.options.multiSubnetFailover !== undefined) {
        if (typeof config.options.multiSubnetFailover !== 'boolean') {
          throw new TypeError('The "config.options.multiSubnetFailover" property must be of type boolean.');
        }

        this.config.options.multiSubnetFailover = config.options.multiSubnetFailover;
      }

      if (config.options.packetSize !== undefined) {
        if (typeof config.options.packetSize !== 'number') {
          throw new TypeError('The "config.options.packetSize" property must be of type number.');
        }

        this.config.options.packetSize = config.options.packetSize;
      }

      if (config.options.port !== undefined) {
        if (typeof config.options.port !== 'number') {
          throw new TypeError('The "config.options.port" property must be of type number.');
        }

        if (config.options.port <= 0 || config.options.port >= 65536) {
          throw new RangeError('The "config.options.port" property must be > 0 and < 65536');
        }

        this.config.options.port = config.options.port;
        this.config.options.instanceName = undefined;
      }

      if (config.options.readOnlyIntent !== undefined) {
        if (typeof config.options.readOnlyIntent !== 'boolean') {
          throw new TypeError('The "config.options.readOnlyIntent" property must be of type boolean.');
        }

        this.config.options.readOnlyIntent = config.options.readOnlyIntent;
      }

      if (config.options.requestTimeout !== undefined) {
        if (typeof config.options.requestTimeout !== 'number') {
          throw new TypeError('The "config.options.requestTimeout" property must be of type number.');
        }

        this.config.options.requestTimeout = config.options.requestTimeout;
      }

      if (config.options.maxRetriesOnTransientErrors !== undefined) {
        if (typeof config.options.maxRetriesOnTransientErrors !== 'number') {
          throw new TypeError('The "config.options.maxRetriesOnTransientErrors" property must be of type number.');
        }

        if (config.options.maxRetriesOnTransientErrors < 0) {
          throw new TypeError('The "config.options.maxRetriesOnTransientErrors" property must be equal or greater than 0.');
        }

        this.config.options.maxRetriesOnTransientErrors = config.options.maxRetriesOnTransientErrors;
      }

      if (config.options.connectionRetryInterval !== undefined) {
        if (typeof config.options.connectionRetryInterval !== 'number') {
          throw new TypeError('The "config.options.connectionRetryInterval" property must be of type number.');
        }

        if (config.options.connectionRetryInterval <= 0) {
          throw new TypeError('The "config.options.connectionRetryInterval" property must be greater than 0.');
        }

        this.config.options.connectionRetryInterval = config.options.connectionRetryInterval;
      }

      if (config.options.rowCollectionOnDone !== undefined) {
        if (typeof config.options.rowCollectionOnDone !== 'boolean') {
          throw new TypeError('The "config.options.rowCollectionOnDone" property must be of type boolean.');
        }

        this.config.options.rowCollectionOnDone = config.options.rowCollectionOnDone;
      }

      if (config.options.rowCollectionOnRequestCompletion !== undefined) {
        if (typeof config.options.rowCollectionOnRequestCompletion !== 'boolean') {
          throw new TypeError('The "config.options.rowCollectionOnRequestCompletion" property must be of type boolean.');
        }

        this.config.options.rowCollectionOnRequestCompletion = config.options.rowCollectionOnRequestCompletion;
      }

      if (config.options.tdsVersion !== undefined) {
        if (typeof config.options.tdsVersion !== 'string') {
          throw new TypeError('The "config.options.tdsVersion" property must be of type string.');
        }

        this.config.options.tdsVersion = config.options.tdsVersion;
      }

      if (config.options.textsize !== undefined) {
        if (typeof config.options.textsize !== 'number' && config.options.textsize !== null) {
          throw new TypeError('The "config.options.textsize" property must be of type number or null.');
        }

        if (config.options.textsize > 2147483647) {
          throw new TypeError('The "config.options.textsize" can\'t be greater than 2147483647.');
        } else if (config.options.textsize < -1) {
          throw new TypeError('The "config.options.textsize" can\'t be smaller than -1.');
        }

        this.config.options.textsize = config.options.textsize | 0;
      }

      if (config.options.trustServerCertificate !== undefined) {
        if (typeof config.options.trustServerCertificate !== 'boolean') {
          throw new TypeError('The "config.options.trustServerCertificate" property must be of type boolean.');
        }

        this.config.options.trustServerCertificate = config.options.trustServerCertificate;
      } else {
        emitTrustServerCertificateWarning();
      }

      if (config.options.useColumnNames !== undefined) {
        if (typeof config.options.useColumnNames !== 'boolean') {
          throw new TypeError('The "config.options.useColumnNames" property must be of type boolean.');
        }

        this.config.options.useColumnNames = config.options.useColumnNames;
      }

      if (config.options.useUTC !== undefined) {
        if (typeof config.options.useUTC !== 'boolean') {
          throw new TypeError('The "config.options.useUTC" property must be of type boolean.');
        }

        this.config.options.useUTC = config.options.useUTC;
      }

      if (config.options.workstationId !== undefined) {
        if (typeof config.options.workstationId !== 'string') {
          throw new TypeError('The "config.options.workstationId" property must be of type string.');
        }

        this.config.options.workstationId = config.options.workstationId;
      }

      if (config.options.lowerCaseGuids !== undefined) {
        if (typeof config.options.lowerCaseGuids !== 'boolean') {
          throw new TypeError('The "config.options.lowerCaseGuids" property must be of type boolean.');
        }

        this.config.options.lowerCaseGuids = config.options.lowerCaseGuids;
      }
    }

    let credentialsDetails = this.config.options.cryptoCredentialsDetails;

    if (credentialsDetails.secureOptions === undefined) {
      // If the caller has not specified their own `secureOptions`,
      // we set `SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS` here.
      // Older SQL Server instances running on older Windows versions have
      // trouble with the BEAST workaround in OpenSSL.
      // As BEAST is a browser specific exploit, we can just disable this option here.
      credentialsDetails = Object.create(credentialsDetails, {
        secureOptions: {
          value: _constants.default.SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS
        }
      });
    }

    this.secureContext = (0, _tls.createSecureContext)(credentialsDetails);
    this.debug = this.createDebug();
    this.inTransaction = false;
    this.transactionDescriptors = [Buffer.from([0, 0, 0, 0, 0, 0, 0, 0])]; // 'beginTransaction', 'commitTransaction' and 'rollbackTransaction'
    // events are utilized to maintain inTransaction property state which in
    // turn is used in managing transactions. These events are only fired for
    // TDS version 7.2 and beyond. The properties below are used to emulate
    // equivalent behavior for TDS versions before 7.2.

    this.transactionDepth = 0;
    this.isSqlBatch = false;
    this.closed = false;
    this.messageBuffer = Buffer.alloc(0);
    this.curTransientRetryCount = 0;
    this.transientErrorLookup = new _transientErrorLookup.TransientErrorLookup();
    this.state = this.STATE.INITIALIZED;

    this._cancelAfterRequestSent = () => {
      this.messageIo.sendMessage(_packet.TYPE.ATTENTION);
      this.createCancelTimer();
    };
  }

  connect(connectListener) {
    if (this.state !== this.STATE.INITIALIZED) {
      throw new _errors.ConnectionError('`.connect` can not be called on a Connection in `' + this.state.name + '` state.');
    }

    if (connectListener) {
      const onConnect = err => {
        this.removeListener('error', onError);
        connectListener(err);
      };

      const onError = err => {
        this.removeListener('connect', onConnect);
        connectListener(err);
      };

      this.once('connect', onConnect);
      this.once('error', onError);
    }

    this.transitionTo(this.STATE.CONNECTING);
  }
  /**
   * The server has reported that the charset has changed.
   */


  on(event, listener) {
    return super.on(event, listener);
  }
  /**
   * @private
   */


  emit(event, ...args) {
    return super.emit(event, ...args);
  }
  /**
   * Closes the connection to the database.
   *
   * The [[Event_end]] will be emitted once the connection has been closed.
   */


  close() {
    this.transitionTo(this.STATE.FINAL);
  }
  /**
   * @private
   */


  initialiseConnection() {
    const signal = this.createConnectTimer();

    if (this.config.options.port) {
      return this.connectOnPort(this.config.options.port, this.config.options.multiSubnetFailover, signal);
    } else {
      return new _instanceLookup.InstanceLookup().instanceLookup({
        server: this.config.server,
        instanceName: this.config.options.instanceName,
        timeout: this.config.options.connectTimeout,
        signal: signal
      }, (err, port) => {
        if (err) {
          if (err.name === 'AbortError') {
            return;
          }

          this.emit('connect', new _errors.ConnectionError(err.message, 'EINSTLOOKUP'));
        } else {
          this.connectOnPort(port, this.config.options.multiSubnetFailover, signal);
        }
      });
    }
  }
  /**
   * @private
   */


  cleanupConnection(cleanupType) {
    if (!this.closed) {
      this.clearConnectTimer();
      this.clearRequestTimer();
      this.clearRetryTimer();
      this.closeConnection();

      if (cleanupType === CLEANUP_TYPE.REDIRECT) {
        this.emit('rerouting');
      } else if (cleanupType !== CLEANUP_TYPE.RETRY) {
        process.nextTick(() => {
          this.emit('end');
        });
      }

      const request = this.request;

      if (request) {
        const err = new _errors.RequestError('Connection closed before request completed.', 'ECLOSE');
        request.callback(err);
        this.request = undefined;
      }

      this.closed = true;
      this.loginError = undefined;
    }
  }
  /**
   * @private
   */


  createDebug() {
    const debug = new _debug.default(this.config.options.debug);
    debug.on('debug', message => {
      this.emit('debug', message);
    });
    return debug;
  }
  /**
   * @private
   */


  createTokenStreamParser(message, handler) {
    return new _tokenStreamParser.Parser(message, this.debug, handler, this.config.options);
  }

  connectOnPort(port, multiSubnetFailover, signal) {
    const connectOpts = {
      host: this.routingData ? this.routingData.server : this.config.server,
      port: this.routingData ? this.routingData.port : port,
      localAddress: this.config.options.localAddress
    };
    new _connector.Connector(connectOpts, signal, multiSubnetFailover).execute((err, socket) => {
      if (err) {
        if (err.name === 'AbortError') {
          return;
        }

        return this.socketError(err);
      }

      socket = socket;
      socket.on('error', error => {
        this.socketError(error);
      });
      socket.on('close', () => {
        this.socketClose();
      });
      socket.on('end', () => {
        this.socketEnd();
      });
      socket.setKeepAlive(true, KEEP_ALIVE_INITIAL_DELAY);
      this.messageIo = new _messageIo.default(socket, this.config.options.packetSize, this.debug);
      this.messageIo.on('secure', cleartext => {
        this.emit('secure', cleartext);
      });
      this.socket = socket;
      this.closed = false;
      this.debug.log('connected to ' + this.config.server + ':' + this.config.options.port);
      this.sendPreLogin();
      this.transitionTo(this.STATE.SENT_PRELOGIN);
    });
  }
  /**
   * @private
   */


  closeConnection() {
    if (this.socket) {
      this.socket.destroy();
    }
  }
  /**
   * @private
   */


  createConnectTimer() {
    const controller = new _nodeAbortController.AbortController();
    this.connectTimer = setTimeout(() => {
      controller.abort();
      this.connectTimeout();
    }, this.config.options.connectTimeout);
    return controller.signal;
  }
  /**
   * @private
   */


  createCancelTimer() {
    this.clearCancelTimer();
    const timeout = this.config.options.cancelTimeout;

    if (timeout > 0) {
      this.cancelTimer = setTimeout(() => {
        this.cancelTimeout();
      }, timeout);
    }
  }
  /**
   * @private
   */


  createRequestTimer() {
    this.clearRequestTimer(); // release old timer, just to be safe

    const request = this.request;
    const timeout = request.timeout !== undefined ? request.timeout : this.config.options.requestTimeout;

    if (timeout) {
      this.requestTimer = setTimeout(() => {
        this.requestTimeout();
      }, timeout);
    }
  }
  /**
   * @private
   */


  createRetryTimer() {
    this.clearRetryTimer();
    this.retryTimer = setTimeout(() => {
      this.retryTimeout();
    }, this.config.options.connectionRetryInterval);
  }
  /**
   * @private
   */


  connectTimeout() {
    const message = `Failed to connect to ${this.config.server}${this.config.options.port ? `:${this.config.options.port}` : `\\${this.config.options.instanceName}`} in ${this.config.options.connectTimeout}ms`;
    this.debug.log(message);
    this.emit('connect', new _errors.ConnectionError(message, 'ETIMEOUT'));
    this.connectTimer = undefined;
    this.dispatchEvent('connectTimeout');
  }
  /**
   * @private
   */


  cancelTimeout() {
    const message = `Failed to cancel request in ${this.config.options.cancelTimeout}ms`;
    this.debug.log(message);
    this.dispatchEvent('socketError', new _errors.ConnectionError(message, 'ETIMEOUT'));
  }
  /**
   * @private
   */


  requestTimeout() {
    this.requestTimer = undefined;
    const request = this.request;
    request.cancel();
    const timeout = request.timeout !== undefined ? request.timeout : this.config.options.requestTimeout;
    const message = 'Timeout: Request failed to complete in ' + timeout + 'ms';
    request.error = new _errors.RequestError(message, 'ETIMEOUT');
  }
  /**
   * @private
   */


  retryTimeout() {
    this.retryTimer = undefined;
    this.emit('retry');
    this.transitionTo(this.STATE.CONNECTING);
  }
  /**
   * @private
   */


  clearConnectTimer() {
    if (this.connectTimer) {
      clearTimeout(this.connectTimer);
      this.connectTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearCancelTimer() {
    if (this.cancelTimer) {
      clearTimeout(this.cancelTimer);
      this.cancelTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearRequestTimer() {
    if (this.requestTimer) {
      clearTimeout(this.requestTimer);
      this.requestTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearRetryTimer() {
    if (this.retryTimer) {
      clearTimeout(this.retryTimer);
      this.retryTimer = undefined;
    }
  }
  /**
   * @private
   */


  transitionTo(newState) {
    if (this.state === newState) {
      this.debug.log('State is already ' + newState.name);
      return;
    }

    if (this.state && this.state.exit) {
      this.state.exit.call(this, newState);
    }

    this.debug.log('State change: ' + (this.state ? this.state.name : 'undefined') + ' -> ' + newState.name);
    this.state = newState;

    if (this.state.enter) {
      this.state.enter.apply(this);
    }
  }
  /**
   * @private
   */


  getEventHandler(eventName) {
    const handler = this.state.events[eventName];

    if (!handler) {
      throw new Error(`No event '${eventName}' in state '${this.state.name}'`);
    }

    return handler;
  }
  /**
   * @private
   */


  dispatchEvent(eventName, ...args) {
    const handler = this.state.events[eventName];

    if (handler) {
      handler.apply(this, args);
    } else {
      this.emit('error', new Error(`No event '${eventName}' in state '${this.state.name}'`));
      this.close();
    }
  }
  /**
   * @private
   */


  socketError(error) {
    if (this.state === this.STATE.CONNECTING || this.state === this.STATE.SENT_TLSSSLNEGOTIATION) {
      const message = `Failed to connect to ${this.config.server}:${this.config.options.port} - ${error.message}`;
      this.debug.log(message);
      this.emit('connect', new _errors.ConnectionError(message, 'ESOCKET'));
    } else {
      const message = `Connection lost - ${error.message}`;
      this.debug.log(message);
      this.emit('error', new _errors.ConnectionError(message, 'ESOCKET'));
    }

    this.dispatchEvent('socketError', error);
  }
  /**
   * @private
   */


  socketEnd() {
    this.debug.log('socket ended');

    if (this.state !== this.STATE.FINAL) {
      const error = new Error('socket hang up');
      error.code = 'ECONNRESET';
      this.socketError(error);
    }
  }
  /**
   * @private
   */


  socketClose() {
    this.debug.log('connection to ' + this.config.server + ':' + this.config.options.port + ' closed');

    if (this.state === this.STATE.REROUTING) {
      this.debug.log('Rerouting to ' + this.routingData.server + ':' + this.routingData.port);
      this.dispatchEvent('reconnect');
    } else if (this.state === this.STATE.TRANSIENT_FAILURE_RETRY) {
      const server = this.routingData ? this.routingData.server : this.config.server;
      const port = this.routingData ? this.routingData.port : this.config.options.port;
      this.debug.log('Retry after transient failure connecting to ' + server + ':' + port);
      this.dispatchEvent('retry');
    } else {
      this.transitionTo(this.STATE.FINAL);
    }
  }
  /**
   * @private
   */


  sendPreLogin() {
    var _$exec;

    const [, major, minor, build] = (_$exec = /^(\d+)\.(\d+)\.(\d+)/.exec(_package.version)) !== null && _$exec !== void 0 ? _$exec : ['0.0.0', '0', '0', '0'];
    const payload = new _preloginPayload.default({
      encrypt: this.config.options.encrypt,
      version: {
        major: Number(major),
        minor: Number(minor),
        build: Number(build),
        subbuild: 0
      }
    });
    this.messageIo.sendMessage(_packet.TYPE.PRELOGIN, payload.data);
    this.debug.payload(function () {
      return payload.toString('  ');
    });
  }
  /**
   * @private
   */


  emptyMessageBuffer() {
    this.messageBuffer = Buffer.alloc(0);
  }
  /**
   * @private
   */


  addToMessageBuffer(data) {
    this.messageBuffer = Buffer.concat([this.messageBuffer, data]);
  }
  /**
   * @private
   */


  sendLogin7Packet() {
    const payload = new _login7Payload.default({
      tdsVersion: _tdsVersions.versions[this.config.options.tdsVersion],
      packetSize: this.config.options.packetSize,
      clientProgVer: 0,
      clientPid: process.pid,
      connectionId: 0,
      clientTimeZone: new Date().getTimezoneOffset(),
      clientLcid: 0x00000409
    });
    const {
      authentication
    } = this.config;

    switch (authentication.type) {
      case 'azure-active-directory-password':
        payload.fedAuth = {
          type: 'ADAL',
          echo: this.fedAuthRequired,
          workflow: 'default'
        };
        break;

      case 'azure-active-directory-access-token':
        payload.fedAuth = {
          type: 'SECURITYTOKEN',
          echo: this.fedAuthRequired,
          fedAuthToken: authentication.options.token
        };
        break;

      case 'azure-active-directory-msi-vm':
      case 'azure-active-directory-default':
      case 'azure-active-directory-msi-app-service':
      case 'azure-active-directory-service-principal-secret':
        payload.fedAuth = {
          type: 'ADAL',
          echo: this.fedAuthRequired,
          workflow: 'integrated'
        };
        break;

      case 'ntlm':
        payload.sspi = (0, _ntlm.createNTLMRequest)({
          domain: authentication.options.domain
        });
        break;

      default:
        payload.userName = authentication.options.userName;
        payload.password = authentication.options.password;
    }

    payload.hostname = this.config.options.workstationId || _os.default.hostname();
    payload.serverName = this.routingData ? this.routingData.server : this.config.server;
    payload.appName = this.config.options.appName || 'Tedious';
    payload.libraryName = _library.name;
    payload.language = this.config.options.language;
    payload.database = this.config.options.database;
    payload.clientId = Buffer.from([1, 2, 3, 4, 5, 6]);
    payload.readOnlyIntent = this.config.options.readOnlyIntent;
    payload.initDbFatal = !this.config.options.fallbackToDefaultDb;
    this.routingData = undefined;
    this.messageIo.sendMessage(_packet.TYPE.LOGIN7, payload.toBuffer());
    this.debug.payload(function () {
      return payload.toString('  ');
    });
  }
  /**
   * @private
   */


  sendFedAuthTokenMessage(token) {
    const accessTokenLen = Buffer.byteLength(token, 'ucs2');
    const data = Buffer.alloc(8 + accessTokenLen);
    let offset = 0;
    offset = data.writeUInt32LE(accessTokenLen + 4, offset);
    offset = data.writeUInt32LE(accessTokenLen, offset);
    data.write(token, offset, 'ucs2');
    this.messageIo.sendMessage(_packet.TYPE.FEDAUTH_TOKEN, data); // sent the fedAuth token message, the rest is similar to standard login 7

    this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
  }
  /**
   * @private
   */


  sendInitialSql() {
    const payload = new _sqlbatchPayload.default(this.getInitialSql(), this.currentTransactionDescriptor(), this.config.options);
    const message = new _message.default({
      type: _packet.TYPE.SQL_BATCH
    });
    this.messageIo.outgoingMessageStream.write(message);

    _stream.Readable.from(payload).pipe(message);
  }
  /**
   * @private
   */


  getInitialSql() {
    const options = [];

    if (this.config.options.enableAnsiNull === true) {
      options.push('set ansi_nulls on');
    } else if (this.config.options.enableAnsiNull === false) {
      options.push('set ansi_nulls off');
    }

    if (this.config.options.enableAnsiNullDefault === true) {
      options.push('set ansi_null_dflt_on on');
    } else if (this.config.options.enableAnsiNullDefault === false) {
      options.push('set ansi_null_dflt_on off');
    }

    if (this.config.options.enableAnsiPadding === true) {
      options.push('set ansi_padding on');
    } else if (this.config.options.enableAnsiPadding === false) {
      options.push('set ansi_padding off');
    }

    if (this.config.options.enableAnsiWarnings === true) {
      options.push('set ansi_warnings on');
    } else if (this.config.options.enableAnsiWarnings === false) {
      options.push('set ansi_warnings off');
    }

    if (this.config.options.enableArithAbort === true) {
      options.push('set arithabort on');
    } else if (this.config.options.enableArithAbort === false) {
      options.push('set arithabort off');
    }

    if (this.config.options.enableConcatNullYieldsNull === true) {
      options.push('set concat_null_yields_null on');
    } else if (this.config.options.enableConcatNullYieldsNull === false) {
      options.push('set concat_null_yields_null off');
    }

    if (this.config.options.enableCursorCloseOnCommit === true) {
      options.push('set cursor_close_on_commit on');
    } else if (this.config.options.enableCursorCloseOnCommit === false) {
      options.push('set cursor_close_on_commit off');
    }

    if (this.config.options.datefirst !== null) {
      options.push(`set datefirst ${this.config.options.datefirst}`);
    }

    if (this.config.options.dateFormat !== null) {
      options.push(`set dateformat ${this.config.options.dateFormat}`);
    }

    if (this.config.options.enableImplicitTransactions === true) {
      options.push('set implicit_transactions on');
    } else if (this.config.options.enableImplicitTransactions === false) {
      options.push('set implicit_transactions off');
    }

    if (this.config.options.language !== null) {
      options.push(`set language ${this.config.options.language}`);
    }

    if (this.config.options.enableNumericRoundabort === true) {
      options.push('set numeric_roundabort on');
    } else if (this.config.options.enableNumericRoundabort === false) {
      options.push('set numeric_roundabort off');
    }

    if (this.config.options.enableQuotedIdentifier === true) {
      options.push('set quoted_identifier on');
    } else if (this.config.options.enableQuotedIdentifier === false) {
      options.push('set quoted_identifier off');
    }

    if (this.config.options.textsize !== null) {
      options.push(`set textsize ${this.config.options.textsize}`);
    }

    if (this.config.options.connectionIsolationLevel !== null) {
      options.push(`set transaction isolation level ${this.getIsolationLevelText(this.config.options.connectionIsolationLevel)}`);
    }

    if (this.config.options.abortTransactionOnError === true) {
      options.push('set xact_abort on');
    } else if (this.config.options.abortTransactionOnError === false) {
      options.push('set xact_abort off');
    }

    return options.join('\n');
  }
  /**
   * @private
   */


  processedInitialSql() {
    this.clearConnectTimer();
    this.emit('connect');
  }
  /**
   * Execute the SQL batch represented by [[Request]].
   * There is no param support, and unlike [[Request.execSql]],
   * it is not likely that SQL Server will reuse the execution plan it generates for the SQL.
   *
   * In almost all cases, [[Request.execSql]] will be a better choice.
   *
   * @param request A [[Request]] object representing the request.
   */


  execSqlBatch(request) {
    this.makeRequest(request, _packet.TYPE.SQL_BATCH, new _sqlbatchPayload.default(request.sqlTextOrProcedure, this.currentTransactionDescriptor(), this.config.options));
  }
  /**
   *  Execute the SQL represented by [[Request]].
   *
   * As `sp_executesql` is used to execute the SQL, if the same SQL is executed multiples times
   * using this function, the SQL Server query optimizer is likely to reuse the execution plan it generates
   * for the first execution. This may also result in SQL server treating the request like a stored procedure
   * which can result in the [[Event_doneInProc]] or [[Event_doneProc]] events being emitted instead of the
   * [[Event_done]] event you might expect. Using [[execSqlBatch]] will prevent this from occurring but may have a negative performance impact.
   *
   * Beware of the way that scoping rules apply, and how they may [affect local temp tables](http://weblogs.sqlteam.com/mladenp/archive/2006/11/03/17197.aspx)
   * If you're running in to scoping issues, then [[execSqlBatch]] may be a better choice.
   * See also [issue #24](https://github.com/pekim/tedious/issues/24)
   *
   * @param request A [[Request]] object representing the request.
   */


  execSql(request) {
    try {
      request.validateParameters(this.databaseCollation);
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'statement',
      value: request.sqlTextOrProcedure,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });

    if (request.parameters.length) {
      parameters.push({
        type: _dataType.TYPES.NVarChar,
        name: 'params',
        value: request.makeParamsParameter(request.parameters),
        output: false,
        length: undefined,
        precision: undefined,
        scale: undefined
      });
      parameters.push(...request.parameters);
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_executesql', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Creates a new BulkLoad instance.
   *
   * @param table The name of the table to bulk-insert into.
   * @param options A set of bulk load options.
   */


  newBulkLoad(table, callbackOrOptions, callback) {
    let options;

    if (callback === undefined) {
      callback = callbackOrOptions;
      options = {};
    } else {
      options = callbackOrOptions;
    }

    if (typeof options !== 'object') {
      throw new TypeError('"options" argument must be an object');
    }

    return new _bulkLoad.default(table, this.databaseCollation, this.config.options, options, callback);
  }
  /**
   * Execute a [[BulkLoad]].
   *
   * ```js
   * // We want to perform a bulk load into a table with the following format:
   * // CREATE TABLE employees (first_name nvarchar(255), last_name nvarchar(255), day_of_birth date);
   *
   * const bulkLoad = connection.newBulkLoad('employees', (err, rowCount) => {
   *   // ...
   * });
   *
   * // First, we need to specify the columns that we want to write to,
   * // and their definitions. These definitions must match the actual table,
   * // otherwise the bulk load will fail.
   * bulkLoad.addColumn('first_name', TYPES.NVarchar, { nullable: false });
   * bulkLoad.addColumn('last_name', TYPES.NVarchar, { nullable: false });
   * bulkLoad.addColumn('date_of_birth', TYPES.Date, { nullable: false });
   *
   * // Execute a bulk load with a predefined list of rows.
   * //
   * // Note that these rows are held in memory until the
   * // bulk load was performed, so if you need to write a large
   * // number of rows (e.g. by reading from a CSV file),
   * // passing an `AsyncIterable` is advisable to keep memory usage low.
   * connection.execBulkLoad(bulkLoad, [
   *   { 'first_name': 'Steve', 'last_name': 'Jobs', 'day_of_birth': new Date('02-24-1955') },
   *   { 'first_name': 'Bill', 'last_name': 'Gates', 'day_of_birth': new Date('10-28-1955') }
   * ]);
   * ```
   *
   * @param bulkLoad A previously created [[BulkLoad]].
   * @param rows A [[Iterable]] or [[AsyncIterable]] that contains the rows that should be bulk loaded.
   */


  execBulkLoad(bulkLoad, rows) {
    bulkLoad.executionStarted = true;

    if (rows) {
      if (bulkLoad.streamingMode) {
        throw new Error("Connection.execBulkLoad can't be called with a BulkLoad that was put in streaming mode.");
      }

      if (bulkLoad.firstRowWritten) {
        throw new Error("Connection.execBulkLoad can't be called with a BulkLoad that already has rows written to it.");
      }

      const rowStream = _stream.Readable.from(rows); // Destroy the packet transform if an error happens in the row stream,
      // e.g. if an error is thrown from within a generator or stream.


      rowStream.on('error', err => {
        bulkLoad.rowToPacketTransform.destroy(err);
      }); // Destroy the row stream if an error happens in the packet transform,
      // e.g. if the bulk load is cancelled.

      bulkLoad.rowToPacketTransform.on('error', err => {
        rowStream.destroy(err);
      });
      rowStream.pipe(bulkLoad.rowToPacketTransform);
    } else if (!bulkLoad.streamingMode) {
      // If the bulkload was not put into streaming mode by the user,
      // we end the rowToPacketTransform here for them.
      //
      // If it was put into streaming mode, it's the user's responsibility
      // to end the stream.
      bulkLoad.rowToPacketTransform.end();
    }

    const onCancel = () => {
      request.cancel();
    };

    const payload = new _bulkLoadPayload.BulkLoadPayload(bulkLoad);
    const request = new _request.default(bulkLoad.getBulkInsertSql(), error => {
      bulkLoad.removeListener('cancel', onCancel);

      if (error) {
        if (error.code === 'UNKNOWN') {
          error.message += ' This is likely because the schema of the BulkLoad does not match the schema of the table you are attempting to insert into.';
        }

        bulkLoad.error = error;
        bulkLoad.callback(error);
        return;
      }

      this.makeRequest(bulkLoad, _packet.TYPE.BULK_LOAD, payload);
    });
    bulkLoad.once('cancel', onCancel);
    this.execSqlBatch(request);
  }
  /**
   * Prepare the SQL represented by the request.
   *
   * The request can then be used in subsequent calls to
   * [[execute]] and [[unprepare]]
   *
   * @param request A [[Request]] object representing the request.
   *   Parameters only require a name and type. Parameter values are ignored.
   */


  prepare(request) {
    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      value: undefined,
      output: true,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'params',
      value: request.parameters.length ? request.makeParamsParameter(request.parameters) : null,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'stmt',
      value: request.sqlTextOrProcedure,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    request.preparing = true; // TODO: We need to clean up this event handler, otherwise this leaks memory

    request.on('returnValue', (name, value) => {
      if (name === 'handle') {
        request.handle = value;
      } else {
        request.error = new _errors.RequestError(`Tedious > Unexpected output parameter ${name} from sp_prepare`);
      }
    });
    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_prepare', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Release the SQL Server resources associated with a previously prepared request.
   *
   * @param request A [[Request]] object representing the request.
   *   Parameters only require a name and type.
   *   Parameter values are ignored.
   */


  unprepare(request) {
    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      // TODO: Abort if `request.handle` is not set
      value: request.handle,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_unprepare', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Execute previously prepared SQL, using the supplied parameters.
   *
   * @param request A previously prepared [[Request]].
   * @param parameters  An object whose names correspond to the names of
   *   parameters that were added to the [[Request]] before it was prepared.
   *   The object's values are passed as the parameters' values when the
   *   request is executed.
   */


  execute(request, parameters) {
    const executeParameters = [];
    executeParameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      // TODO: Abort if `request.handle` is not set
      value: request.handle,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });

    try {
      for (let i = 0, len = request.parameters.length; i < len; i++) {
        const parameter = request.parameters[i];
        executeParameters.push({ ...parameter,
          value: parameter.type.validate(parameters ? parameters[parameter.name] : null, this.databaseCollation)
        });
      }
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_execute', executeParameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Call a stored procedure represented by [[Request]].
   *
   * @param request A [[Request]] object representing the request.
   */


  callProcedure(request) {
    try {
      request.validateParameters(this.databaseCollation);
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default(request.sqlTextOrProcedure, request.parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Start a transaction.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string. Required when `isolationLevel`
   *   is present.
   * @param isolationLevel The isolation level that the transaction is to be run with.
   *
   *   The isolation levels are available from `require('tedious').ISOLATION_LEVEL`.
   *   * `READ_UNCOMMITTED`
   *   * `READ_COMMITTED`
   *   * `REPEATABLE_READ`
   *   * `SERIALIZABLE`
   *   * `SNAPSHOT`
   *
   *   Optional, and defaults to the Connection's isolation level.
   */


  beginTransaction(callback, name = '', isolationLevel = this.config.options.isolationLevel) {
    (0, _transaction.assertValidIsolationLevel)(isolationLevel, 'isolationLevel');
    const transaction = new _transaction.Transaction(name, isolationLevel);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('SET TRANSACTION ISOLATION LEVEL ' + transaction.isolationLevelToTSQL() + ';BEGIN TRAN ' + transaction.name, err => {
        this.transactionDepth++;

        if (this.transactionDepth === 1) {
          this.inTransaction = true;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, err => {
      return callback(err, this.currentTransactionDescriptor());
    });
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.beginPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Commit a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string. Required when `isolationLevel`is present.
   */


  commitTransaction(callback, name = '') {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('COMMIT TRAN ' + transaction.name, err => {
        this.transactionDepth--;

        if (this.transactionDepth === 0) {
          this.inTransaction = false;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.commitPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Rollback a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string.
   *   Required when `isolationLevel` is present.
   */


  rollbackTransaction(callback, name = '') {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('ROLLBACK TRAN ' + transaction.name, err => {
        this.transactionDepth--;

        if (this.transactionDepth === 0) {
          this.inTransaction = false;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.rollbackPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Set a savepoint within a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.\
   *   Optional, and defaults to an empty string.
   *   Required when `isolationLevel` is present.
   */


  saveTransaction(callback, name) {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('SAVE TRAN ' + transaction.name, err => {
        this.transactionDepth++;
        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.savePayload(this.currentTransactionDescriptor()));
  }
  /**
   * Run the given callback after starting a transaction, and commit or
   * rollback the transaction afterwards.
   *
   * This is a helper that employs [[beginTransaction]], [[commitTransaction]],
   * [[rollbackTransaction]], and [[saveTransaction]] to greatly simplify the
   * use of database transactions and automatically handle transaction nesting.
   *
   * @param cb
   * @param isolationLevel
   *   The isolation level that the transaction is to be run with.
   *
   *   The isolation levels are available from `require('tedious').ISOLATION_LEVEL`.
   *   * `READ_UNCOMMITTED`
   *   * `READ_COMMITTED`
   *   * `REPEATABLE_READ`
   *   * `SERIALIZABLE`
   *   * `SNAPSHOT`
   *
   *   Optional, and defaults to the Connection's isolation level.
   */


  transaction(cb, isolationLevel) {
    if (typeof cb !== 'function') {
      throw new TypeError('`cb` must be a function');
    }

    const useSavepoint = this.inTransaction;

    const name = '_tedious_' + _crypto.default.randomBytes(10).toString('hex');

    const txDone = (err, done, ...args) => {
      if (err) {
        if (this.inTransaction && this.state === this.STATE.LOGGED_IN) {
          this.rollbackTransaction(txErr => {
            done(txErr || err, ...args);
          }, name);
        } else {
          done(err, ...args);
        }
      } else if (useSavepoint) {
        if (this.config.options.tdsVersion < '7_2') {
          this.transactionDepth--;
        }

        done(null, ...args);
      } else {
        this.commitTransaction(txErr => {
          done(txErr, ...args);
        }, name);
      }
    };

    if (useSavepoint) {
      return this.saveTransaction(err => {
        if (err) {
          return cb(err);
        }

        if (isolationLevel) {
          return this.execSqlBatch(new _request.default('SET transaction isolation level ' + this.getIsolationLevelText(isolationLevel), err => {
            return cb(err, txDone);
          }));
        } else {
          return cb(null, txDone);
        }
      }, name);
    } else {
      return this.beginTransaction(err => {
        if (err) {
          return cb(err);
        }

        return cb(null, txDone);
      }, name, isolationLevel);
    }
  }
  /**
   * @private
   */


  makeRequest(request, packetType, payload) {
    if (this.state !== this.STATE.LOGGED_IN) {
      const message = 'Requests can only be made in the ' + this.STATE.LOGGED_IN.name + ' state, not the ' + this.state.name + ' state';
      this.debug.log(message);
      request.callback(new _errors.RequestError(message, 'EINVALIDSTATE'));
    } else if (request.canceled) {
      process.nextTick(() => {
        request.callback(new _errors.RequestError('Canceled.', 'ECANCEL'));
      });
    } else {
      if (packetType === _packet.TYPE.SQL_BATCH) {
        this.isSqlBatch = true;
      } else {
        this.isSqlBatch = false;
      }

      this.request = request;
      request.connection = this;
      request.rowCount = 0;
      request.rows = [];
      request.rst = [];

      const onCancel = () => {
        payloadStream.unpipe(message);
        payloadStream.destroy(new _errors.RequestError('Canceled.', 'ECANCEL')); // set the ignore bit and end the message.

        message.ignore = true;
        message.end();

        if (request instanceof _request.default && request.paused) {
          // resume the request if it was paused so we can read the remaining tokens
          request.resume();
        }
      };

      request.once('cancel', onCancel);
      this.createRequestTimer();
      const message = new _message.default({
        type: packetType,
        resetConnection: this.resetConnectionOnNextRequest
      });
      this.messageIo.outgoingMessageStream.write(message);
      this.transitionTo(this.STATE.SENT_CLIENT_REQUEST);
      message.once('finish', () => {
        request.removeListener('cancel', onCancel);
        request.once('cancel', this._cancelAfterRequestSent);
        this.resetConnectionOnNextRequest = false;
        this.debug.payload(function () {
          return payload.toString('  ');
        });
      });

      const payloadStream = _stream.Readable.from(payload);

      payloadStream.once('error', error => {
        var _request$error;

        payloadStream.unpipe(message); // Only set a request error if no error was set yet.

        (_request$error = request.error) !== null && _request$error !== void 0 ? _request$error : request.error = error;
        message.ignore = true;
        message.end();
      });
      payloadStream.pipe(message);
    }
  }
  /**
   * Cancel currently executed request.
   */


  cancel() {
    if (!this.request) {
      return false;
    }

    if (this.request.canceled) {
      return false;
    }

    this.request.cancel();
    return true;
  }
  /**
   * Reset the connection to its initial state.
   * Can be useful for connection pool implementations.
   *
   * @param callback
   */


  reset(callback) {
    const request = new _request.default(this.getInitialSql(), err => {
      if (this.config.options.tdsVersion < '7_2') {
        this.inTransaction = false;
      }

      callback(err);
    });
    this.resetConnectionOnNextRequest = true;
    this.execSqlBatch(request);
  }
  /**
   * @private
   */


  currentTransactionDescriptor() {
    return this.transactionDescriptors[this.transactionDescriptors.length - 1];
  }
  /**
   * @private
   */


  getIsolationLevelText(isolationLevel) {
    switch (isolationLevel) {
      case _transaction.ISOLATION_LEVEL.READ_UNCOMMITTED:
        return 'read uncommitted';

      case _transaction.ISOLATION_LEVEL.REPEATABLE_READ:
        return 'repeatable read';

      case _transaction.ISOLATION_LEVEL.SERIALIZABLE:
        return 'serializable';

      case _transaction.ISOLATION_LEVEL.SNAPSHOT:
        return 'snapshot';

      default:
        return 'read committed';
    }
  }

}

let azureADPasswordClientIdDeprecationWarningEmitted = false;

function emitAzureADPasswordClientIdDeprecationWarning() {
  if (azureADPasswordClientIdDeprecationWarningEmitted) {
    return;
  }

  azureADPasswordClientIdDeprecationWarningEmitted = true;
  process.emitWarning('When using the `azure-active-directory-password` authentication method, please provide a value for the `clientId` option. ' + 'This option will be required in a future release.', 'DeprecationWarning', Connection.prototype.on);
}

function isTransientError(error) {
  if (error instanceof _esAggregateError.default) {
    error = error.errors[0];
  }

  return error instanceof _errors.ConnectionError && !!error.isTransient;
}

var _default = Connection;
exports.default = _default;
module.exports = Connection;
Connection.prototype.STATE = {
  INITIALIZED: {
    name: 'Initialized',
    events: {}
  },
  CONNECTING: {
    name: 'Connecting',
    enter: function () {
      this.initialiseConnection();
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_PRELOGIN: {
    name: 'SentPrelogin',
    enter: function () {
      (async () => {
        let messageBuffer = Buffer.alloc(0);
        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        }

        for await (const data of message) {
          messageBuffer = Buffer.concat([messageBuffer, data]);
        }

        const preloginPayload = new _preloginPayload.default(messageBuffer);
        this.debug.payload(function () {
          return preloginPayload.toString('  ');
        });

        if (preloginPayload.fedAuthRequired === 1) {
          this.fedAuthRequired = true;
        }

        if (preloginPayload.encryptionString === 'ON' || preloginPayload.encryptionString === 'REQ') {
          var _this$routingData$ser, _this$routingData;

          if (!this.config.options.encrypt) {
            this.emit('connect', new _errors.ConnectionError("Server requires encryption, set 'encrypt' config option to true.", 'EENCRYPT'));
            return this.close();
          }

          this.messageIo.startTls(this.secureContext, (_this$routingData$ser = (_this$routingData = this.routingData) === null || _this$routingData === void 0 ? void 0 : _this$routingData.server) !== null && _this$routingData$ser !== void 0 ? _this$routingData$ser : this.config.server, this.config.options.trustServerCertificate);
          this.transitionTo(this.STATE.SENT_TLSSSLNEGOTIATION);
        } else {
          this.sendLogin7Packet();
          const {
            authentication
          } = this.config;

          if (authentication.type === 'ntlm') {
            this.transitionTo(this.STATE.SENT_LOGIN7_WITH_NTLM);
          } else {
            this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
          }
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  REROUTING: {
    name: 'ReRouting',
    enter: function () {
      this.cleanupConnection(CLEANUP_TYPE.REDIRECT);
    },
    events: {
      message: function () {},
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      reconnect: function () {
        this.transitionTo(this.STATE.CONNECTING);
      }
    }
  },
  TRANSIENT_FAILURE_RETRY: {
    name: 'TRANSIENT_FAILURE_RETRY',
    enter: function () {
      this.curTransientRetryCount++;
      this.cleanupConnection(CLEANUP_TYPE.RETRY);
    },
    events: {
      message: function () {},
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      retry: function () {
        this.createRetryTimer();
      }
    }
  },
  SENT_TLSSSLNEGOTIATION: {
    name: 'SentTLSSSLNegotiation',
    enter: function () {
      (async () => {
        while (!this.messageIo.tlsNegotiationComplete) {
          let message;

          try {
            message = await this.messageIo.readMessage();
          } catch (err) {
            return this.socketError(err);
          }

          for await (const data of message) {
            this.messageIo.tlsHandshakeData(data);
          }
        }

        this.sendLogin7Packet();
        const {
          authentication
        } = this.config;

        switch (authentication.type) {
          case 'azure-active-directory-password':
          case 'azure-active-directory-msi-vm':
          case 'azure-active-directory-msi-app-service':
          case 'azure-active-directory-service-principal-secret':
          case 'azure-active-directory-default':
            this.transitionTo(this.STATE.SENT_LOGIN7_WITH_FEDAUTH);
            break;

          case 'ntlm':
            this.transitionTo(this.STATE.SENT_LOGIN7_WITH_NTLM);
            break;

          default:
            this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
            break;
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_LOGIN7_WITH_STANDARD_LOGIN: {
    name: 'SentLogin7WithStandardLogin',
    enter: function () {
      (async () => {
        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        }

        const handler = new _handler.Login7TokenHandler(this);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        await (0, _events.once)(tokenStreamParser, 'end');

        if (handler.loginAckReceived) {
          if (handler.routingData) {
            this.routingData = handler.routingData;
            this.transitionTo(this.STATE.REROUTING);
          } else {
            this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
          }
        } else if (this.loginError) {
          if (isTransientError(this.loginError)) {
            this.debug.log('Initiating retry on transient error');
            this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
          } else {
            this.emit('connect', this.loginError);
            this.transitionTo(this.STATE.FINAL);
          }
        } else {
          this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
          this.transitionTo(this.STATE.FINAL);
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_LOGIN7_WITH_NTLM: {
    name: 'SentLogin7WithNTLMLogin',
    enter: function () {
      (async () => {
        while (true) {
          let message;

          try {
            message = await this.messageIo.readMessage();
          } catch (err) {
            return this.socketError(err);
          }

          const handler = new _handler.Login7TokenHandler(this);
          const tokenStreamParser = this.createTokenStreamParser(message, handler);
          await (0, _events.once)(tokenStreamParser, 'end');

          if (handler.loginAckReceived) {
            if (handler.routingData) {
              this.routingData = handler.routingData;
              return this.transitionTo(this.STATE.REROUTING);
            } else {
              return this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
            }
          } else if (this.ntlmpacket) {
            const authentication = this.config.authentication;
            const payload = new _ntlmPayload.default({
              domain: authentication.options.domain,
              userName: authentication.options.userName,
              password: authentication.options.password,
              ntlmpacket: this.ntlmpacket
            });
            this.messageIo.sendMessage(_packet.TYPE.NTLMAUTH_PKT, payload.data);
            this.debug.payload(function () {
              return payload.toString('  ');
            });
            this.ntlmpacket = undefined;
          } else if (this.loginError) {
            if (isTransientError(this.loginError)) {
              this.debug.log('Initiating retry on transient error');
              return this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
            } else {
              this.emit('connect', this.loginError);
              return this.transitionTo(this.STATE.FINAL);
            }
          } else {
            this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
            return this.transitionTo(this.STATE.FINAL);
          }
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_LOGIN7_WITH_FEDAUTH: {
    name: 'SentLogin7Withfedauth',
    enter: function () {
      (async () => {
        var _authentication$optio;

        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        }

        const handler = new _handler.Login7TokenHandler(this);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        await (0, _events.once)(tokenStreamParser, 'end');

        if (handler.loginAckReceived) {
          if (handler.routingData) {
            this.routingData = handler.routingData;
            this.transitionTo(this.STATE.REROUTING);
          } else {
            this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
          }

          return;
        }

        const fedAuthInfoToken = handler.fedAuthInfoToken;

        if (fedAuthInfoToken && fedAuthInfoToken.stsurl && fedAuthInfoToken.spn) {
          const authentication = this.config.authentication;
          const tokenScope = new _url.URL('/.default', fedAuthInfoToken.spn).toString();
          let credentials;

          switch (authentication.type) {
            case 'azure-active-directory-password':
              credentials = new _identity.UsernamePasswordCredential((_authentication$optio = authentication.options.tenantId) !== null && _authentication$optio !== void 0 ? _authentication$optio : 'common', authentication.options.clientId, authentication.options.userName, authentication.options.password);
              break;

            case 'azure-active-directory-msi-vm':
            case 'azure-active-directory-msi-app-service':
              const msiArgs = authentication.options.clientId ? [authentication.options.clientId, {}] : [{}];
              credentials = new _identity.ManagedIdentityCredential(...msiArgs);
              break;

            case 'azure-active-directory-default':
              const args = authentication.options.clientId ? {
                managedIdentityClientId: authentication.options.clientId
              } : {};
              credentials = new _identity.DefaultAzureCredential(args);
              break;

            case 'azure-active-directory-service-principal-secret':
              credentials = new _identity.ClientSecretCredential(authentication.options.tenantId, authentication.options.clientId, authentication.options.clientSecret);
              break;
          }

          let tokenResponse;

          try {
            tokenResponse = await credentials.getToken(tokenScope);
          } catch (err) {
            this.loginError = new _esAggregateError.default([new _errors.ConnectionError('Security token could not be authenticated or authorized.', 'EFEDAUTH'), err]);
            this.emit('connect', this.loginError);
            this.transitionTo(this.STATE.FINAL);
            return;
          }

          const token = tokenResponse.token;
          this.sendFedAuthTokenMessage(token);
        } else if (this.loginError) {
          if (isTransientError(this.loginError)) {
            this.debug.log('Initiating retry on transient error');
            this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
          } else {
            this.emit('connect', this.loginError);
            this.transitionTo(this.STATE.FINAL);
          }
        } else {
          this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
          this.transitionTo(this.STATE.FINAL);
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  LOGGED_IN_SENDING_INITIAL_SQL: {
    name: 'LoggedInSendingInitialSql',
    enter: function () {
      (async () => {
        this.sendInitialSql();
        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        }

        const tokenStreamParser = this.createTokenStreamParser(message, new _handler.InitialSqlTokenHandler(this));
        await (0, _events.once)(tokenStreamParser, 'end');
        this.transitionTo(this.STATE.LOGGED_IN);
        this.processedInitialSql();
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function socketError() {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  LOGGED_IN: {
    name: 'LoggedIn',
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_CLIENT_REQUEST: {
    name: 'SentClientRequest',
    enter: function () {
      (async () => {
        var _this$request, _this$request3, _this$request10;

        this.emptyMessageBuffer();
        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        } // request timer is stopped on first data package


        this.clearRequestTimer();
        const tokenStreamParser = this.createTokenStreamParser(message, new _handler.RequestTokenHandler(this, this.request)); // If the request was canceled and we have a `cancelTimer`
        // defined, we send a attention message after the
        // request message was fully sent off.
        //
        // We already started consuming the current message
        // (but all the token handlers should be no-ops), and
        // need to ensure the next message is handled by the
        // `SENT_ATTENTION` state.

        if ((_this$request = this.request) !== null && _this$request !== void 0 && _this$request.canceled && this.cancelTimer) {
          return this.transitionTo(this.STATE.SENT_ATTENTION);
        }

        const onResume = () => {
          tokenStreamParser.resume();
        };

        const onPause = () => {
          var _this$request2;

          tokenStreamParser.pause();
          (_this$request2 = this.request) === null || _this$request2 === void 0 ? void 0 : _this$request2.once('resume', onResume);
        };

        (_this$request3 = this.request) === null || _this$request3 === void 0 ? void 0 : _this$request3.on('pause', onPause);

        if (this.request instanceof _request.default && this.request.paused) {
          onPause();
        }

        const onCancel = () => {
          var _this$request4, _this$request5;

          tokenStreamParser.removeListener('end', onEndOfMessage);

          if (this.request instanceof _request.default && this.request.paused) {
            // resume the request if it was paused so we can read the remaining tokens
            this.request.resume();
          }

          (_this$request4 = this.request) === null || _this$request4 === void 0 ? void 0 : _this$request4.removeListener('pause', onPause);
          (_this$request5 = this.request) === null || _this$request5 === void 0 ? void 0 : _this$request5.removeListener('resume', onResume); // The `_cancelAfterRequestSent` callback will have sent a
          // attention message, so now we need to also switch to
          // the `SENT_ATTENTION` state to make sure the attention ack
          // message is processed correctly.

          this.transitionTo(this.STATE.SENT_ATTENTION);
        };

        const onEndOfMessage = () => {
          var _this$request6, _this$request7, _this$request8, _this$request9;

          (_this$request6 = this.request) === null || _this$request6 === void 0 ? void 0 : _this$request6.removeListener('cancel', this._cancelAfterRequestSent);
          (_this$request7 = this.request) === null || _this$request7 === void 0 ? void 0 : _this$request7.removeListener('cancel', onCancel);
          (_this$request8 = this.request) === null || _this$request8 === void 0 ? void 0 : _this$request8.removeListener('pause', onPause);
          (_this$request9 = this.request) === null || _this$request9 === void 0 ? void 0 : _this$request9.removeListener('resume', onResume);
          this.transitionTo(this.STATE.LOGGED_IN);
          const sqlRequest = this.request;
          this.request = undefined;

          if (this.config.options.tdsVersion < '7_2' && sqlRequest.error && this.isSqlBatch) {
            this.inTransaction = false;
          }

          sqlRequest.callback(sqlRequest.error, sqlRequest.rowCount, sqlRequest.rows);
        };

        tokenStreamParser.once('end', onEndOfMessage);
        (_this$request10 = this.request) === null || _this$request10 === void 0 ? void 0 : _this$request10.once('cancel', onCancel);
      })();
    },
    exit: function (nextState) {
      this.clearRequestTimer();
    },
    events: {
      socketError: function (err) {
        const sqlRequest = this.request;
        this.request = undefined;
        this.transitionTo(this.STATE.FINAL);
        sqlRequest.callback(err);
      }
    }
  },
  SENT_ATTENTION: {
    name: 'SentAttention',
    enter: function () {
      (async () => {
        this.emptyMessageBuffer();
        let message;

        try {
          message = await this.messageIo.readMessage();
        } catch (err) {
          return this.socketError(err);
        }

        const handler = new _handler.AttentionTokenHandler(this, this.request);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        await (0, _events.once)(tokenStreamParser, 'end'); // 3.2.5.7 Sent Attention State
        // Discard any data contained in the response, until we receive the attention response

        if (handler.attentionReceived) {
          this.clearCancelTimer();
          const sqlRequest = this.request;
          this.request = undefined;
          this.transitionTo(this.STATE.LOGGED_IN);

          if (sqlRequest.error && sqlRequest.error instanceof _errors.RequestError && sqlRequest.error.code === 'ETIMEOUT') {
            sqlRequest.callback(sqlRequest.error);
          } else {
            sqlRequest.callback(new _errors.RequestError('Canceled.', 'ECANCEL'));
          }
        }
      })().catch(err => {
        process.nextTick(() => {
          throw err;
        });
      });
    },
    events: {
      socketError: function (err) {
        const sqlRequest = this.request;
        this.request = undefined;
        this.transitionTo(this.STATE.FINAL);
        sqlRequest.callback(err);
      }
    }
  },
  FINAL: {
    name: 'Final',
    enter: function () {
      this.cleanupConnection(CLEANUP_TYPE.NORMAL);
    },
    events: {
      connectTimeout: function () {// Do nothing, as the timer should be cleaned up.
      },
      message: function () {// Do nothing
      },
      socketError: function () {// Do nothing
      }
    }
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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