"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SequentialConnectionStrategy = exports.ParallelConnectionStrategy = exports.Connector = void 0;

var _net = _interopRequireDefault(require("net"));

var _dns = _interopRequireDefault(require("dns"));

var punycode = _interopRequireWildcard(require("punycode"));

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class AbortError extends Error {
  constructor() {
    super('The operation was aborted');
    this.code = void 0;
    this.code = 'ABORT_ERR';
    this.name = 'AbortError';
  }

}

class ParallelConnectionStrategy {
  constructor(addresses, signal, options) {
    this.addresses = void 0;
    this.options = void 0;
    this.signal = void 0;
    this.addresses = addresses;
    this.options = options;
    this.signal = signal;
  }

  connect(callback) {
    const signal = this.signal;

    if (signal.aborted) {
      return process.nextTick(callback, new AbortError());
    }

    const addresses = this.addresses;
    const sockets = new Array(addresses.length);
    let errorCount = 0;

    function onError(_err) {
      errorCount += 1;
      this.removeListener('error', onError);
      this.removeListener('connect', onConnect);
      this.destroy();

      if (errorCount === addresses.length) {
        signal.removeEventListener('abort', onAbort);
        callback(new Error('Could not connect (parallel)'));
      }
    }

    function onConnect() {
      signal.removeEventListener('abort', onAbort);

      for (let j = 0; j < sockets.length; j++) {
        const socket = sockets[j];

        if (this === socket) {
          continue;
        }

        socket.removeListener('error', onError);
        socket.removeListener('connect', onConnect);
        socket.destroy();
      }

      callback(null, this);
    }

    const onAbort = () => {
      for (let j = 0; j < sockets.length; j++) {
        const socket = sockets[j];
        socket.removeListener('error', onError);
        socket.removeListener('connect', onConnect);
        socket.destroy();
      }

      callback(new AbortError());
    };

    for (let i = 0, len = addresses.length; i < len; i++) {
      const socket = sockets[i] = _net.default.connect({ ...this.options,
        host: addresses[i].address,
        family: addresses[i].family
      });

      socket.on('error', onError);
      socket.on('connect', onConnect);
    }

    signal.addEventListener('abort', onAbort, {
      once: true
    });
  }

}

exports.ParallelConnectionStrategy = ParallelConnectionStrategy;

class SequentialConnectionStrategy {
  constructor(addresses, signal, options) {
    this.addresses = void 0;
    this.options = void 0;
    this.signal = void 0;
    this.addresses = addresses;
    this.options = options;
    this.signal = signal;
  }

  connect(callback) {
    if (this.signal.aborted) {
      return process.nextTick(callback, new AbortError());
    }

    const next = this.addresses.shift();

    if (!next) {
      return callback(new Error('Could not connect (sequence)'));
    }

    const socket = _net.default.connect({ ...this.options,
      host: next.address,
      family: next.family
    });

    const onAbort = () => {
      socket.removeListener('error', onError);
      socket.removeListener('connect', onConnect);
      socket.destroy();
      callback(new AbortError());
    };

    const onError = _err => {
      this.signal.removeEventListener('abort', onAbort);
      socket.removeListener('error', onError);
      socket.removeListener('connect', onConnect);
      socket.destroy();
      this.connect(callback);
    };

    const onConnect = () => {
      this.signal.removeEventListener('abort', onAbort);
      socket.removeListener('error', onError);
      socket.removeListener('connect', onConnect);
      callback(null, socket);
    };

    this.signal.addEventListener('abort', onAbort, {
      once: true
    });
    socket.on('error', onError);
    socket.on('connect', onConnect);
  }

}

exports.SequentialConnectionStrategy = SequentialConnectionStrategy;

class Connector {
  constructor(options, signal, multiSubnetFailover) {
    var _options$lookup;

    this.options = void 0;
    this.multiSubnetFailover = void 0;
    this.lookup = void 0;
    this.signal = void 0;
    this.options = options;
    this.lookup = (_options$lookup = options.lookup) !== null && _options$lookup !== void 0 ? _options$lookup : _dns.default.lookup;
    this.signal = signal;
    this.multiSubnetFailover = multiSubnetFailover;
  }

  execute(cb) {
    if (this.signal.aborted) {
      return process.nextTick(cb, new AbortError());
    }

    this.lookupAllAddresses(this.options.host, (err, addresses) => {
      if (this.signal.aborted) {
        return cb(new AbortError());
      }

      if (err) {
        return cb(err);
      }

      if (this.multiSubnetFailover) {
        new ParallelConnectionStrategy(addresses, this.signal, this.options).connect(cb);
      } else {
        new SequentialConnectionStrategy(addresses, this.signal, this.options).connect(cb);
      }
    });
  }

  lookupAllAddresses(host, callback) {
    if (_net.default.isIPv6(host)) {
      process.nextTick(callback, null, [{
        address: host,
        family: 6
      }]);
    } else if (_net.default.isIPv4(host)) {
      process.nextTick(callback, null, [{
        address: host,
        family: 4
      }]);
    } else {
      this.lookup.call(null, punycode.toASCII(host), {
        all: true
      }, callback);
    }
  }

}

exports.Connector = Connector;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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