"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InstanceLookup = void 0;

var _dns = _interopRequireDefault(require("dns"));

var _nodeAbortController = require("node-abort-controller");

var _sender = require("./sender");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const SQL_SERVER_BROWSER_PORT = 1434;
const TIMEOUT = 2 * 1000;
const RETRIES = 3; // There are three bytes at the start of the response, whose purpose is unknown.

const MYSTERY_HEADER_LENGTH = 3;

class AbortError extends Error {
  constructor() {
    super('The operation was aborted');
    this.code = void 0;
    this.code = 'ABORT_ERR';
    this.name = 'AbortError';
  }

} // Most of the functionality has been determined from from jTDS's MSSqlServerInfo class.


class InstanceLookup {
  instanceLookup(options, callback) {
    var _options$lookup, _options$port;

    const server = options.server;

    if (typeof server !== 'string') {
      throw new TypeError('Invalid arguments: "server" must be a string');
    }

    const instanceName = options.instanceName;

    if (typeof instanceName !== 'string') {
      throw new TypeError('Invalid arguments: "instanceName" must be a string');
    }

    const timeout = options.timeout === undefined ? TIMEOUT : options.timeout;

    if (typeof timeout !== 'number') {
      throw new TypeError('Invalid arguments: "timeout" must be a number');
    }

    const retries = options.retries === undefined ? RETRIES : options.retries;

    if (typeof retries !== 'number') {
      throw new TypeError('Invalid arguments: "retries" must be a number');
    }

    if (options.lookup !== undefined && typeof options.lookup !== 'function') {
      throw new TypeError('Invalid arguments: "lookup" must be a function');
    }

    const lookup = (_options$lookup = options.lookup) !== null && _options$lookup !== void 0 ? _options$lookup : _dns.default.lookup;

    if (options.port !== undefined && typeof options.port !== 'number') {
      throw new TypeError('Invalid arguments: "port" must be a number');
    }

    const port = (_options$port = options.port) !== null && _options$port !== void 0 ? _options$port : SQL_SERVER_BROWSER_PORT;
    const signal = options.signal;

    if (typeof callback !== 'function') {
      throw new TypeError('Invalid arguments: "callback" must be a function');
    }

    if (signal.aborted) {
      return process.nextTick(callback, new AbortError());
    }

    let retriesLeft = retries;

    const makeAttempt = () => {
      if (retriesLeft >= 0) {
        retriesLeft--;
        const controller = new _nodeAbortController.AbortController();

        const abortCurrentAttempt = () => {
          controller.abort();
        }; // If the overall instance lookup is aborted,
        // forward the abort to the controller of the current
        // lookup attempt.


        signal.addEventListener('abort', abortCurrentAttempt, {
          once: true
        });
        const request = Buffer.from([0x02]);
        const sender = new _sender.Sender(options.server, port, lookup, controller.signal, request);
        const timer = setTimeout(abortCurrentAttempt, timeout);
        sender.execute((err, response) => {
          clearTimeout(timer);

          if (err) {
            if ((err === null || err === void 0 ? void 0 : err.name) === 'AbortError') {
              // If the overall instance lookup was aborted,
              // do not perform any further attempts.
              if (signal.aborted) {
                return callback(new AbortError());
              }

              return makeAttempt();
            }

            return callback(new Error('Failed to lookup instance on ' + server + ' - ' + err.message));
          }

          const message = response.toString('ascii', MYSTERY_HEADER_LENGTH);
          const port = this.parseBrowserResponse(message, instanceName);

          if (port) {
            callback(undefined, port);
          } else {
            callback(new Error('Port for ' + instanceName + ' not found in ' + options.server));
          }
        });
      } else {
        callback(new Error('Failed to get response from SQL Server Browser on ' + server));
      }
    };

    makeAttempt();
  }

  parseBrowserResponse(response, instanceName) {
    let getPort;
    const instances = response.split(';;');

    for (let i = 0, len = instances.length; i < len; i++) {
      const instance = instances[i];
      const parts = instance.split(';');

      for (let p = 0, partsLen = parts.length; p < partsLen; p += 2) {
        const name = parts[p];
        const value = parts[p + 1];

        if (name === 'tcp' && getPort) {
          const port = parseInt(value, 10);
          return port;
        }

        if (name === 'InstanceName') {
          if (value.toUpperCase() === instanceName.toUpperCase()) {
            getPort = true;
          } else {
            getPort = false;
          }
        }
      }
    }
  }

}

exports.InstanceLookup = InstanceLookup;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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