"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.readCollation = readCollation;

var _collation = require("./collation");

var _dataType = require("./data-type");

var _sprintfJs = require("sprintf-js");

function readCollation(parser, callback) {
  // s2.2.5.1.2
  parser.readBuffer(5, collationData => {
    callback(_collation.Collation.fromBuffer(collationData));
  });
}

function readSchema(parser, callback) {
  // s2.2.5.5.3
  parser.readUInt8(schemaPresent => {
    if (schemaPresent === 0x01) {
      parser.readBVarChar(dbname => {
        parser.readBVarChar(owningSchema => {
          parser.readUsVarChar(xmlSchemaCollection => {
            callback({
              dbname: dbname,
              owningSchema: owningSchema,
              xmlSchemaCollection: xmlSchemaCollection
            });
          });
        });
      });
    } else {
      callback(undefined);
    }
  });
}

function readUDTInfo(parser, callback) {
  parser.readUInt16LE(maxByteSize => {
    parser.readBVarChar(dbname => {
      parser.readBVarChar(owningSchema => {
        parser.readBVarChar(typeName => {
          parser.readUsVarChar(assemblyName => {
            callback({
              maxByteSize: maxByteSize,
              dbname: dbname,
              owningSchema: owningSchema,
              typeName: typeName,
              assemblyName: assemblyName
            });
          });
        });
      });
    });
  });
}

function metadataParse(parser, options, callback) {
  (options.tdsVersion < '7_2' ? parser.readUInt16LE : parser.readUInt32LE).call(parser, userType => {
    parser.readUInt16LE(flags => {
      parser.readUInt8(typeNumber => {
        const type = _dataType.TYPE[typeNumber];

        if (!type) {
          throw new Error((0, _sprintfJs.sprintf)('Unrecognised data type 0x%02X', typeNumber));
        }

        switch (type.name) {
          case 'Null':
          case 'TinyInt':
          case 'SmallInt':
          case 'Int':
          case 'BigInt':
          case 'Real':
          case 'Float':
          case 'SmallMoney':
          case 'Money':
          case 'Bit':
          case 'SmallDateTime':
          case 'DateTime':
          case 'Date':
            return callback({
              userType: userType,
              flags: flags,
              type: type,
              collation: undefined,
              precision: undefined,
              scale: undefined,
              dataLength: undefined,
              schema: undefined,
              udtInfo: undefined
            });

          case 'IntN':
          case 'FloatN':
          case 'MoneyN':
          case 'BitN':
          case 'UniqueIdentifier':
          case 'DateTimeN':
            return parser.readUInt8(dataLength => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: dataLength,
                schema: undefined,
                udtInfo: undefined
              });
            });

          case 'Variant':
            return parser.readUInt32LE(dataLength => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: dataLength,
                schema: undefined,
                udtInfo: undefined
              });
            });

          case 'VarChar':
          case 'Char':
          case 'NVarChar':
          case 'NChar':
            return parser.readUInt16LE(dataLength => {
              readCollation(parser, collation => {
                callback({
                  userType: userType,
                  flags: flags,
                  type: type,
                  collation: collation,
                  precision: undefined,
                  scale: undefined,
                  dataLength: dataLength,
                  schema: undefined,
                  udtInfo: undefined
                });
              });
            });

          case 'Text':
          case 'NText':
            return parser.readUInt32LE(dataLength => {
              readCollation(parser, collation => {
                callback({
                  userType: userType,
                  flags: flags,
                  type: type,
                  collation: collation,
                  precision: undefined,
                  scale: undefined,
                  dataLength: dataLength,
                  schema: undefined,
                  udtInfo: undefined
                });
              });
            });

          case 'VarBinary':
          case 'Binary':
            return parser.readUInt16LE(dataLength => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: dataLength,
                schema: undefined,
                udtInfo: undefined
              });
            });

          case 'Image':
            return parser.readUInt32LE(dataLength => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: dataLength,
                schema: undefined,
                udtInfo: undefined
              });
            });

          case 'Xml':
            return readSchema(parser, schema => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: undefined,
                schema: schema,
                udtInfo: undefined
              });
            });

          case 'Time':
          case 'DateTime2':
          case 'DateTimeOffset':
            return parser.readUInt8(scale => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: scale,
                dataLength: undefined,
                schema: undefined,
                udtInfo: undefined
              });
            });

          case 'NumericN':
          case 'DecimalN':
            return parser.readUInt8(dataLength => {
              parser.readUInt8(precision => {
                parser.readUInt8(scale => {
                  callback({
                    userType: userType,
                    flags: flags,
                    type: type,
                    collation: undefined,
                    precision: precision,
                    scale: scale,
                    dataLength: dataLength,
                    schema: undefined,
                    udtInfo: undefined
                  });
                });
              });
            });

          case 'UDT':
            return readUDTInfo(parser, udtInfo => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: undefined,
                schema: undefined,
                udtInfo: udtInfo
              });
            });

          default:
            throw new Error((0, _sprintfJs.sprintf)('Unrecognised type %s', type.name));
        }
      });
    });
  });
}

var _default = metadataParse;
exports.default = _default;
module.exports = metadataParse;
module.exports.readCollation = readCollation;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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