"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _writableTrackingBuffer = _interopRequireDefault(require("./tracking-buffer/writable-tracking-buffer"));

var crypto = _interopRequireWildcard(require("crypto"));

var _jsbi = _interopRequireDefault(require("jsbi"));

var _jsMd = _interopRequireDefault(require("js-md4"));

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class NTLMResponsePayload {
  constructor(loginData) {
    this.data = void 0;
    this.data = this.createResponse(loginData);
  }

  toString(indent = '') {
    return indent + 'NTLM Auth';
  }

  createResponse(challenge) {
    const client_nonce = this.createClientNonce();
    const lmv2len = 24;
    const ntlmv2len = 16;
    const domain = challenge.domain;
    const username = challenge.userName;
    const password = challenge.password;
    const ntlmData = challenge.ntlmpacket;
    const server_data = ntlmData.target;
    const server_nonce = ntlmData.nonce;
    const bufferLength = 64 + domain.length * 2 + username.length * 2 + lmv2len + ntlmv2len + 8 + 8 + 8 + 4 + server_data.length + 4;
    const data = new _writableTrackingBuffer.default(bufferLength);
    data.position = 0;
    data.writeString('NTLMSSP\u0000', 'utf8');
    data.writeUInt32LE(0x03);
    const baseIdx = 64;
    const dnIdx = baseIdx;
    const unIdx = dnIdx + domain.length * 2;
    const l2Idx = unIdx + username.length * 2;
    const ntIdx = l2Idx + lmv2len;
    data.writeUInt16LE(lmv2len);
    data.writeUInt16LE(lmv2len);
    data.writeUInt32LE(l2Idx);
    data.writeUInt16LE(ntlmv2len);
    data.writeUInt16LE(ntlmv2len);
    data.writeUInt32LE(ntIdx);
    data.writeUInt16LE(domain.length * 2);
    data.writeUInt16LE(domain.length * 2);
    data.writeUInt32LE(dnIdx);
    data.writeUInt16LE(username.length * 2);
    data.writeUInt16LE(username.length * 2);
    data.writeUInt32LE(unIdx);
    data.writeUInt16LE(0);
    data.writeUInt16LE(0);
    data.writeUInt32LE(baseIdx);
    data.writeUInt16LE(0);
    data.writeUInt16LE(0);
    data.writeUInt32LE(baseIdx);
    data.writeUInt16LE(0x8201);
    data.writeUInt16LE(0x08);
    data.writeString(domain, 'ucs2');
    data.writeString(username, 'ucs2');
    const lmv2Data = this.lmv2Response(domain, username, password, server_nonce, client_nonce);
    data.copyFrom(lmv2Data);
    const genTime = new Date().getTime();
    const ntlmDataBuffer = this.ntlmv2Response(domain, username, password, server_nonce, server_data, client_nonce, genTime);
    data.copyFrom(ntlmDataBuffer);
    data.writeUInt32LE(0x0101);
    data.writeUInt32LE(0x0000);
    const timestamp = this.createTimestamp(genTime);
    data.copyFrom(timestamp);
    data.copyFrom(client_nonce);
    data.writeUInt32LE(0x0000);
    data.copyFrom(server_data);
    data.writeUInt32LE(0x0000);
    return data.data;
  }

  createClientNonce() {
    const client_nonce = Buffer.alloc(8, 0);
    let nidx = 0;

    while (nidx < 8) {
      client_nonce.writeUInt8(Math.ceil(Math.random() * 255), nidx);
      nidx++;
    }

    return client_nonce;
  }

  ntlmv2Response(domain, user, password, serverNonce, targetInfo, clientNonce, mytime) {
    const timestamp = this.createTimestamp(mytime);
    const hash = this.ntv2Hash(domain, user, password);
    const dataLength = 40 + targetInfo.length;
    const data = Buffer.alloc(dataLength, 0);
    serverNonce.copy(data, 0, 0, 8);
    data.writeUInt32LE(0x101, 8);
    data.writeUInt32LE(0x0, 12);
    timestamp.copy(data, 16, 0, 8);
    clientNonce.copy(data, 24, 0, 8);
    data.writeUInt32LE(0x0, 32);
    targetInfo.copy(data, 36, 0, targetInfo.length);
    data.writeUInt32LE(0x0, 36 + targetInfo.length);
    return this.hmacMD5(data, hash);
  }

  createTimestamp(time) {
    const tenthsOfAMicrosecond = _jsbi.default.multiply(_jsbi.default.add(_jsbi.default.BigInt(time), _jsbi.default.BigInt(11644473600)), _jsbi.default.BigInt(10000000));

    const lo = _jsbi.default.toNumber(_jsbi.default.bitwiseAnd(tenthsOfAMicrosecond, _jsbi.default.BigInt(0xffffffff)));

    const hi = _jsbi.default.toNumber(_jsbi.default.bitwiseAnd(_jsbi.default.signedRightShift(tenthsOfAMicrosecond, _jsbi.default.BigInt(32)), _jsbi.default.BigInt(0xffffffff)));

    const result = Buffer.alloc(8);
    result.writeUInt32LE(lo, 0);
    result.writeUInt32LE(hi, 4);
    return result;
  }

  lmv2Response(domain, user, password, serverNonce, clientNonce) {
    const hash = this.ntv2Hash(domain, user, password);
    const data = Buffer.alloc(serverNonce.length + clientNonce.length, 0);
    serverNonce.copy(data);
    clientNonce.copy(data, serverNonce.length, 0, clientNonce.length);
    const newhash = this.hmacMD5(data, hash);
    const response = Buffer.alloc(newhash.length + clientNonce.length, 0);
    newhash.copy(response);
    clientNonce.copy(response, newhash.length, 0, clientNonce.length);
    return response;
  }

  ntv2Hash(domain, user, password) {
    const hash = this.ntHash(password);
    const identity = Buffer.from(user.toUpperCase() + domain.toUpperCase(), 'ucs2');
    return this.hmacMD5(identity, hash);
  }

  ntHash(text) {
    const unicodeString = Buffer.from(text, 'ucs2');
    return Buffer.from(_jsMd.default.arrayBuffer(unicodeString));
  }

  hmacMD5(data, key) {
    return crypto.createHmac('MD5', key).update(data).digest();
  }

}

var _default = NTLMResponsePayload;
exports.default = _default;
module.exports = NTLMResponsePayload;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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