"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Sender = exports.ParallelSendStrategy = void 0;

var _dgram = _interopRequireDefault(require("dgram"));

var _net = _interopRequireDefault(require("net"));

var punycode = _interopRequireWildcard(require("punycode"));

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class AbortError extends Error {
  constructor() {
    super('The operation was aborted');
    this.code = void 0;
    this.code = 'ABORT_ERR';
    this.name = 'AbortError';
  }

}

class ParallelSendStrategy {
  constructor(addresses, port, signal, request) {
    this.addresses = void 0;
    this.port = void 0;
    this.request = void 0;
    this.signal = void 0;
    this.addresses = addresses;
    this.port = port;
    this.request = request;
    this.signal = signal;
  }

  send(cb) {
    const signal = this.signal;

    if (signal.aborted) {
      return cb(new AbortError());
    }

    const sockets = [];
    let errorCount = 0;

    const onError = err => {
      errorCount++;

      if (errorCount === this.addresses.length) {
        signal.removeEventListener('abort', onAbort);
        clearSockets();
        cb(err);
      }
    };

    const onMessage = message => {
      signal.removeEventListener('abort', onAbort);
      clearSockets();
      cb(null, message);
    };

    const onAbort = () => {
      clearSockets();
      cb(new AbortError());
    };

    const clearSockets = () => {
      for (const socket of sockets) {
        socket.removeListener('error', onError);
        socket.removeListener('message', onMessage);
        socket.close();
      }
    };

    signal.addEventListener('abort', onAbort, {
      once: true
    });

    for (let j = 0; j < this.addresses.length; j++) {
      const udpType = this.addresses[j].family === 6 ? 'udp6' : 'udp4';

      const socket = _dgram.default.createSocket(udpType);

      sockets.push(socket);
      socket.on('error', onError);
      socket.on('message', onMessage);
      socket.send(this.request, 0, this.request.length, this.port, this.addresses[j].address);
    }
  }

}

exports.ParallelSendStrategy = ParallelSendStrategy;

class Sender {
  constructor(host, port, lookup, signal, request) {
    this.host = void 0;
    this.port = void 0;
    this.request = void 0;
    this.lookup = void 0;
    this.signal = void 0;
    this.host = host;
    this.port = port;
    this.request = request;
    this.lookup = lookup;
    this.signal = signal;
  }

  execute(cb) {
    if (_net.default.isIP(this.host)) {
      this.executeForIP(cb);
    } else {
      this.executeForHostname(cb);
    }
  }

  executeForIP(cb) {
    this.executeForAddresses([{
      address: this.host,
      family: _net.default.isIPv6(this.host) ? 6 : 4
    }], cb);
  } // Wrapper for stubbing. Sinon does not have support for stubbing module functions.


  invokeLookupAll(host, cb) {
    this.lookup.call(null, punycode.toASCII(host), {
      all: true
    }, cb);
  }

  executeForHostname(cb) {
    this.invokeLookupAll(this.host, (err, addresses) => {
      if (err) {
        return cb(err);
      }

      this.executeForAddresses(addresses, cb);
    });
  }

  executeForAddresses(addresses, cb) {
    const parallelSendStrategy = new ParallelSendStrategy(addresses, this.port, this.signal, this.request);
    parallelSendStrategy.send(cb);
  }

}

exports.Sender = Sender;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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