"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _collation = require("../collation");

var _token = require("./token");

const types = {
  1: {
    name: 'DATABASE',
    event: 'databaseChange'
  },
  2: {
    name: 'LANGUAGE',
    event: 'languageChange'
  },
  3: {
    name: 'CHARSET',
    event: 'charsetChange'
  },
  4: {
    name: 'PACKET_SIZE',
    event: 'packetSizeChange'
  },
  7: {
    name: 'SQL_COLLATION',
    event: 'sqlCollationChange'
  },
  8: {
    name: 'BEGIN_TXN',
    event: 'beginTransaction'
  },
  9: {
    name: 'COMMIT_TXN',
    event: 'commitTransaction'
  },
  10: {
    name: 'ROLLBACK_TXN',
    event: 'rollbackTransaction'
  },
  13: {
    name: 'DATABASE_MIRRORING_PARTNER',
    event: 'partnerNode'
  },
  17: {
    name: 'TXN_ENDED'
  },
  18: {
    name: 'RESET_CONNECTION',
    event: 'resetConnection'
  },
  20: {
    name: 'ROUTING_CHANGE',
    event: 'routingChange'
  }
};

function readNewAndOldValue(parser, length, type, callback) {
  switch (type.name) {
    case 'DATABASE':
    case 'LANGUAGE':
    case 'CHARSET':
    case 'PACKET_SIZE':
    case 'DATABASE_MIRRORING_PARTNER':
      return parser.readBVarChar(newValue => {
        parser.readBVarChar(oldValue => {
          switch (type.name) {
            case 'PACKET_SIZE':
              return callback(new _token.PacketSizeEnvChangeToken(parseInt(newValue), parseInt(oldValue)));

            case 'DATABASE':
              return callback(new _token.DatabaseEnvChangeToken(newValue, oldValue));

            case 'LANGUAGE':
              return callback(new _token.LanguageEnvChangeToken(newValue, oldValue));

            case 'CHARSET':
              return callback(new _token.CharsetEnvChangeToken(newValue, oldValue));

            case 'DATABASE_MIRRORING_PARTNER':
              return callback(new _token.DatabaseMirroringPartnerEnvChangeToken(newValue, oldValue));
          }
        });
      });

    case 'SQL_COLLATION':
    case 'BEGIN_TXN':
    case 'COMMIT_TXN':
    case 'ROLLBACK_TXN':
    case 'RESET_CONNECTION':
      return parser.readBVarByte(newValue => {
        parser.readBVarByte(oldValue => {
          switch (type.name) {
            case 'SQL_COLLATION':
              {
                const newCollation = newValue.length ? _collation.Collation.fromBuffer(newValue) : undefined;
                const oldCollation = oldValue.length ? _collation.Collation.fromBuffer(oldValue) : undefined;
                return callback(new _token.CollationChangeToken(newCollation, oldCollation));
              }

            case 'BEGIN_TXN':
              return callback(new _token.BeginTransactionEnvChangeToken(newValue, oldValue));

            case 'COMMIT_TXN':
              return callback(new _token.CommitTransactionEnvChangeToken(newValue, oldValue));

            case 'ROLLBACK_TXN':
              return callback(new _token.RollbackTransactionEnvChangeToken(newValue, oldValue));

            case 'RESET_CONNECTION':
              return callback(new _token.ResetConnectionEnvChangeToken(newValue, oldValue));
          }
        });
      });

    case 'ROUTING_CHANGE':
      return parser.readUInt16LE(valueLength => {
        // Routing Change:
        // Byte 1: Protocol (must be 0)
        // Bytes 2-3 (USHORT): Port number
        // Bytes 4-5 (USHORT): Length of server data in unicode (2byte chars)
        // Bytes 6-*: Server name in unicode characters
        parser.readBuffer(valueLength, routePacket => {
          const protocol = routePacket.readUInt8(0);

          if (protocol !== 0) {
            throw new Error('Unknown protocol byte in routing change event');
          }

          const port = routePacket.readUInt16LE(1);
          const serverLen = routePacket.readUInt16LE(3); // 2 bytes per char, starting at offset 5

          const server = routePacket.toString('ucs2', 5, 5 + serverLen * 2);
          const newValue = {
            protocol: protocol,
            port: port,
            server: server
          };
          parser.readUInt16LE(oldValueLength => {
            parser.readBuffer(oldValueLength, oldValue => {
              callback(new _token.RoutingEnvChangeToken(newValue, oldValue));
            });
          });
        });
      });

    default:
      console.error('Tedious > Unsupported ENVCHANGE type ' + type.name); // skip unknown bytes

      parser.readBuffer(length - 1, () => {
        callback(undefined);
      });
  }
}

function envChangeParser(parser, _options, callback) {
  parser.readUInt16LE(length => {
    parser.readUInt8(typeNumber => {
      const type = types[typeNumber];

      if (!type) {
        console.error('Tedious > Unsupported ENVCHANGE type ' + typeNumber); // skip unknown bytes

        return parser.readBuffer(length - 1, () => {
          callback(undefined);
        });
      }

      readNewAndOldValue(parser, length, type, token => {
        callback(token);
      });
    });
  });
}

var _default = envChangeParser;
exports.default = _default;
module.exports = envChangeParser;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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